//
//  AliyunTemplateEditor.h
//  AliyunVideoSDKPro
//
//  Created by Bingo on 2021/11/9.
//  Copyright © 2021 aliyun. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "AliyunTemplate.h"
#import "AliyunTemplateNode.h"
#import "AliyunEditor.h"
#import "AliyunTemplateResource.h"
#import "AliyunDraftManager.h"

/**
 应用模板编辑器
 API_AVAILABLE(3.27.0)
 */
@interface AliyunTemplateEditor : NSObject

/**
 初始化Editor，可以新建或打开一个已有模板工程

 @param taskPath 工作文件夹路径
 @param templateTaskPath 模板文件夹路径，为nil时尝试加载一个编辑过的模板工程
 @return AliyunTemplateEditor
 */
- (instancetype)initWithPath:(NSString *)taskPath templateTaskPath:(NSString *)templateTaskPath;

/**
 taskPath文件夹路径
 */
@property(nonatomic, copy, readonly) NSString *taskPath;


/**
 预览视图
 */
@property(nonatomic, strong) UIView *preview;

/**
 当前模板
 
 @return AliyunTemplate
 */
- (AliyunTemplate *)getCurrentTemplate;


/**
 内容视图
 
 @return AliyunEditorRenderWrapper
 */
- (AliyunEditorRenderWrapper *)getRenderWrapper;

/**
 编辑工程模型
 
 @return AliyunEditorProject
 */
- (AliyunEditorProject *)getEditorProject;


/**
 获取播放器接口实例

 @return AliyunIPlayer
 */
- (id<AliyunIPlayer>)getPlayer;

/**
 播放回调
 */
@property(nonatomic, weak) id<AliyunIPlayerCallback> playerCallback;

/**
 获取导出接口实例

 @return AliyunIExporter
 */
- (id<AliyunIExporter>)getExporter;

/**
 导出回调
 */
@property(nonatomic, weak) id<AliyunIExporterCallback> exporterCallback;


/**
 加载编辑器

 @return
 ALIVC_COMMON_RETURN_SUCCESS  正常返回
 ALIVC_COMMON_RETURN_FAILED 失败返回
 */
- (int)loadEditor;


/**
 释放编辑器

 @return
 ALIVC_COMMON_RETURN_SUCCESS  正常返回
 ALIVC_COMMON_RETURN_FAILED 失败返回
 */
- (int)releaseEditor;

/**
 开始编辑，创建相关资源

 @return
 ALIVC_COMMON_RETURN_SUCCESS  正常返回
 ALIVC_COMMON_RETURN_FAILED 失败返回
 其他返回参考常见错误码
 */
- (int)startEdit;

/**
 停止编辑，释放相关资源

 @return
 ALIVC_COMMON_RETURN_SUCCESS  正常返回
 ALIVC_COMMON_RETURN_FAILED 失败返回
 其他返回参考常见错误码
 */
- (int)stopEdit;

/**
 获取所有片段节点

 @return AliyunTemplateClipNode类型的列表
 */
- (NSArray<AliyunTemplateClipNode *> *)clipNodes;

/**
 获取所有字幕节点

 @return AliyunTemplateCaptionNode类型的列表
 */
- (NSArray<AliyunTemplateCaptionNode *> *)captionNodes;

/**
 更新片段节点内容
 @param node 节点
 @param clipPath 要替换的片段文件路径
 @param clipType  要替换的片段类型
 @return
 ALIVC_COMMON_RETURN_SUCCESS  正常返回
 ALIVC_COMMON_RETURN_FAILED 失败返回
 */
- (int)updateClipNode:(AliyunTemplateClipNode *)node clipPath:(NSString *)clipPath clipType:(AliyunClipType)clipType;

/**
 更新字幕节点内容
 @param node 节点
 @param text 要替换的文字
 @return
 ALIVC_COMMON_RETURN_SUCCESS  正常返回
 ALIVC_COMMON_RETURN_FAILED 失败返回
 */
- (int)updateCaptionNode:(AliyunTemplateCaptionNode *)node text:(NSString *)text;


/**
 获取节点在时间轴上的开始播放时间
 @param node 节点
 @return 播放开始时间
 */
- (NSTimeInterval)playStartTimeWithNode:(AliyunTemplateNode *)node;


@end


@interface AliyunTemplateEditor (Draft)

/**
 保存到草稿
 
 @param draftMgr 草稿管理
 @return 草稿对象
 */
- (AliyunDraft *) saveToDraft:(AliyunDraftManager *)draftMgr;

/**
 保存到草稿
 
 @param draftMgr 草稿管理
 @param title 草稿标题
 @return 草稿对象
 */
- (AliyunDraft *) saveToDraft:(AliyunDraftManager *)draftMgr withTitle:(NSString *)title;


@end

@interface AliyunTemplateEditor (Global)


/**
 新建一个模板编辑器
 
 @param templateTaskPath 模板目录
 @param taskPath 工作目录
 @return AliyunTemplateEditor
 */
+ (AliyunTemplateEditor *)createTemplateEditor:(NSString *)templateTaskPath onTaskPath:(NSString *)taskPath;


/**
 打开一个已有的模板编辑器
 
 @param taskPath 工作目录
 @return AliyunTemplateEditor
 */
+ (AliyunTemplateEditor *)openTemplateEditor:(NSString *)taskPath;


@end
