//
//  AliyunVideoLicense.h
//  AliyunVideoSDKPro
//
//  Created by coder.pi on 2021/11/19.
//  Copyright © 2021 aliyun. All rights reserved.
//

#import <Foundation/Foundation.h>

/**
 增值服务类型
 
 - AliyunVideoFeatureTypeMV: MV
 - AliyunVideoFeatureTypeSticker: 动态贴纸
 - AliyunVideoFeatureTypeCropCompose: 剪裁压缩
 - AliyunVideoFeatureTypeCaption: 字幕
 - AliyunVideoFeatureTypeAETemplate: AE模板
 */
typedef NS_ENUM(NSUInteger, AliyunVideoFeatureType) {
    AliyunVideoFeatureTypeMV                = 1,
    AliyunVideoFeatureTypeSticker           = 2,
    AliyunVideoFeatureTypeCropCompose       = 4,
    AliyunVideoFeatureTypeCaption           = 5,
    AliyunVideoFeatureTypeAETemplate        = 7,
};

/**
 增值服务权限信息
 */
@interface AliyunVideoLicenseFeature : NSObject
/**
 增值服务类型
 */
@property (nonatomic, readonly) AliyunVideoFeatureType feature;
/**
 权限开始时间
 */
@property (nonatomic, readonly) NSTimeInterval startTime;
/**
 权限结束时间
 */
@property (nonatomic, readonly) NSTimeInterval endTime;
@end

/**
 证书信息
 */
@interface AliyunVideoLicense : NSObject
/**
 是否为试用版本
 */
@property (nonatomic, readonly) BOOL isTrialVersion;
/**
 证书签发时间
 */
@property (nonatomic, readonly) NSTimeInterval certSignTime;
/**
 证书过期时间
 */
@property (nonatomic, readonly) NSTimeInterval certExpireTime;
/**
 业务过期时间
 */
@property (nonatomic, readonly) NSTimeInterval expireTime;
/**
 增值服务列表
 */
@property (nonatomic, readonly) NSArray<AliyunVideoLicenseFeature *> *features;
@end

/**
 证书验证结果
 
 - AliyunVideoLicenseResultCodeUninitialized: 未初始化
 - AliyunVideoLicenseResultCodeSuccess: 验证成功
 - AliyunVideoLicenseResultCodeExpired: 已经过期
 - AliyunVideoLicenseResultCodeInvalid: 证书无效
 - AliyunVideoLicenseResultCodeUnknownError: 未知错误
 */
typedef NS_ENUM(NSInteger, AliyunVideoLicenseResultCode) {
    AliyunVideoLicenseResultCodeUninitialized   = -1,
    AliyunVideoLicenseResultCodeSuccess         = 0,
    AliyunVideoLicenseResultCodeExpired         = 1,
    AliyunVideoLicenseResultCodeInvalid         = 2,
    AliyunVideoLicenseResultCodeUnknownError    = 3,
};

/**
 证书更新结果
 
 - AliyunVideoLicenseRefreshCodeUninitialized:  未初始化
 - AliyunVideoLicenseRefreshCodeSuccess: 更新成功
 - AliyunVideoLicenseRefreshCodeInvalid: 没找有效证书
 - AliyunVideoLicenseRefreshCodeServerError: 服务端错误
 - AliyunVideoLicenseRefreshCodeNetworkError: 网络错误
 - AliyunVideoLicenseRefreshCodeUnknownError: 未知错误
 */
typedef NS_ENUM(NSInteger, AliyunVideoLicenseRefreshCode) {
    AliyunVideoLicenseRefreshCodeUninitialized  = -1,
    AliyunVideoLicenseRefreshCodeSuccess        = 0,
    AliyunVideoLicenseRefreshCodeInvalid        = 1,
    AliyunVideoLicenseRefreshCodeServerError    = 2,
    AliyunVideoLicenseRefreshCodeNetworkError   = 3,
    AliyunVideoLicenseRefreshCodeUnknownError   = 4,
};

/**
 短视频证书相关事件通知
 */
@protocol AliyunVideoLicenseEventDelegate <NSObject>
@optional
/**
 短视频使用功能时权限错误
 
 @param errCode 错误码
 */
- (void) onAliyunVideoLicenseCheckError:(AliyunVideoLicenseResultCode)errCode;

/**
 短视频使用增值服务时权限错误
 
 @param featureType 增值服务
 @param errCode 错误码
 */
- (void) onAliyunVideoLicenseFeatureCheck:(AliyunVideoFeatureType)featureType error:(AliyunVideoLicenseResultCode)errCode;
@end

/**
 短视频证书管理类
 */
@interface AliyunVideoLicenseManager : NSObject
/**
 当前是否已经初始化成功
 */
@property (nonatomic, class, readonly) BOOL HasInitialized;
/**
 当前的证书信息
 */
@property (nonatomic, class, readonly) AliyunVideoLicense *CurrentLicense;

/**
 短视频证书相关事件通知
 */
@property (nonatomic, class) id<AliyunVideoLicenseEventDelegate> EventDelegate;

/**
 初始化
 
 请在工程的Info.plist中增加AlivcLicenseKey和AlivcLicenseFile字段
 * 在AlivcLicenseKey中填入您的LicenseKey
 * 在AlivcLicenseFile中填入您的LicenseFile路径（相对于mainBundle）；例如您的LicenseFile为"license.crt"放到mainBundle下，就填入license.crt
 
 LicenseKey和LicenseFile的获取请参考文档：https://help.aliyun.com/document_detail/274386.html
 */
+ (NSError *) Setup;

/**
 更新证书
 @param callback 更新完成回调
 */
+ (void) Refresh:(void(^)(AliyunVideoLicenseRefreshCode))callback;

/**
 验证证书
 @return 验证结果
 */
/**
 Check the license
 @return the result code of license check
 */
+ (AliyunVideoLicenseResultCode) Check;

/**
 检查配置，可以帮助您检查license相关配置是否正确
 @param callback 检查完成回调
 */
+ (void) CheckSetting:(void(^)(NSError *error))callback;
@end

#ifdef NDEBUG
#define ALIVC_CHECK_LICENSE_SETTING
#else
#define ALIVC_CHECK_LICENSE_SETTING \
[AliyunVideoLicenseManager CheckSetting:^(NSError *error){ \
    NSAssert(error == nil, @"License Setting Error!"); \
}];
#endif
