//
//  AVPDef.h
//  AVPSDK
//
//  Created by shiping.csp on 2018/11/16.
//  Copyright © 2018 com.alibaba.AliyunPlayer. All rights reserved.
//

#ifndef AVPDef_h
#define AVPDef_h

#import <Foundation/Foundation.h>

#if TARGET_OS_OSX
#import <AppKit/AppKit.h>
#define AVPView NSView
#define AVPImage NSImage
#elif TARGET_OS_IPHONE
#import <UIKit/UIKit.h>
#define AVPView UIView
#define AVPImage UIImage
#endif // TARGET_OS_OSX

#import "AVPErrorCode.h"

/**
 * @brief 播放URL转换回调函数
 *
 *
 * @param srcURL [in] 输入的原始URL字符串，不能为NULL。
 * @param srcFormat [in] 输入URL的格式，通常为"m3u8"、"mp4"或其他格式标识，格式未知时为空字符串。
 * @param destURL [out] 输出的转换后URL字符串。函数需负责分配内存，调用者需在使用后释放该内存。
 * @return true表示URL转换成功，false表示转换失败。转换失败时，播放器将使用原始URL继续播放。
 *
 * @see AliPlayer::setPlayUrlConvertCallback
 */
typedef bool (*PlayURLConverCallback)(const char* srcURL, const char* srcFormat, char** destURL);

/**
 * @brief 播放器状态枚举
 */
typedef enum AVPStatus: NSUInteger {
    /** @brief 空转，闲时，静态 */
    AVPStatusIdle = 0,
    /** @brief 初始化完成 */
    AVPStatusInitialzed,
    /** @brief 准备完成 */
    AVPStatusPrepared,
    /** @brief 正在播放 */
    AVPStatusStarted,
    /** @brief 播放暂停 */
    AVPStatusPaused,
    /** @brief 播放停止 */
    AVPStatusStopped,
    /** @brief 播放完成 */
    AVPStatusCompletion,
    /** @brief 播放错误 */
    AVPStatusError
} AVPStatus;

/**
 * @brief 跳转模式枚举。控制seek操作时是否精确定位。
 *
 * 精准seek会准确地跳转到指定的时间位置，但速度相对较慢。
 * 不精准seek会跳转到指定时间位置之前的最近一个关键帧开始播放，速度较快，
 * 但可能与指定位置存在偏差。
 *
 * @see AliPlayer::seekToTime
 */
typedef enum AVPSeekMode: NSUInteger {
    /**
     * @brief 精确跳转模式。准确定位到指定的时间点。
     */
    AVP_SEEKMODE_ACCURATE = 0x01,
    
    /**
     * @brief 非精确跳转模式。快速跳转到指定时间之前的关键帧。
     */
    AVP_SEEKMODE_INACCURATE = 0x10,
} AVPSeekMode;

/**
 * @brief 自适应降级切换URL的应用场景枚举。用于标识URL切换发生在播放器还是预加载器中。
 *
 * 当播放器或预加载器因编码格式不支持（如H.265不支持）需要切换到备用URL时，
 * 此枚举用于区分切换发生的具体场景。
 *
 * @see AliPlayerGlobalSettings::AdaptiveDecoderGetBackupURLCallback
 */
typedef enum BizScene : NSUInteger {
    /**
     * @brief 播放器场景。URL切换发生在播放器中。
     */
    AVP_Player = 0,
    
    /**
     * @brief 预加载器场景。URL切换发生在预加载器中。
     */
    AVP_Loader = 1
} AVPBizScene;

/**
 * @brief 自适应降级时的原始URL编码格式枚举。
 *
 * 用于表示在自适应降级场景下，原始URL所对应的编码格式。
 * 当编码格式不被支持时，播放器可基于此信息切换到备用URL。
 *
 * @see AliPlayerGlobalSettings::AdaptiveDecoderGetBackupURLCallback
 */
typedef enum CodecType : NSUInteger {
    /**
     * @brief H.265编码格式（HEVC）。
     */
    AVP_H265 = 0
} AVPCodecType;

/**
 * @brief 缩放模式枚举。用于控制视频画面的缩放和填充方式。
 *
 * 定义了视频在显示区域内如何进行缩放、裁剪或拉伸的几种模式。
 */
typedef enum AVPScalingMode: NSUInteger {
    /**
     * @brief 不保持纵横比平铺（拉伸填充）。视频画面被拉伸以填满整个显示区域，可能导致画面变形。
     */
    AVP_SCALINGMODE_SCALETOFILL,
    
    /**
     * @brief 保持纵横比，显示黑边（适应填充）。视频保持原始纵横比显示，空余区域用黑色填充。
     */
    AVP_SCALINGMODE_SCALEASPECTFIT,
    
    /**
     * @brief 保持纵横比填充，可能需要裁剪（裁剪填充）。视频保持纵横比，填满显示区域，超出部分被裁剪。
     */
    AVP_SCALINGMODE_SCALEASPECTFILL,
} AVPScalingMode;

/**
 * @brief 旋转模式枚举。用于控制视频画面的旋转角度。
 *
 * 所有旋转均为顺时针旋转。
 */
typedef enum AVPRotateMode: NSUInteger {
    /**
     * @brief 不旋转，0度。
     */
    AVP_ROTATE_0 = 0,
    
    /**
     * @brief 顺时针旋转90度。
     */
    AVP_ROTATE_90 = 90,
    
    /**
     * @brief 顺时针旋转180度。
     */
    AVP_ROTATE_180 = 180,
    
    /**
     * @brief 顺时针旋转270度。
     */
    AVP_ROTATE_270 = 270
} AVPRotateMode;

/**
 * @brief 镜像模式枚举。用于控制视频画面的镜像翻转效果。
 *
 * 支持水平镜像（左右翻转）和垂直镜像（上下翻转）两种效果。
 */
typedef enum AVPMirrorMode: NSUInteger {
    /**
     * @brief 无镜像。画面保持原始方向，不进行任何翻转。
     */
    AVP_MIRRORMODE_NONE,
    
    /**
     * @brief 水平镜像（左右翻转）。画面沿垂直轴翻转，左右互换。
     */
    AVP_MIRRORMODE_HORIZONTAL,
    
    /**
     * @brief 垂直镜像（上下翻转）。画面沿水平轴翻转，上下颠倒。
     */
    AVP_MIRRORMODE_VERTICAL,
} AVPMirrorMode;
/**
 * @brief Alpha渲染模式
 * 用于处理包含Alpha通道（透明度）的视频帧。视频帧被划分为图像区域和Alpha通道区域，
 * 播放器将根据指定的布局方式，从帧中分离出RGB图像和Alpha数据，并进行透明合成渲染。
 * 常用于直播礼物、动态贴纸等需要透明叠加的场景。
 * 可以参考 <a href="https://help.aliyun.com/zh/vod/developer-reference/advanced-features-1#0f5bcbe637non">视频帧处理</a>
 * 
 */
typedef enum AVPAlphaRenderMode: NSUInteger {
    /**
     * @brief 无Alpha渲染。视频帧为普通RGB图像，不包含透明通道。
     */
    AVP_RENDERMODE_ALPHA_NONE = 0,
    
    /**
     * @brief Alpha通道位于帧的右侧。视频帧的左半部分为RGB图像，右半部分为Alpha通道（灰度图）。
     * 渲染时根据右侧Alpha值对左侧图像进行透明合成。
     */
    AVP_RENDERMODE_ALPHA_AT_RIGHT = 1,
    
    /**
     * @brief Alpha通道位于帧的左侧。视频帧的右半部分为RGB图像，左半部分为Alpha通道（灰度图）。
     * 渲染时根据左侧Alpha值对右侧图像进行透明合成。
     */
    AVP_RENDERMODE_ALPHA_AT_LEFT = 2,
    
    /**
     * @brief Alpha通道位于帧的上侧。视频帧的下半部分为RGB图像，上半部分为Alpha通道（灰度图）。
     * 渲染时根据上侧Alpha值对下侧图像进行透明合成。
     */
    AVP_RENDERMODE_ALPHA_AT_TOP = 3,
    
    /**
     * @brief Alpha通道位于帧的下侧。视频帧的上半部分为RGB图像，下半部分为Alpha通道（灰度图）。
     * 渲染时根据下侧Alpha值对上侧图像进行透明合成。
     */
    AVP_RENDERMODE_ALPHA_AT_BOTTOM = 4,
} AVPAlphaRenderMode;

/**
 * @brief 音频输出声道枚举。用于指定音频输出时选择的声道。
 *
 * 当输入音源为双声道（立体声）时，可通过此枚举选择仅输出左声道或右声道；
 * 若输入为单声道，则此设置无效，音频将正常播放。
 */
typedef enum AVPOutputAudioChannel:NSUInteger {
    /**@brief 不指定声道，默认值*/
    AVP_AUDIO_CHANNEL_NONE = 0,
    /**@brief 左声道*/
    AVP_AUDIO_CHANNEL_LEFT = 1,
    /**@brief 右声道*/
    AVP_AUDIO_CHANNEL_RIGHT = 2
} AVPOutputAudioChannel;

/**@brief 播放器事件类型*/
typedef enum AVPEventType: NSUInteger {
    /**@brief 准备完成事件*/
    AVPEventPrepareDone,
    /**@brief 自动启播事件*/
    AVPEventAutoPlayStart,
    /**@brief 首帧显示事件*/
    AVPEventFirstRenderedStart,
    /**@brief 播放完成事件*/
    AVPEventCompletion,
    /**@brief 缓冲开始事件*/
    AVPEventLoadingStart,
    /**@brief 缓冲完成事件*/
    AVPEventLoadingEnd,
    /**@brief 跳转完成事件*/
    AVPEventSeekEnd,
    /**@brief 循环播放开始事件*/
    AVPEventLoopingStart,
    /**@brief 清屏完成事件*/
    AVPEventClearScreenDone,
} AVPEventType;

/**
 * @brief 获取播放器信息的关键字枚举。用于查询播放器的特定属性或统计信息。
 *
 * 通过这些关键字可获取HTTP响应信息、连接信息等调试和性能相关的数据。
 * @see AliPlayer::getPropertyString
 */
typedef enum AVPPropertyKey: NSUInteger {
    /**@brief Http的response信息
     * 返回的字符串是JSON数组，每个对象带response和type字段。type字段可以是url/video/audio/subtitle，根据流是否有相应Track返回。
     * 例如：[{"response":"response string","type":"url"},{"response":"","type":"video"}]
     */
    AVP_KEY_RESPONSE_INFO = 0,

    /**@brief 主URL的连接信息
     * 返回的字符串是JSON对象，带url/ip/eagleID/cdnVia/cdncip/cdnsip等字段（如果解析不到则不添加）
     * 例如：{"url":"http://xxx","openCost":23,"ip":"11.111.111.11","cdnVia":"xxx","cdncip":"22.222.222.22","cdnsip":"xxx"}
     */
    AVP_KEY_CONNECT_INFO  = 1,
} AVPPropertyKey;

/**
 * @brief IP地址解析类型枚举。用于指定网络连接时使用的IP地址类型。
 *
 * 在DNS解析时，可指定优先使用IPv4或IPv6，或支持两者自动选择。
 * @see AliPlayer::setIPResolveType
 */
typedef enum AVPIpResolveType: NSUInteger {
    AVPIpResolveWhatEver,
    AVPIpResolveV4,
    AVPIpResolveV6,
} AVPIpResolveType;

/**
 * @brief 播放器可获取的选项枚举。用于实时查询播放器的性能指标。
 *
 * 通过这些选项可获取当前的渲染帧率、网络码率、视频码率等动态信息。
 * @see AliPlayer getOption
 */
typedef enum AVPOption : NSUInteger {
    /**
     * @brief 渲染的帧率（FPS）。返回值类型为Float，单位：帧/秒。
     */
    AVP_OPTION_RENDER_FPS = 0,

    /**
     * @brief 当前的网络下行码率。返回值类型为Float，单位：bps（比特/秒）。
     */
    AVP_OPTION_DOWNLOAD_BITRATE = 1,

    /**
     * @brief 当前播放的视频码率。返回值类型为Float，单位：bps（比特/秒）。
     */
    AVP_OPTION_VIDEO_BITRATE = 2,

    /**
     * @brief 当前播放的音频码率。返回值类型为Float，单位：bps（比特/秒）。
     */
    AVP_OPTION_AUDIO_BITRATE = 3,

    /**
     * @brief 自适应比特率（ABR）切换状态指示。返回值类型为Int。
     *
     * 在多码率HLS流且启用自动码率切换模式下，此选项用于判断当前是否处于ABR切换的状态。
     * 返回"1"表示正在进行ABR切换，返回"0"表示未处于切换状态。
     */
    AVP_OPTION_IS_ABRSWITCHING = 4,
} AVPOption;

/**
 * 策略类型
 */
typedef enum AVPStrategyType : NSUInteger {
    /**
     * 动态预加载时长策略
     */
    AVP_STRATEGY_DYNAMIC_PRELOAD = 1,
} AVPStrategyType;

/**
 * 多码率预加载类型，只对多码率HLS流生效
 * @see AliListPlayer SetMultiBitratesMode
 */
typedef enum AVPMultiBitratesMode : NSUInteger {
    /**
     * 默认配置，播放和预加载默认码率
     */
    AVPMultiBitratesMode_Default = 0,
    /**
     * 首帧优先配置，起播视频默认播放已完成预加载的码率
     */
    AVPMultiBitratesMode_FCPrio = 1,
    /**
     * 兼顾首帧和播放平滑，切换前后（moveToNext）的视频码率一致，且兼顾首帧性能
     */
    AVPMultiBitratesMode_FC_AND_SMOOTH = 2,
    /**
     * 播放平滑优先配置，起播视频默认播放前一个视频的码率
     */
    AVPMultiBitratesMode_SmoothPrio = 3,
} AVPMultiBitratesMode;


/**
 * @brief 播放场景类型枚举。用于指定媒体的播放场景，以便播放器优化相关配置。
 *
 * 阿里云播放器为不同场景预设了最优配置，通过指定场景类型可自动应用相应的优化策略，
 * 无需手动调整各个配置参数。
 * @see AliListPlayer::setScene
 */
typedef enum AVPSceneType : NSInteger {
    /**
     * 场景：无
     */
    AVP_SCENE_NONE = -1,
    /**
     * 超短视频场景：适用于30s以下
     */
    AVP_VERY_SHOR_VIDEO = 0,
    /**
     * 短视频场景：适用于30s-5min
     */
    AVP_SHORT_VIDEO = 1,
    /**
     * 中视频场景：适用于5min-30min
     */
    AVP_MIDDLE_VIDEO = 2,
    /**
     * 长视频场景：适用于30min以上
     */
    AVP_LONG_VIDEO = 3,
} AVPSceneType;


/**
 * @brief 画中画（PIP）显示模式枚举。用于控制画中画功能的UI展示方式。
 *
 * 定义了在启用画中画模式时，控制按钮的显示或隐藏方式。
 * @see AliPlayer::setPictureinPictureShowMode
 */
typedef enum AVPPIPShowMode : NSUInteger {
    /**
     * @brief 默认模式。正常显示所有画中画相关按钮。
     */
    AVP_SHOW_MODE_DEFAULT = 0,
    
    /**
     * @brief 隐藏快进快退模式。隐藏快进/快退按钮。
     */
    AVP_SHOW_MODE_HIDE_FAST_FORWARD_REWIND = 1,
} AVPPIPShowMode;

/**
 * @brief 播放错误信息模型类。用于描述和传递播放过程中发生的错误信息。
 *
 * 该类封装了错误码、错误描述、附加信息等多个字段，
 * 便于开发者诊断和处理播放错误。
 */
OBJC_EXPORT
@interface AVPErrorModel : NSObject

/**
 * @brief 播放错误码。
 *
 * 表示错误的具体类型和原因。详见 @link AVPErrorCode @endlink。
 *
 * @see AVPErrorCode
 */
@property (nonatomic, assign) AVPErrorCode code;

/**
 * @brief 播放错误描述信息。
 *
 * 对错误的文字描述，通常为英文，便于理解错误原因。
 */
@property (nonatomic, copy) NSString *message;

/**
 @brief extra为播放错误信息描述的附加信息
 */
@property (nonatomic, copy) NSString *extra;

/**
 @brief requestId为播放错误信息requestID
 */
@property (nonatomic, copy) NSString *requestId;

/**
 @brief videoId为播放错误发生的videoID
 */
@property (nonatomic, copy) NSString *videoId;

@end


/***
@brief AVPNetworkLevel 网络质量级别枚举
 */
/***
@brief AVPNetworkLevel represents network quality levels.
*/
typedef enum AVPNetworkLevel: NSInteger {
    /***
    @brief 网络状态未知。
     */
    /***
    @brief The network status is unknown.
    */
    AVPNetwork_UNKNOWN = -1,
    
    /**
     @brief 网络质量非常优秀。
     */
    AVPNetwork_EXCELLENT = 0,
    
    /**
     @brief 网络质量良好。
     */
    AVPNetwork_GOOD = 1,
    
    
    /**
     @brief 网络质量较差。
     */
    AVPNetwork_POOR = 2,
    
    /**
     @brief 网络质量很差。
     */
    AVPNetwork_BAD = 3,
    
    /**
     @brief 网络质量非常差。
     */
    AVPNetwork_VERY_BAD = 4,
    
    /**
     @brief 网络已断开连接。
     */
    AVPNetwork_DISCONNECTED = 5,
} AVPNetworkLevel;

/**
@brief 网络统计信息类，用于收集和表示播放器应用的网络状况。
*/
/***
@brief AVPNetworkStat represents network statistics for the player.
It is used to collect and describe network-related metrics.
*/
OBJC_EXPORT
@interface AVPNetworkStat : NSObject

/**
@brief 视频下载速度，单位 Kbps。默认值为 0.0。
*/
/***
@brief Video download speed in kilobits per second (Kbps). Default is 0.0.
*/
@property (nonatomic, assign) double videoDownloadSpeedKbps;

/**
@brief 音频下载速度，单位 Kbps。默认值为 0.0。
*/
/***
@brief Audio download speed in kilobits per second (Kbps). Default is 0.0.
*/
@property (nonatomic, assign) double audioDownloadSpeedKbps;

/**
@brief 往返时延（RTT），单位毫秒（ms）。表示从客户端到服务器再返回所需的时间。默认值为 0。
*/
/***
@brief Round-trip time (RTT) in milliseconds (ms).
Indicates the time for a packet to travel from client to server and back.
Default is 0.
*/
@property (nonatomic, assign) int64_t rttMs;

/**
@brief 丢包率（百分比），范围 0 ~ 100%。较高的丢包率可能意味着网络质量差。默认值为 0。
*/
/***
@brief Packet loss rate as a percentage, ranging from 0 to 100%.
A higher packet loss rate may indicate poor network quality. Default is 0.
*/
@property (nonatomic, assign) int64_t packetLossRatePercentage;

/**
@brief 当前网络质量级别，使用 AVPNetworkLevel 表示。默认值为 AVPNetwork_UNKNOWN。
*/
/***
@brief The current network quality level, represented by AVPNetworkLevel.
Default is AVPNetwork_UNKNOWN.
*/
@property (nonatomic, assign) AVPNetworkLevel level;

@end


/**
 @brief AVPTimeShiftModel直播时移描述
 */
OBJC_EXPORT
@interface AVPTimeShiftModel : NSObject

/**
 @brief startTime直播时移开始时间
 */
@property (nonatomic, assign) NSTimeInterval startTime;

/**
 @brief endTime直播时移结束时间
 */
@property (nonatomic, assign) NSTimeInterval endTime;

/**
 @brief currentTime直播时移当前时间
 */
@property (nonatomic, assign) NSTimeInterval currentTime;
@end

/**
 * @brief 日志级别枚举。用于控制播放器日志输出的详细程度。
 *
 * 日志级别从低到高分别为：NONE、FATAL、ERROR、WARNING、INFO、DEBUG、TRACE。
 * 数值越大，输出的日志信息越详细。
 */
typedef enum AVPLogLevel: NSUInteger {
    LOG_LEVEL_NONE    = 0,
    LOG_LEVEL_FATAL   = 8,
    LOG_LEVEL_ERROR   = 16,
    LOG_LEVEL_WARNING = 24,
    LOG_LEVEL_INFO    = 32,
    LOG_LEVEL_DEBUG   = 48,
    LOG_LEVEL_TRACE   = 56,
} AVPLogLevel;

/**
 * @brief 特定分辨率的像素数量参考值枚举。
 *
 * 该枚举列举了常见视频分辨率对应的像素总数（宽×高）。
 * 这些值可用于 @link AVPConfig::maxAllowedAbrVideoPixelNumber @endlink 的参考
 *
 * @note 像素数量为参考值，仅供参考，实际分辨率可能略有差异。
 */
typedef enum AVPPixelNumber: NSUInteger {
    Resolution_360P = 172800, //480 * 360
    Resolution_480P = 345600, //720 * 480
    Resolution_540P = 518400, //960 * 540
    Resolution_720P = 921600, //1280 * 720
    Resolution_1080P = 2073600,//1920 * 1080
    Resolution_2K = 3686400, //2560 * 1440
    Resolution_4K = 8847360, //4096 * 2160
    Resolution_NoLimit = INT_MAX,
} AVPPixelNumber;

/**
 * @brief 日志选项枚举。用于控制日志输出的特定功能。
 *
 * 通过这些选项可启用或禁用日志系统的特定功能。
 */
typedef enum AVPLogOption: NSUInteger {
    /**
     * @brief 启用帧级日志记录。当启用时，播放器将输出每一帧的详细日志信息。
     */
    FRAME_LEVEL_LOGGING_ENABLED = 1,
} AVPLogOption;

/**
 * @brief STS（临时安全令牌）状态枚举。用于表示STS令牌的有效状态。
 */
typedef enum _AVPStsStatus {
    /**
     * @brief 有效。STS令牌已获取且仍在有效期内。
     */
    Valid,
    
    /**
     * @brief 无效。STS令牌已过期或无法使用。
     */
    Invalid,
    
    /**
     * @brief 待定。STS令牌正在获取中，暂时不可用。
     */
    Pending
} AVPStsStatus;

/**
 * @brief STS信息结构体。用于存储临时安全令牌相关的凭证信息。
 *
 * 包含访问ID、访问密钥、安全令牌、地域和格式等信息，
 * 用于身份验证和授权访问阿里云资源。
 */
typedef struct _AVPStsInfo {
    /**
     * @brief 访问ID。用于身份认证的访问密钥ID。
     */
    NSString *accId;
    
    /**
     * @brief 访问密钥。用于签名请求的密钥。
     */
    NSString *accSecret;
    
    /**
     * @brief 安全令牌。临时安全令牌，有有效期限制。
     */
    NSString *token;
    
    /**
     * @brief 地域。指定资源所在的地域。
     */
    NSString *region;
    
    /**
     * @brief 格式。指定媒体格式，如"mp4"、"m3u8"等。
     */
    NSString *formats;
} AVPStsInfo;

/**
 * @brief 功能支持类型枚举。用于查询播放器是否支持特定的硬件或软件功能。
 *
 * 通过这些类型可检查设备是否支持特定的高级音视频功能。
 */
typedef enum _SupportFeatureType : NSUInteger {
    /**
     * @brief 杜比音频（Dolby Audio）支持。检查硬件是否支持杜比音频解码和输出。
     */
    FeatureDolbyAudio
} SupportFeatureType;

/**
<<<<<<< Updated upstream
 * @brief GlobalSettings的特定功能选项。全局设置选项枚举。用于配置播放器全局行为的选项。
 * @link +setOption:value:@endlink
 * @link +setOption:valueInt:@endlink
 */
typedef enum _GlobalOption: NSUInteger {
    /**
     * @brief 预连接域名选项
     * @link +setOption:value:@endlink
     *
     * <p>设置预连接的域名，用于优化首次连接速度。
     */
    SET_PRE_CONNECT_DOMAIN = 0,
    
    /**
     * @brief DNS 本地优先解析选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>启用后，DNS 解析将优先使用本地 DNS 而非远程 DNS。
     */
    SET_DNS_PRIORITY_LOCAL_FIRST = 1,
    
    /**
     * @brief 启用 HTTP/2 多路复用选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>启用 HTTP/2 协议的多路复用功能，提高网络传输效率。
     */
    ENABLE_H2_MULTIPLEX = 2,
    
    /**
     * @brief 设置额外数据选项
     * @link +setOption:value:@endlink
     *
     * <p>用于传递自定义的额外数据到播放器内部。
     */
    SET_EXTRA_DATA = 3,
    
    /**
     * @brief 启用 Android 解码器复用选项（仅 Android 平台）
     * @link +setOption:valueInt:@endlink
     *
     * <p>Android 平台专用占位符选项，用于启用解码器复用功能。
     */
    ENABLE_ANDROID_DECODE_REUSE = 4, //android only, placeholder
    
    /**
     * @brief 准备时不暂停选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>启用后，在准备过程中不会暂停播放。
     */
    NOT_PAUSE_WHEN_PREPARING = 5, // do not pause when preparing
    
    /**
     * @brief 允许 RTS 降级选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>允许超低延迟直播(RTS)在必要时降级到普通直播模式。
     */
    ALLOW_RTS_DEGRADE = 6,
    
    /**
     * @brief 启用解码器快速首帧选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>优化解码器初始化流程，加快首帧显示速度。
     */
    ENABLE_DECODER_FAST_FIRST_FRAME = 7,
    
    /**
     * @brief 禁用截图缩放选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>禁用截图时的自动缩放功能。
     */
    DISABLE_CAPTURE_SCALE = 8,
    
    /**
     * @brief 允许边界变化动画选项（仅 iOS 平台）
     * @link +setOption:valueInt:@endlink
     *
     * <p>iOS 平台专用选项，允许视图边界变化时显示动画效果。
     */
    ALLOW_BOUNDS_CHANGE_ANIMATION = 10, // iOS only
    
    /**
     * @brief AV3A 解码模型路径选项
     * @link +setOption:value:@endlink
     *
     * <p>指定 AV3A 解码模型的文件路径。
     */
    AV3A_DECODE_MODEL_PATH = 11,
    
    /**
     * @brief 忽略 DAR 缩放选项（仅 iOS 平台）
     * @link +setOption:valueInt:@endlink
     *
     * <p>iOS 平台专用选项，忽略显示宽高比(DAR)的缩放处理。
     */
    RENDER_IGNORE_DAR_SCALE = 14, //iOS only
    
    /**
     * @brief 硬件解码最大错误帧数选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>设置硬件解码器允许的最大连续错误帧数，超过此值将触发降级。
     */
    MAX_ERROR_FRAMES_HARDWARE_DECODE = 15,
    
    /**
     * @brief 允许 AudioUnit 选项（仅 iOS 平台）
     * @link +setOption:valueInt:@endlink
     *
     * <p>iOS 平台专用选项，允许使用 AudioUnit 进行音频处理。
     */
    ALLOW_AUDIOUNIT = 16, //iOS only
    
    /**
     * @brief 禁用低延迟音频队列追赶选项（仅 iOS 平台）
     * @link +setOption:valueInt:@endlink
     *
     * <p>iOS 平台专用选项，禁用低延迟音频队列中的追赶机制。
     */
    DISABLE_CATCHUP_IN_LOWLATENCY_AUDIOQUEUE = 17, //iOS only
    
    /**
     * @brief 启用 403 禁止错误时的 URL 刷新选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>当收到 HTTP 403 Forbidden 错误时，自动触发 URL 刷新机制。
     */
    ENABLE_URL_REFRESH_ON_FORBIDDEN = 18,

    /**
     * @brief 启用 VTT 对话提示的 JSON 解析选项
     * @link +setOption:valueInt:@endlink
     *
     * <p>启用后，VTT 字幕中的对话提示将通过 JSON 格式进行解析。
     */
    ENABLE_PARSE_VTT_DIALOGUE_CUE_BY_JSON =  100,
} GlobalOption;

/**
 * @brief 播放器特定功能选项枚举。用于配置单个播放器实例的行为选项。
 *
 * 这些选项可在创建播放器后进行设置，每个播放器实例可单独配置。
 * @see AliPlayer::setOption
 */
typedef enum _PlayerOption: NSUInteger {
    SET_MEDIA_TYPE = 0,
    
    /**
     * @brief 允许后台解码。在应用进入后台时继续进行视频解码，保持播放状态。
     */
    ALLOW_DECODE_BACKGROUND = 1,
    
    /**
     * @brief 允许预渲染。启用视频帧的预先渲染，加速首帧显示。
     */
    ALLOW_PRE_RENDER = 2,
    
    /**
     * @brief 播放时长包含播放速度。计算播放时长时考虑播放倍速的影响。
     * @see AliPlayer::getPlayedDuration
     */
    PLAYED_DURATION_INCLUDE_SPEED = 3,
    
    /**
     * @brief 预渲染最大缓冲时长。指定预渲染功能的最大缓冲时长，单位毫秒。
     * 仅在ALLOW_PRE_RENDER启用时生效。
     */
    PRE_RENDER_MAX_BUFFER_MS = 5,
    
    /**
     * @brief 线路优先设置备流。为多线路场景设置备用流URL，提高播放质量和可用性。
     */
    LINE_FALLBACK_URL = 6,
} PlayerOption;

/**
 * @brief 播放场景枚举。用于为播放器指定最适合的播放场景配置。
 *
 * 播放器会根据指定的场景自动应用优化的配置策略，
 * 无需开发者手动调整各个参数。
 * @see AliPlayer::setPlayerScene
 */
typedef enum _AVPScene {
    /**
     * 场景：无
     */
    SceneNone,
    /**
     * 长视频场景：适用于30min以上
     */
    SceneLong,
    /**
     * 中视频场景：适用于5min-30min
     */
    SceneMedium,
    /**
     * 短视频场景：适用于0s-5min
     */
    SceneShort,
    /**
     * 直播场景
     */
    SceneLive,
    /**
    超低延迟直播场景
      */
    SceneRTSLive
} AVPScene;

/**
 @brief URL刷新结果
 */
OBJC_EXPORT
@interface AVPUrlResult : NSObject
@property (nonatomic, strong) NSString *url;

- (instancetype)initWithUrl:(NSString *)url;
@end

#endif /* AVPDef_h */
