//
//  AVPDelegate.h
//  AliPlayerSDK
//
//  Created by shiping.csp on 2018/11/16.
//  Copyright © 2018 com.alibaba.AliyunPlayer. All rights reserved.
//

#ifndef AVPDelegate_h
#define AVPDelegate_h

@class AliPlayer;
@class AVPTrackInfo;

#import <Foundation/Foundation.h>
#import "AVPDef.h"


@protocol AVPUrlRefreshCallback <NSObject>
/**
 @brief 刷新成功
 @param newUrl 新的URL
 */
- (void)onSuccess:(NSString *)newUrl;

/**
 @brief 刷新失败
 @param errorMsg 错误信息
 */
- (void)onError: (NSString *)errorMsg;
@end

/**
 @protocol AVPSourceRefreshCallback
 @brief 播放源刷新回调协议，开发者需实现以处理播放源的刷新结果。

 此回调用于在播放器请求刷新播放源时通知开发。
 播放器会调用协议中的方法以向开发者提供刷新结果，包括成功或失败。

 @note 此协议适用于 URL 源、VidAuth 源等需要刷新逻辑的场景。
 */
@protocol AVPSourceRefreshCallback <NSObject>

/**
 @brief 当刷新操作成功时由播放器调用
 @param newSource 包含更新信息的新播放源对象。
 
 此方法表示刷新操作已成功完成。开发者需要将更新后的 `newSource` 返回给播放器，
 以使播放器能够加载新的资源。
 */
- (void)onSuccess:(id)newSource;

/**
 @brief 当刷新操作失败时由播放器调用
 @param errorMsg 描述失败原因的错误信息。
 
 此方法表示刷新操作失败。开发者可通过 `errorMsg` 捕获失败信息并进行后续处理。
 */
- (void)onError:(NSString *)errorMsg;

@end

@protocol AVPDelegate <NSObject>
@optional

/**
 @brief 播放器事件回调
 @param player 播放器player指针
 @param eventType 播放器事件类型
 @see AVPEventType
 */
-(void)onPlayerEvent:(AliPlayer*)player eventType:(AVPEventType)eventType;

/**
 @brief 播放器事件回调（带字符串描述）
 @param player 播放器player指针
 @param eventWithString 播放器事件类型字符串
 @param description 播放器事件详细说明
 @see AVPEventType
 */
-(void)onPlayerEvent:(AliPlayer*)player eventWithString:(AVPEventWithString)eventWithString description:(NSString *)description;

/**
 @brief 错误代理回调
 @param player 播放器player指针
 @param errorModel 播放器错误描述，参考AVPErrorModel
 触发时机：播放过程中发生错误时回调，包括网络错误、解码错误、文件错误等
 @see AVPErrorModel
 */
- (void)onError:(AliPlayer*)player errorModel:(AVPErrorModel *)errorModel;

/**
 @brief 视频大小变化回调
 @param player 播放器player指针
 @param width 视频宽度
 @param height 视频高度
 @param rotation 视频旋转角度
 触发时机：视频元数据加载完成或视频分辨率发生变化时回调
 */
- (void)onVideoSizeChanged:(AliPlayer*)player width:(int)width height:(int)height rotation:(int)rotation;

/**
 @brief 视频当前播放位置回调
 @param player 播放器player指针
 @param position 视频当前播放位置，单位毫秒
 触发频率：通过AVPConfig::positionTimerIntervalMs调整
 @see AliPlayer::currentPosition
 */
- (void)onCurrentPositionUpdate:(AliPlayer*)player position:(int64_t)position;

/**
 @brief 视频当前播放内容对应的utc时间回调
 @param player 播放器player指针
 @param time utc时间，单位毫秒
 适用于直播场景，获取当前播放内容对应的UTC时间戳
 @see AliPlayer::currentUtcTime
 */
- (void)onCurrentUtcTimeUpdate:(AliPlayer *)player time:(int64_t)time;

/**
 @brief 视频起播缓存命中回调
 @param player 播放器player指针
 @param size 文件大小,单位为字节。大于0代表命中缓存，此时值代表命中缓存的字节数。
 触发时机：播放器开始播放时，如果使用了本地缓存文件，则会回调此方法
 @see AliPlayer::localCacheLoadedSize
 */
- (void)onLocalCacheLoaded:(AliPlayer *)player size:(int64_t)size;

/**
 @brief 视频缓冲位置回调
 @param player 播放器player指针
 @param position 视频当前缓冲位置，单位毫秒
 表示当前已缓冲到的位置，可用于显示缓冲进度条，触发频率可通过AVPConfig::positionTimerIntervalMs调整
 @see AliPlayer::bufferedPosition
 */
- (void)onBufferedPositionUpdate:(AliPlayer*)player position:(int64_t)position;

/**
 @brief 视频缓冲进度回调，用于显示Loading进度
 @param player 播放器player指针
 @param progress 缓存进度0-100，表示缓冲完成的百分比
 */
- (void)onLoadingProgress:(AliPlayer*)player progress:(float)progress;


/**
 @brief 当前下载速度回调
 @param player 播放器player指针
 @param speed 当前下载速度，单位bits per second
 @see AliPlayer::currentDownloadSpeed
 直播场景下，会持续有数据；而点播场景下，可能因使用的缓存而无下载速度，是正常情况。
 */
- (void)onCurrentDownloadSpeed:(AliPlayer *)player speed:(int64_t)speed;

/**
 @brief 获取track信息回调
 @param player 播放器player指针
 @param info track流信息数组，包含视频、音频、字幕等轨道信息
 触发时机：prepare完成后，获取到媒体信息时回调
 @see AVPTrackInfo
 @see AliPlayer::getMediaInfo
 @see AliPlayer::prepare
 */
- (void)onTrackReady:(AliPlayer*)player info:(NSArray<AVPTrackInfo*>*)info;

/**
 @brief 获取子流track信息回调
 @param player 播放器player指针
 @param info 子流track信息数组
 适用于HLS打包多清晰度流。
 @see AVPTrackInfo
 @see AliPlayer::getSubMediaInfo
 */
- (void)onSubTrackReady:(AliPlayer*)player info:(NSArray<AVPTrackInfo*>*)info;


/**
 @brief 获取网络质量信息回调, 当前该监听器仅对超低延时直播（RTS）拉流场景生效。
 @param player 播放器player指针
 @param stat 网络质量信息，包含丢包率、延迟等指标
 @see AVPNetworkStat
 */
- (void)onNetworkStat: (AliPlayer*)player stat:(AVPNetworkStat*) stat;

/**
 @brief 选择希望播放的流
 @param player 播放器player指针
 @param info track流信息数组
 @return 返回要播放的流索引
 @see AVPTrackInfo
 @see AliPlayer::selectTrack
 */
- (int)onChooseTrackIndex:(AliPlayer *)player info:(NSArray<AVPTrackInfo *> *)info;

/**
 @brief track切换完成回调
 @param player 播放器player指针
 @param info 切换后的信息 参考AVPTrackInfo
 触发时机：调用{@link AliPlayer::selectTrack}后切换完成时回调
 @see AVPTrackInfo
 @see AliPlayer::selectTrack
 */
- (void)onTrackChanged:(AliPlayer*)player info:(AVPTrackInfo*)info;

/**
 @brief 根据url切换清晰度流成功通知
 @param player 播放器player指针
 @param url 切换后的url
 触发时机：调用{@link AliPlayer::switchStream}成功后回调
 注意：仅适用于阿里云直播地址的多清晰度切换
 @see AliPlayer::switchStream
 */
- (void)onStreamSwitchedSuccess:(AliPlayer*)player URL:(NSString*)URL;

/**
 @brief 根据url切换清晰度流失败通知
 @param player 播放器player指针
 @param URL 切换的url
 @param errorModel 播放器错误描述，参考AVPErrorModel
 触发时机：调用{@link AliPlayer::switchStream}失败后回调
 @see AliPlayer::switchStream
 */
- (void)onStreamSwitchedFail:(AliPlayer*)player URL:(NSString*)URL errorModel:(AVPErrorModel *)errorModel;

/**
 @brief 外挂字幕被添加
 @param player 播放器player指针
 @param trackIndex 字幕轨道索引
 @param URL 字幕url
 触发时机：调用{@link AliPlayer::addExtSubtitle}成功后回调
 @see AliPlayer::addExtSubtitle
 */
- (void)onSubtitleExtAdded:(AliPlayer*)player trackIndex:(int)trackIndex URL:(NSString *)URL;


/**
 @brief 字幕头信息回调。用于ASS或VTT字幕等需要头信息的格式。如果实现此回调，播放器将不会自动渲染字幕，需要调用者自行完成字幕渲染。
 @param player 播放器实例指针
 @param trackIndex 字幕轨道索引
 @param header 字幕头信息内容
 外挂字幕用法参考<a href="https://help.aliyun.com/zh/vod/developer-reference/advanced-features-1#fe50ee8e82fax">外挂字幕</a>
 */
- (void)onSubtitleHeader:(AliPlayer *)player trackIndex:(int)trackIndex Header:(NSString *)header;

/**
 @brief 字幕显示回调
 @param player 播放器player指针
 @param trackIndex 字幕流索引.
 @param subtitleID  字幕ID.
 @param subtitle 字幕显示的字符串
 触发时机：字幕需要显示时回调
 @see AliPlayer::selectExtSubtitle
 */
- (void)onSubtitleShow:(AliPlayer*)player trackIndex:(int)trackIndex subtitleID:(long)subtitleID subtitle:(NSString *)subtitle;

/**
 @brief 字幕隐藏回调
 @param player 播放器player指针
 @param trackIndex 字幕流索引.
 @param subtitleID  字幕ID.
 触发时机：字幕需要隐藏时回调
 @see AliPlayer::selectExtSubtitle
 */
- (void)onSubtitleHide:(AliPlayer*)player trackIndex:(int)trackIndex subtitleID:(long)subtitleID;

/**
 @brief 获取缩略图成功回调
 @param positionMs 指定的缩略图位置
 @param fromPos 此缩略图的开始位置
 @param toPos 此缩略图的结束位置
 @param image 缩图略图像指针,对于mac是NSImage，iOS平台是UIImage指针
 触发时机：调用{@link AliPlayer::getThumbnail}成功后回调
 @see AliPlayer::getThumbnail
 */
- (void)onGetThumbnailSuc:(int64_t)positionMs fromPos:(int64_t)fromPos toPos:(int64_t)toPos image:(id)image;

/**
 @brief 获取缩略图失败回调
 @param positionMs 指定的缩略图位置
 触发时机：调用{@link AliPlayer::getThumbnail}失败后回调
 @see AliPlayer::getThumbnail
 */
- (void)onGetThumbnailFailed:(int64_t)positionMs;

/**
 @brief 播放器状态改变回调
 @param player 播放器player指针
 @param oldStatus 老的播放器状态 参考AVPStatus
 @param newStatus 新的播放器状态 参考AVPStatus
 触发时机：播放器状态发生变化时回调，包括idle、prepared、started、paused、stopped、completed、error等状态
 @see AVPStatus
 @see AliPlayer::playerStatus
 */
- (void)onPlayerStatusChanged:(AliPlayer*)player oldStatus:(AVPStatus)oldStatus newStatus:(AVPStatus)newStatus;

/**
 @brief 获取截图回调，触发自{@link Aliplayer::snapShot}
 @param player 播放器player指针
 @param image 图像，iOS平台返回UIImage，而mac平台返回NSImage
 @see AliPlayer::snapShot
 */
- (void)onCaptureScreen:(AliPlayer*)player image:(AVPImage*)image;

/**
 @brief SEI回调
 @param player 播放器player指针
 @param type 类型
 @param uuid 数据（当type=5时,uuid值有效，播放器sdk 6.11.0及之后的版本新增）
 @param data 数据（不包含uuid部分，跟6.10.0及之前的版本语义保持一致）
 触发时机：解码器解析到SEI数据时回调
 */
- (void)onSEIData:(AliPlayer*)player type:(int)type uuid:(NSData *)uuid data:(NSData *)data;

/**
 @brief 播放器渲染信息回调
 @param player 播放器player指针
 @param timeMs 渲染时的系统时间，单位毫秒
 @param pts  视频帧pts
 触发时机：每帧视频渲染完成时回调，可用于帧级监控和性能分析
 @see AliPlayer::setOnVideoRenderedListener
 */
-(void)onVideoRendered:(AliPlayer*)player timeMs:(int64_t)timeMs pts:(int64_t)pts;

/**
 @brief URL过期回调
 @param player 播放器player指针
 @param expiredUrl 过期的URL
 @param callback URL刷新回调
 触发时机：播放器检测到URL过期时回调，可用于实现URL自动刷新功能
 */
- (void)onUrlExpired:(AliPlayer*)player expiredUrl:(NSString*)expiredUrl callback:(id<AVPUrlRefreshCallback>)callback;

/**
 @brief URL源过期回调
 @param player 播放器player指针
 @param expiredSource 过期的URL源对象
 @param callback 源刷新回调
 */
- (void)onURLSourceExpired:(AliPlayer*)player expiredSource:(id)expiredSource callback:(id<AVPSourceRefreshCallback>)callback;

/**
 @brief VidAuth源过期回调
 @param player 播放器player指针
 @param expiredSource 过期的VidAuth源对象
 @param callback 源刷新回调
 */
- (void)onVidAuthExpired:(AliPlayer*)player expiredSource:(id)expiredSource callback:(id<AVPSourceRefreshCallback>)callback;

@end

/**
 @brief 埋点事件参数回调
 */
@protocol AVPEventReportParamsDelegate <NSObject>
/**
 @brief 回调
 @param params  埋点事件参数，包含事件类型、时间戳、播放器状态等信息
 可用于业务埋点统一走播放器的埋点回调，方便业务统一流程
 @see AliPlayer::sendCustomEvent
 */
-(void)onEventReportParams:(NSDictionary<NSString *, NSString *>*)params;
@end
#endif /* AVPDelegate_h */
