//
// Created by yiliang on 2023/6/20.
//

#ifndef AVPPRELOADCONFIG_H
#define AVPPRELOADCONFIG_H
#import <Foundation/Foundation.h>

/**
 * @enum AVPPreloadOptionType
 * @brief 多码率流预加载的选项类型。
 *
 * 用于在预加载多码率流时，指定以何种标准（如分辨率、码率、清晰度）来选择要预加载的码率档位。
 */
typedef NS_ENUM(NSInteger, AVPPreloadOptionType) {
    /**
     * @brief 无指定选项。
     */
    AVPPreloadOptionTypeNone = 0,
    /**
     * @brief 按分辨率选择。
     */
    AVPPreloadOptionTypeResolution,
    /**
     * @brief 按码率选择。
     */
    AVPPreloadOptionTypeBandWidth,
    /**
     * @brief 按清晰度描述选择。
     */
    AVPPreloadOptionTypeQuality
};

/**
 * @interface AVPPreloadConfig
 * @brief 预加载配置类。
 *
 * 该类用于配置预加载任务的具体参数，如预加载时长、网络参数以及多码率流的选择策略。
 * 通常与 @link AVPPreloadTask @endlink 一起使用。
 * @see @link AVPPreloadTask @endlink
 */
OBJC_EXPORT
@interface AVPPreloadConfig : NSObject

/**
 * @property preloadDuration
 * @brief 预加载时长，单位为毫秒。
 * @see -[AVPPreloadConfig setDuration:]
 * @see -[AVPPreloadConfig getDuration]
 */
@property (nonatomic, assign) int preloadDuration;

/**
 * @property networkTimeout
 * @brief 网络超时时间，单位为毫秒。
 * @see -[AVPPreloadConfig setNetworkTimeout:]
 * @see -[AVPPreloadConfig getNetworkTimeout]
 */
@property (nonatomic, assign) int networkTimeout;

/**
 * @property networkRetryCount
 * @brief 网络重试次数。
 * @see -[AVPPreloadConfig setNetworkRetryCount:]
 * @see -[AVPPreloadConfig getNetworkRetryCount]
 */
@property (nonatomic, assign) int networkRetryCount;

/**
 * @property referrer
 * @brief HTTP 请求的 Referrer。
 * @see -[AVPPreloadConfig setReferrer:]
 * @see -[AVPPreloadConfig getReferrer]
 */
@property (nonatomic, strong) NSString *referrer;

/**
 * @property customHeaders
 * @brief 自定义 HTTP Headers。
 * @see -[AVPPreloadConfig setCustomHeaders:]
 * @see -[AVPPreloadConfig getCustomHeaders]
 */
@property (nonatomic, strong) NSArray<NSString *> *customHeaders;

/**
 * @property preloadOptionType
 * @brief 多码率流预加载的选项类型。
 * @see @link AVPPreloadOptionType @endlink
 */
@property (nonatomic, assign) AVPPreloadOptionType preloadOptionType;

/**
 * @property optionIntValue
 * @brief 当 `preloadOptionType` 为 @link AVPPreloadOptionTypeResolution @endlink 或 @link AVPPreloadOptionTypeBandWidth @endlink 时，此属性存储对应的整数值。
 */
@property (nonatomic, assign) int optionIntValue;

/**
 * @property optionStringValue
 * @brief 当 `preloadOptionType` 为 @link AVPPreloadOptionTypeQuality @endlink 时，此属性存储对应的字符串值。
 */
@property (nonatomic, strong) NSString *optionStringValue;

/**
 * @brief 默认构造函数。
 * @return 一个已初始化的 `AVPPreloadConfig` 实例。
 */
- (instancetype)init;
/**
 * @brief 使用指定时长构造预加载配置。
 * @param duration 预加载时长，单位为毫秒。
 * @return 一个已初始化的 `AVPPreloadConfig` 实例。
 */
- (instancetype)initWithDuration:(int)duration;

/**
 * @brief 获取预加载时长。
 * @return 预加载时长，单位为毫秒。
 * @see @link AVPPreloadConfig::preloadDuration @endlink
 */
- (int)getDuration;

/**
 * @brief 设置预加载时长。
 * @param duration 预加载时长，单位为毫秒。
 * @see @link AVPPreloadConfig::preloadDuration @endlink
 */
- (void)setDuration:(int)duration;

/**
 * @brief 获取网络超时时间。
 * @return 网络超时时间，单位为毫秒。
 * @see @link AVPPreloadConfig::networkTimeout @endlink
 */
- (int)getNetworkTimeout;

/**
 * @brief 设置网络超时时间。
 * @param networkTimeout 网络超时时间，单位为毫秒。
 * @see @link AVPPreloadConfig::networkTimeout @endlink
 */
- (void)setNetworkTimeout:(int)networkTimeout;

/**
 * @brief 获取网络重试次数。
 * @return 网络重试次数。
 * @see @link AVPPreloadConfig::networkRetryCount @endlink
 */
- (int)getNetworkRetryCount;

/**
 * @brief 设置网络重试次数。
 * @param networkRetryCount 网络重试次数。
 * @see @link AVPPreloadConfig::networkRetryCount @endlink
 */
- (void)setNetworkRetryCount:(int)networkRetryCount;

/**
 * @brief 获取 Referrer。
 * @return Referrer 字符串。
 * @see @link AVPPreloadConfig::referrer @endlink
 */
- (NSString *)getReferrer;

/**
 * @brief 设置 Referrer。
 * @param referrer Referrer 字符串。
 * @see @link AVPPreloadConfig::referrer @endlink
 */
- (void)setReferrer:(NSString *)referrer;

/**
 * @brief 获取自定义 Headers。
 * @return 自定义 Headers 数组。
 * @see @link AVPPreloadConfig::customHeaders @endlink
 */
- (NSArray<NSString *> *)getCustomHeaders;

/**
 * @brief 设置自定义 Headers。
 * @param customHeaders 自定义 Headers 数组。
 * @see @link AVPPreloadConfig::customHeaders @endlink
 */
- (void)setCustomHeaders:(NSArray<NSString *> *)customHeaders;

/**
 * @brief 为多码率流设置预加载的分辨率。
 * @note 调用此方法会设置预加载选项类型为 @link AVPPreloadOptionTypeResolution @endlink。
 * @param resolution 分辨率的宽和高的乘积。SDK 将选择与此值最接近的码率档位进行预加载。
 */
- (void)setDefaultResolution:(int)resolution;

/**
 * @brief 获取已设置的预加载分辨率。
 * @note 仅当预加载选项类型为 @link AVPPreloadOptionTypeResolution @endlink 时，返回值才有意义。
 * @return 分辨率的宽和高的乘积。
 */
- (int)getDefaultResolution;

/**
 * @brief 为多码率流设置预加载的码率。
 * @note 调用此方法会设置预加载选项类型为 @link AVPPreloadOptionTypeBandWidth @endlink。
 * @param bandWidth 目标码率。SDK 将选择与此值最接近的码率档位进行预加载。
 */
- (void)setDefaultBandWidth:(int)bandWidth;

/**
 * @brief 获取已设置的预加载码率。
 * @note 仅当预加载选项类型为 @link AVPPreloadOptionTypeBandWidth @endlink 时，返回值才有意义。
 * @return 目标码率。
 */
- (int)getDefaultBandWidth;

/**
 * @brief 为多码率流设置预加载的清晰度。
 * @note 调用此方法会设置预加载选项类型为 @link AVPPreloadOptionTypeQuality @endlink。
 * @param quality 目标清晰度的描述字符串。
 */
- (void)setDefaultQuality:(NSString *)quality;

/**
 * @brief 获取已设置的预加载清晰度。
 * @note 仅当预加载选项类型为 @link AVPPreloadOptionTypeQuality @endlink 时，返回值才有意义。
 * @return 目标清晰度的描述字符串。
 */
- (NSString *)getDefaultQuality;

/**
 * @brief 获取当前设置的默认预加载选项类型。
 * @return 当前的预加载选项类型。
 * @see @link AVPPreloadOptionType @endlink
 */
- (AVPPreloadOptionType)getDefaultType;

@end

#endif //AVPPRELOADCONFIG_H
