//
//  AVPConfig.h
//  AliPlayerSDK
//
//  Created by shiping.csp on 2018/11/16.
//  Copyright © 2018 com.alibaba.AliyunPlayer. All rights reserved.
//

#ifndef AVPConfig_h
#define AVPConfig_h

#import <Foundation/Foundation.h>

/**
 @brief 播放器配置类
 
 包含播放器的各项配置参数，如缓冲时长、网络超时、HTTP头等。
 配置应在调用 @link AliPlayer::prepare @endlink 前设置。
 */
OBJC_EXPORT
@interface AVPConfig : NSObject

/**
 @brief 直播最大延迟时间
 
 用于限制直播流的最大延迟。当播放延迟超过此值时，播放器会尝试加速追赶。
 单位：毫秒，默认值：5000毫秒（5秒）。
 */
@property (nonatomic, assign) int maxDelayTime;

/**
 @brief 高水位缓冲时长
 
 当播放器缓冲数据大于此值时，播放器恢复播放（从卡顿状态恢复）。
 用于避免频繁的缓冲暂停。单位：毫秒。
 */
@property (nonatomic, assign) int highBufferDuration;

/**
 @brief 起播缓冲时长
 
 开始播放前需要缓冲的数据时长。缓冲数据达到此值后开始播放。
 较小的值可加快起播速度，但可能导致卡顿；较大的值更稳定但延迟更高。
 单位：毫秒，默认值：500毫秒。
 */
@property (nonatomic, assign) int startBufferDuration;

/**
 @brief 最大缓冲时长
 
 播放器允许缓冲的最大数据时长。防止缓冲占用过多内存。
 单位：毫秒，默认值：50000毫秒（50秒）。
 */
@property (nonatomic, assign) int maxBufferDuration;

/**
 @brief 网络超时时间
 
 网络请求的超时时间。超过此时间未获得响应则认为网络连接超时。
 单位：毫秒，默认值：15000毫秒（15秒）。
 */
@property (nonatomic, assign) int networkTimeout;

/**
 @brief 网络重试次数
 
 网络连接失败时的重试次数。每次重试间隔由 @link networkTimeout @endlink 决定。
 设为0表示不进行自动重试，重试策略由应用决定。
 默认值：2次。
 */
@property (nonatomic, assign) int networkRetryCount;

/**
 @brief 探测数据大小（已弃用）
 
 用于探测媒体信息的数据量大小。默认值-1表示不设置，由播放器自动决定。
 
 @deprecated 此属性已弃用。
 */
@property (nonatomic, assign) int maxProbeSize DEPRECATED_MSG_ATTRIBUTE("This property is deprecated");


/**
 @brief HTTP请求Referer
 
 HTTP请求中的Referer字段值。用于防盗链等功能。
 */
@property (nonatomic, copy) NSString *referer;

/**
 @brief HTTP User-Agent
 
 HTTP请求中的User-Agent字段值。标识客户端信息。
 */
@property (nonatomic, copy) NSString *userAgent;

/**
 @brief HTTP代理
 
 用于访问网络资源的HTTP代理地址。
 */
@property (nonatomic, copy) NSString *httpProxy;

/**
 @brief 停止时是否清除显示
 
 调用stop()停止播放后，是否清除最后一帧图像。
 YES表示清除（黑屏），NO表示保留最后一帧。默认值：NO。
 */
@property (nonatomic, assign) BOOL clearShowWhenStop;

/**
 @brief 自定义HTTP头列表
 
 添加自定义的HTTP请求头。可用于传递特定的鉴权信息或其他HTTP头。
 */
@property (nonatomic, strong) NSMutableArray *httpHeaders;

/**
 @brief 启用SEI数据回调
 
 是否启用SEI（Supplemental Enhancement Information）数据的回调。
 启用后可通过回调获取视频中嵌入的SEI数据。
 */
@property (nonatomic, assign) BOOL enableSEI;

/**
 @brief 启用本地缓存
 
 当通过 @link AliPlayerGlobalSettings @endlink API启用本地缓存功能后，
 此配置用于控制当前播放器实例是否允许被缓存。默认值：YES（允许）。
 */
@property(nonatomic, assign) BOOL enableLocalCache;

/**
 @brief 像素缓冲区输出格式
 
 用于renderFrame回调的视频格式。仅对VTB解码器有效，值等于OSType。
 支持的格式包括：420v、420f、y420、BGRA等。
 */
@property (nonatomic, assign) int pixelBufferOutputFormat;

/**
 @brief HLS直播起播分片位置
 
 播放HLS直播流时的起播分片索引。用于控制直播的起播位置。
 */
@property(nonatomic, assign) int liveStartIndex;

/**
 @brief 禁用音频轨道
 
 是否禁用音频轨道。YES表示禁用，播放时不解码和输出音频。默认值：NO。
 */
@property (nonatomic, assign) BOOL disableAudio;

/**
 @brief 禁用视频轨道
 
 是否禁用视频轨道。YES表示禁用，播放时不解码和渲染视频。默认值：NO。
 */
@property (nonatomic, assign) BOOL disableVideo;

/**
 @brief 进度更新频率
 
 播放进度和缓冲进度更新的频率。包括当前播放位置和缓冲位置的回调更新间隔。
 单位：毫秒。
 @see AVPDelegate::onCurrentPositionUpdate
 @see AVPDelegate::onBufferedPositionUpdate
 */
@property(nonatomic, assign) int positionTimerIntervalMs;

/**
 @brief 向前回溯的最大缓冲时长（单位：毫秒）
 默认为0
 */
@property(nonatomic, assign) uint64_t mMAXBackwardDuration;


/**
 @brief 优先保证音频播放
 
 在网络带宽不足的情况下，是否优先保障音频的正常播放而降低视频清晰度。
 目前仅在DASH直播流中有效（视频已切换到最低码率）。
 */
@property (nonatomic, assign) BOOL preferAudio;

/**
 @brief 启用HTTP DNS
 
 是否启用HTTP DNS进行域名解析。取值：-1（跟随全局设置）、0（禁用）或其他值（启用）。
 */
@property(nonatomic, assign) int enableHttpDns;

/**
 @brief 启用增强型HTTP DNS
 
 是否启用增强型HTTP DNS进行域名解析。取值：-1（跟随全局设置）、0（禁用）或其他值（启用）。
 */
@property(nonatomic, assign) int enableEnhancedHttpDns;

/**
 @brief 启用HTTP/3协议
 
 是否使用HTTP/3协议进行请求。支持RFC 9114（HTTP/3）和RFC 9000（QUIC v1）标准。
 如果HTTP/3请求失败，自动降级至普通HTTP。默认值：NO（禁用）。
 */
@property(nonatomic, assign) BOOL enableHttp3;

/**
 @brief 启用严格FLV头检查
 
 用于纯音频或纯视频的RTMP/FLV直播流起播优化策略。当流的header声明只有音频或只有视频，
 且实际流的内容与header声明一致时，启用此选项可加快起播。默认值：NO（禁用）。
 */
@property(nonatomic, assign) BOOL enableStrictFlvHeader;

/**
 @brief 启用严格URL鉴权模式
 
 针对启用了URL鉴权的HLS点播媒体，本地缓存时的鉴权模式：
 - NO（非严格模式）：鉴权结果也被缓存。若上次只缓存部分媒体，下次播放到非缓存部分时，
   会使用缓存的鉴权发起请求。若URL鉴权有效期很短，可能导致播放失败。
 - YES（严格模式）：鉴权结果不缓存，每次起播都进行鉴权。无网络时会导致起播失败。
 
 默认值：YES（版本5.5.4.0~6.21.0默认为NO，7.0.0+默认为YES）。
 */
@property(nonatomic, assign) BOOL enableStrictAuthMode;

/**
 @brief 启用投屏功能
 
 是否允许当前播放器实例进行投屏（投射到外部设备）。
 需要集成投屏SDK来实现完整的投屏功能。默认值：NO（禁用）。
 @deprecated 已废弃
 */
@property(nonatomic, assign) BOOL enableProjection __deprecated;

/**
 @brief 音频打断处理模式
 
 在音频被打断时（如接电话）的处理方式：
 - 0：打断期间暂停播放，打断结束后恢复播放
 - 1：打断期间继续播放
 
 默认值：0。
 */
@property(nonatomic, assign) int audioInterruptMode;

/**
 @brief 视频渲染类型
 
 指定用于视频渲染的渲染器类型：
 - 0：默认渲染器
 - 1：混合渲染器
 - 2：Metal渲染（iOS专用）
 - 3：OpenGL渲染
 
 默认值：0。
 */
@property(nonatomic, assign) int videoRenderType;

/**
 @brief 开始预加载阈值
 
 触发开始预加载的缓冲阈值。当播放缓冲数据高于此值时，开始后台预加载。
 单位：毫秒。
 */
@property(nonatomic, assign) int startBufferLimit;

/**
 @brief 停止预加载阈值
 
 触发停止预加载的缓冲阈值。当播放缓冲数据低于此值时，停止预加载。
 单位：毫秒。
 */
@property(nonatomic, assign) int stopBufferLimit;

/**
 @brief 轨道选择时的缓冲清除模式
 
 调用selectTrack()选择新轨道时，是否清除已缓冲的数据：
 - 0：不清除缓冲
 - 1：清除缓冲
 
 此模式对高清晰度和低清晰度的轨道都适用。
 */
@property(nonatomic, assign) int selectTrackBufferMode;


/**
 @brief 自适应码率最大像素限制
 
 当清晰度切换至AUTO（自动自适应）档位时，允许升路（提高清晰度）的最大像素数。
 例如设置为1280 * 720 = 921600，则最高升路到该清晰度，不会升路到1920 * 1080。
 
 @note 不同清晰度对应的像素数值可参考 @link AVPPixelNumber @endlink
 @note 此设置对 @link AliPlayer::setAutoBandWidth @endlink 接口同样生效
 */
@property(nonatomic, assign) int maxAllowedAbrVideoPixelNumber;

@end

#endif /* AVPConfig_h */
