//
//  AVPDef.h
//  AVPSDK
//
//  Created by shiping.csp on 2018/11/16.
//  Copyright © 2018 com.alibaba.AliyunPlayer. All rights reserved.
//

#ifndef AVPDef_h
#define AVPDef_h

#import <Foundation/Foundation.h>

#if TARGET_OS_OSX
#import <AppKit/AppKit.h>
#define AVPView NSView
#define AVPImage NSImage
#elif TARGET_OS_IPHONE
#import <UIKit/UIKit.h>
#define AVPView UIView
#define AVPImage UIImage
#endif // TARGET_OS_OSX

#import "AVPErrorCode.h"

/**
 * Convert playback URL before playback.
 *
 * For vid playback, try to convert the playback URL before playback
 *
 * @param   srcURL  [in]  input URL.
 * @param   srcFormat [in]  input format. should be "m3u8" "mp4", or "" if unknown.
 * @param   destURL [out] output URL, convertURL function will malloc the memory, and user of PlayURLConverCallback need free it outside.
 *
 * @return  true if success.
 */
typedef bool (*PlayURLConverCallback)(const char* srcURL, const char* srcFormat, char** destURL);


typedef enum AVPStatus: NSUInteger {
    /** @brief 空转，闲时，静态 */
    AVPStatusIdle = 0,
    /** @brief 初始化完成 */
    AVPStatusInitialzed,
    /** @brief 准备完成 */
    AVPStatusPrepared,
    /** @brief 正在播放 */
    AVPStatusStarted,
    /** @brief 播放暂停 */
    AVPStatusPaused,
    /** @brief 播放停止 */
    AVPStatusStopped,
    /** @brief 播放完成 */
    AVPStatusCompletion,
    /** @brief 播放错误 */
    AVPStatusError
} AVPStatus;

 /**@brief 跳转模式，是否为精准跳转*/
typedef enum AVPSeekMode: NSUInteger {
    AVP_SEEKMODE_ACCURATE = 0x01,
    AVP_SEEKMODE_INACCURATE = 0x10,
} AVPSeekMode;

/**@brief 自适应降级切换url的场景，播放器还是预加载器*/
typedef enum BizScene : NSUInteger {
    AVP_Player = 0,
    AVP_Loader = 1
} AVPBizScene;

/**@brief 自适应降级切换url的原始url编码格式*/
typedef enum CodecType : NSUInteger {
    AVP_H265 = 0
} AVPCodecType;


 /**@brief 渲染显示模式*/
typedef enum AVPScalingMode: NSUInteger {
    /**@brief 不保持比例平铺*/
    AVP_SCALINGMODE_SCALETOFILL,
    /**@brief 保持比例，黑边*/
    AVP_SCALINGMODE_SCALEASPECTFIT,
    /**@brief 保持比例填充，需裁剪*/
    AVP_SCALINGMODE_SCALEASPECTFILL,
} AVPScalingMode;

/**@brief 旋转模式*/
typedef enum AVPRotateMode: NSUInteger {
    AVP_ROTATE_0 = 0,
    AVP_ROTATE_90 = 90,
    AVP_ROTATE_180 = 180,
    AVP_ROTATE_270 = 270
} AVPRotateMode;

/**@brief 镜像模式*/
typedef enum AVPMirrorMode: NSUInteger {
    AVP_MIRRORMODE_NONE,
    AVP_MIRRORMODE_HORIZONTAL,
    AVP_MIRRORMODE_VERTICAL,
} AVPMirrorMode;

/**@brief Alpha渲染模式*/
typedef enum AVPAlphaRenderMode: NSUInteger {
    AVP_RENDERMODE_ALPHA_NONE = 0,
    AVP_RENDERMODE_ALPHA_AT_RIGHT = 1,
    AVP_RENDERMODE_ALPHA_AT_LEFT = 2,
    AVP_RENDERMODE_ALPHA_AT_TOP = 3,
    AVP_RENDERMODE_ALPHA_AT_BOTTOM = 4,
} AVPAlphaRenderMode;

/**@brief 音频输出声道*/
typedef enum AVPOutputAudioChannel:NSUInteger {
    /**@brief 不指定声道，默认值*/
    AVP_AUDIO_CHANNEL_NONE = 0,
    /**@brief 左声道*/
    AVP_AUDIO_CHANNEL_LEFT = 1,
    /**@brief 右声道*/
    AVP_AUDIO_CHANNEL_RIGHT = 2
} AVPOutputAudioChannel;

/**@brief 播放器事件类型*/
typedef enum AVPEventType: NSUInteger {
    /**@brief 准备完成事件*/
    AVPEventPrepareDone,
    /**@brief 自动启播事件*/
    AVPEventAutoPlayStart,
    /**@brief 首帧显示事件*/
    AVPEventFirstRenderedStart,
    /**@brief 播放完成事件*/
    AVPEventCompletion,
    /**@brief 缓冲开始事件*/
    AVPEventLoadingStart,
    /**@brief 缓冲完成事件*/
    AVPEventLoadingEnd,
    /**@brief 跳转完成事件*/
    AVPEventSeekEnd,
    /**@brief 循环播放开始事件*/
    AVPEventLoopingStart,
    /**@brief 清屏完成事件*/
    AVPEventClearScreenDone,
} AVPEventType;

/**@brief 获取信息播放器的key*/
typedef enum AVPPropertyKey: NSUInteger {
    /**@brief Http的response信息
     * 返回的字符串是JSON数组，每个对象带response和type字段。type字段可以是url/video/audio/subtitle，根据流是否有相应Track返回。
     * 例如：[{"response":"response string","type":"url"},{"response":"","type":"video"}]
     */
    AVP_KEY_RESPONSE_INFO = 0,

    /**@brief 主URL的连接信息
     * 返回的字符串是JSON对象，带url/ip/eagleID/cdnVia/cdncip/cdnsip等字段（如果解析不到则不添加）
     * 例如：{"url":"http://xxx","openCost":23,"ip":"11.111.111.11","cdnVia":"xxx","cdncip":"22.222.222.22","cdnsip":"xxx"}
     */
    AVP_KEY_CONNECT_INFO  = 1,
} AVPPropertyKey;

/**@brief IP 解析类型*/
/**@brief IP resolve type*/
typedef enum AVPIpResolveType: NSUInteger {
    AVPIpResolveWhatEver,
    AVPIpResolveV4,
    AVPIpResolveV6,
} AVPIpResolveType;

typedef enum AVPOption : NSUInteger {
    /**
     * @brief 渲染的fps。类型为Float
     */
    AVP_OPTION_RENDER_FPS = 0,

    /**
     * 当前的网络下行码率。类型为Float
     */
    AVP_OPTION_DOWNLOAD_BITRATE = 1,

    /**
     * 当前播放的视频码率。类型为Float
     */
    AVP_OPTION_VIDEO_BITRATE = 2,

    /**
     * 当前播放的音频码率。类型为Float
     */
    AVP_OPTION_AUDIO_BITRATE = 3,
    
    /**
     * 切换档位为AUTO时，判断当前是否处于ABR切换的状态。返回类型为Int, 当处于ABR切换状态时，返回"1"， 否则返回"0".
     */
    AVP_OPTION_IS_ABRSWITCHING = 4,
} AVPOption;

/**
 * 策略类型
 */
typedef enum AVPStrategyType : NSUInteger {
    /**
     * 动态预加载时长策略
     */
    AVP_STRATEGY_DYNAMIC_PRELOAD = 1,
} AVPStrategyType;

/**
 * 多码率预加载类型，只对多码率HLS流生效
 */
typedef enum AVPMultiBitratesMode : NSUInteger {
    /**
     * 默认配置，播放和预加载默认码率
     */
    AVPMultiBitratesMode_Default = 0,
    /**
     * 首帧优先配置，起播视频默认播放已完成预加载的码率
     */
    AVPMultiBitratesMode_FCPrio = 1,
    /**
     * 兼顾首帧和播放平滑，切换前后（moveToNext）的视频码率一致，且兼顾首帧性能
     */
    AVPMultiBitratesMode_FC_AND_SMOOTH = 2,
    /**
     * 播放平滑优先配置，起播视频默认播放前一个视频的码率
     */
    AVPMultiBitratesMode_SmoothPrio = 3,
} AVPMultiBitratesMode;


/**
 * 场景类型
 */
typedef enum AVPSceneType : NSInteger {
    /**
     * 场景：无
     */
    AVP_SCENE_NONE = -1,
    /**
     * 超短视频场景：适用于30s以下
     */
    AVP_VERY_SHOR_VIDEO = 0,
    /**
     * 短视频场景：适用于30s-5min
     */
    AVP_SHORT_VIDEO = 1,
    /**
     * 中视频场景：适用于5min-30min
     */
    AVP_MIDDLE_VIDEO = 2,
    /**
     * 长视频场景：适用于30min以上
     */
    AVP_LONG_VIDEO = 3,
} AVPSceneType;


/**
 * 画中画显示模式
 */
typedef enum AVPPIPShowMode : NSUInteger {
    /**
     * 默认模式：正常显示所有画中画按钮
     */
    AVP_SHOW_MODE_DEFAULT = 0,
    /**
     * 隐藏快进快退模式
     */
    AVP_SHOW_MODE_HIDE_FAST_FORWARD_REWIND = 1,
} AVPPIPShowMode;

/**
 @brief AVPErrorModel为播放错误信息描述
 */
OBJC_EXPORT
@interface AVPErrorModel : NSObject

/**
 @brief code为播放错误信息code
 */
@property (nonatomic, assign) AVPErrorCode code;

/**
 @brief message为播放错误信息描述
 */
@property (nonatomic, copy) NSString *message;

/**
 @brief extra为播放错误信息描述的附加信息
 */
@property (nonatomic, copy) NSString *extra;

/**
 @brief requestId为播放错误信息requestID
 */
@property (nonatomic, copy) NSString *requestId;

/**
 @brief videoId为播放错误发生的videoID
 */
@property (nonatomic, copy) NSString *videoId;

@end


/***
@brief AVPNetworkLevel 网络质量级别枚举
 */
/***
@brief AVPNetworkLevel represents network quality levels.
*/
typedef enum AVPNetworkLevel: NSInteger {
    /***
    @brief 网络状态未知。
     */
    /***
    @brief The network status is unknown.
    */
    AVPNetwork_UNKNOWN = -1,
    
    /**
     @brief 网络质量非常优秀。
     */
    AVPNetwork_EXCELLENT = 0,
    
    /**
     @brief 网络质量良好。
     */
    AVPNetwork_GOOD = 1,
    
    
    /**
     @brief 网络质量较差。
     */
    AVPNetwork_POOR = 2,
    
    /**
     @brief 网络质量很差。
     */
    AVPNetwork_BAD = 3,
    
    /**
     @brief 网络质量非常差。
     */
    AVPNetwork_VERY_BAD = 4,
    
    /**
     @brief 网络已断开连接。
     */
    AVPNetwork_DISCONNECTED = 5,
} AVPNetworkLevel;

/**
@brief 网络统计信息类，用于收集和表示播放器应用的网络状况。
*/
/***
@brief AVPNetworkStat represents network statistics for the player.
It is used to collect and describe network-related metrics.
*/
OBJC_EXPORT
@interface AVPNetworkStat : NSObject

/**
@brief 视频下载速度，单位 Kbps。默认值为 0.0。
*/
/***
@brief Video download speed in kilobits per second (Kbps). Default is 0.0.
*/
@property (nonatomic, assign) double videoDownloadSpeedKbps;

/**
@brief 音频下载速度，单位 Kbps。默认值为 0.0。
*/
/***
@brief Audio download speed in kilobits per second (Kbps). Default is 0.0.
*/
@property (nonatomic, assign) double audioDownloadSpeedKbps;

/**
@brief 往返时延（RTT），单位毫秒（ms）。表示从客户端到服务器再返回所需的时间。默认值为 0。
*/
/***
@brief Round-trip time (RTT) in milliseconds (ms).
Indicates the time for a packet to travel from client to server and back.
Default is 0.
*/
@property (nonatomic, assign) int64_t rttMs;

/**
@brief 丢包率（百分比），范围 0 ~ 100%。较高的丢包率可能意味着网络质量差。默认值为 0。
*/
/***
@brief Packet loss rate as a percentage, ranging from 0 to 100%.
A higher packet loss rate may indicate poor network quality. Default is 0.
*/
@property (nonatomic, assign) int64_t packetLossRatePercentage;

/**
@brief 当前网络质量级别，使用 AVPNetworkLevel 表示。默认值为 AVPNetwork_UNKNOWN。
*/
/***
@brief The current network quality level, represented by AVPNetworkLevel.
Default is AVPNetwork_UNKNOWN.
*/
@property (nonatomic, assign) AVPNetworkLevel level;

@end


/**
 @brief AVPTimeShiftModel直播时移描述
 */
OBJC_EXPORT
@interface AVPTimeShiftModel : NSObject

/**
 @brief startTime直播时移开始时间
 */
@property (nonatomic, assign) NSTimeInterval startTime;

/**
 @brief endTime直播时移结束时间
 */
@property (nonatomic, assign) NSTimeInterval endTime;

/**
 @brief currentTime直播时移当前时间
 */
@property (nonatomic, assign) NSTimeInterval currentTime;
@end

/**
 @brief logLevel
 */
typedef enum AVPLogLevel: NSUInteger {
    LOG_LEVEL_NONE    = 0,
    LOG_LEVEL_FATAL   = 8,
    LOG_LEVEL_ERROR   = 16,
    LOG_LEVEL_WARNING = 24,
    LOG_LEVEL_INFO    = 32,
    LOG_LEVEL_DEBUG   = 48,
    LOG_LEVEL_TRACE   = 56,
} AVPLogLevel;

/**
 @brief pixelNumber of specific Resolution, for reference only.
 */
typedef enum AVPPixelNumber: NSUInteger {
    Resolution_360P = 172800, //480 * 360
    Resolution_480P = 345600, //720 * 480
    Resolution_540P = 518400, //960 * 540
    Resolution_720P = 921600, //1280 * 720
    Resolution_1080P = 2073600,//1920 * 1080
    Resolution_2K = 3686400, //2560 * 1440
    Resolution_4K = 8847360, //4096 * 2160
    Resolution_NoLimit = INT_MAX,
} AVPPixelNumber;

typedef enum AVPLogOption: NSUInteger  {
    FRAME_LEVEL_LOGGING_ENABLED = 1,
}AVPLogOption;

typedef enum _AVPStsStatus {
  Valid,
  Invalid,
  Pending } AVPStsStatus;

typedef struct _AVPStsInfo {
    NSString *accId;
    NSString *accSecret;
    NSString *token;
    NSString *region;
    NSString *formats;
} AVPStsInfo;

/**
 @brief 是否支持的功能的类型
 */
typedef enum _SupportFeatureType : NSUInteger {
    /**
     * 硬件是否支持杜比音频
    */
    FeatureDolbyAudio
} SupportFeatureType;

/**
 @brief GlobalSettings的特定功能选项
 */
typedef enum _GlobalOption: NSUInteger {
    SET_PRE_CONNECT_DOMAIN = 0,
    SET_DNS_PRIORITY_LOCAL_FIRST = 1,
    ENABLE_H2_MULTIPLEX = 2,
    SET_EXTRA_DATA = 3,
    ENABLE_ANDROID_DECODE_REUSE = 4, //android only, placeholder
    NOT_PAUSE_WHEN_PREPARING = 5, // do not pause when preparing
    ALLOW_RTS_DEGRADE = 6,
    ENABLE_DECODER_FAST_FIRST_FRAME = 7,
    DISABLE_CAPTURE_SCALE = 8,
    
    ALLOW_BOUNDS_CHANGE_ANIMATION = 10, // iOS only
    AV3A_DECODE_MODEL_PATH = 11,
//    ENABLE_DECODER_REUSE_CROSS_INSTANCE = 12, //android only
//    DECODER_POOL_CAPACITY_CROSS_INSTANCE = 13, //android only
    RENDER_IGNORE_DAR_SCALE = 14, //iOS only
    MAX_ERROR_FRAMES_HARDWARE_DECODE = 15,
    ALLOW_AUDIOUNIT = 16, //iOS only
    DISABLE_CATCHUP_IN_LOWLATENCY_AUDIOQUEUE = 17, //iOS only
} GlobalOption;

/**
 @brief AliPlayer的特定功能选项
 */
typedef enum _PlayerOption: NSUInteger {
    SET_MEDIA_TYPE = 0,
    ALLOW_DECODE_BACKGROUND = 1,
    ALLOW_PRE_RENDER = 2,
    PLAYED_DURATION_INCLUDE_SPEED = 3,
    // it works only if ALLOW_PRE_RENDER is true
    PRE_RENDER_MAX_BUFFER_MS = 5,
    LINE_FALLBACK_URL = 6,
} PlayerOption;

typedef enum _AVPScene {
    /**
     * 场景：无
     */
    SceneNone,
    /**
     * 长视频场景：适用于30min以上
     */
    SceneLong,
    /**
     * 中视频场景：适用于5min-30min
     */
    SceneMedium,
    /**
     * 短视频场景：适用于0s-5min
     */
    SceneShort,
    /**
     * 直播场景
     */
    SceneLive
} AVPScene;

#endif /* AVPDef_h */
