"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SrcCoverage = void 0;
const coverage = require("istanbul-lib-coverage");
const FileCoverage = coverage.classes.FileCoverage;
const cloneLoc = (loc) => {
    let locObj = {
        start: {
            line: loc && loc.start.line,
            column: loc && loc.start.column,
        },
        end: {
            line: loc && loc.end.line,
            column: loc && loc.end.column,
        },
    };
    return locObj;
};
/**
 * SourceCoverage provides mutation methods to manipulate the structure of
 * a file coverage object. Used by the instrumenter to create a full coverage
 * object for a file incrementally.
 *
 * @private
 * @param pathOrObj {String|Object} - see the argument for {@link FileCoverage}
 * @extends FileCoverage
 * @constructor
 */
class SrcCoverage extends FileCoverage {
    constructor(pathOrObj) {
        super(pathOrObj);
        this.meta = {
            final: {
                s: 0,
                f: 0,
                b: 0,
            },
        };
    }
    newStatement(loc) {
        const s = this.meta.final.s;
        this.data.statementMap[s] = cloneLoc(loc);
        this.data.s[s] = 0;
        this.meta.final.s += 1;
        return s;
    }
    newFunction(name, decl, loc) {
        const f = this.meta.final.f;
        name = name || `(anonymous_${f})`;
        this.data.fnMap[f] = {
            name,
            decl: cloneLoc(decl),
            loc: cloneLoc(loc),
        };
        this.data.f[f] = 0;
        this.meta.final.f += 1;
        return f;
    }
    newBranch(type, loc, isReportLogic = false) {
        const b = this.meta.final.b;
        this.data.b[b] = [];
        this.data.branchMap[b] = {
            loc: cloneLoc(loc),
            type,
            locations: [],
            line: loc && loc.start.line,
        };
        this.meta.final.b += 1;
        this.maybeBranchTrue(type, b, isReportLogic);
        return b;
    }
    maybeBranchTrue(type, name, isReportLogic) {
        if (!isReportLogic) {
            return;
        }
        if (type !== 'binary-expr') {
            return;
        }
        this.data.bT = this.data.bT || {};
        this.data.bT[name] = [];
    }
    addBranchPath(name, location) {
        const bMeta = this.data.branchMap[name];
        const counts = this.data.b[name];
        /* istanbul ignore if: paranoid check */
        if (!bMeta) {
            throw new Error('Invalid branch ' + name);
        }
        bMeta.locations.push(cloneLoc(location));
        counts.push(0);
        this.maybeAddBranchTrue(name);
        return counts.length - 1;
    }
    maybeAddBranchTrue(name) {
        if (!this.data.bT) {
            return;
        }
        const countsTrue = this.data.bT[name];
        if (!countsTrue) {
            return;
        }
        countsTrue.push(0);
    }
    getInputSource(sourceMap) {
        this.data.inputSourceMap = sourceMap;
    }
    deleteBranch() {
        const branchMap = this.data.branchMap;
        const b = this.data.b;
        const bT = this.data.bT ? this.data.bT : {};
        for (let key of Object.keys(branchMap)) {
            if (branchMap[key].locations.length === 0) {
                delete branchMap[key];
                delete b[key];
                delete bT[key];
            }
        }
    }
}
exports.SrcCoverage = SrcCoverage;
