# hos-sdkmanager-js

HarmonyOS SDK Manager的js实现
## api说明
```ts
/**
 * OneHarmonyOS SDKInfoHandler
 * HarmonyOS发布的SDK与OpenHarmony发布的SDK应该都在实现该接口，原有需要获取HarmonyOS页签下SDK都可以通过该接口获取
 *
 * @since 2023-05-04
 */
export interface UniSdkInfoHandler {

  /**
   * 根据version获取SDK，这个version可以是HarmonyOS Version，可以是API Version，根据实现，其含义会存在区别
   *
   * 注意：由于HarmonyOS Version 3.0.0同时对应api7和api8，参数传入3.0.0时，这里无法返回多个值，因此做特殊处理，只返回api8的SDK，
   * 正常情况下调用者查询api<9的SDK时只会传入OpenHarmony的API Version，所以正常调用是没有问题的
   *
   * @param version apiVersion的字符串形式或者HarmonyOSVersion
   * @return path与Component的键值对
   */
  getLocalSdks(version: string): Map<string, Component>;

  /**
   * 根据version获取SDK，这个version可以是HarmonyOS Version，可以是API Version，根据实现，其含义会存在区别
   * 如果SDK不存在，就去下载
   *
   * @param version version apiVersion的字符串形式或者HarmonyOSVersion
   * @param paths 需要的SDK的paths集合
   * @return path与Component的键值对
   */
  getOrDownload(version: string, paths: string[]): Promise<Map<string, Component>>;
}
```
## 使用方法

```ts
import { SimpleProgress } from "@ohos/sdkmanager-common";

const prjSdkConfig = HosPrjSdkConfig.builder('sdk根路径').nodeHome('node路径').npmConfig('.npmrc路径').sdkProxy('代理信息');
// 可以自己传入一个自定义的progress用于打印sdk内部的日志信息，这里的SimpleProgress不打印任何日志
const handler = new SimpleHosPrjSdkHandler(prjSdkConfig, new SimpleProgress());

1、获取HarmonyOS SDK
const hosSdkInfoHandler = handler.getSdkHandler(HosPrjSdkType.HARMONYOS);
// 获取本地hosVersion为3.1.0的HarmonyOS SDK
const localSdks = hosSdkInfoHandler.getLocalSdks('3.1.0');
// 从本地hosVersion为3.1.0的SDK中查找toolchains
const toolchains = localSdks.get('toolchains');
// 获取hosVersion为3.1.0的toolchains和ets，如果本地不存在该SDK，则会触发下载
const downloadSdks = hosSdkInfoHandler.getOrDownload('3.1.0', ['toolchains', 'ets'])
// 从下载的SDK中获取toolchains
const toolchains = downloadSdks.get('toolchains');

2、获取HarmonyOS下面的OpenHarmony SDK
const ohSdkInfoHandler = handler.getSdkHandler(HosPrjSdkType.OPENHARMONY);
// 获取本地HarmonyOS 下的API9的OpenHarmony SDK
const localSdks = ohSdkInfoHandler.getLocalSdks('9');
// 从API9的SDK中查找toolchains
const toolchains = localSdks.get('toolchains');
// 获取本地HarmonyOS 下的API9的OpenHarmony SDK(toolchains和ets)，如果本地不存在该SDK，则会触发下载
const downloadSdks = ohSdkInfoHandler.getOrDownload('9', ['toolchains', 'ets'])
// 从下载的SDK中获取toolchains
const toolchains = downloadSdks.get('toolchains');
```

## 引用方式
```json
"dependencies": {
  "@ohos/hos-sdkmanager-common": "x.x.x"
},
```