"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.unzip = void 0;
const jszip_1 = __importDefault(require("jszip"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const sdk_exception_1 = require("../api/sdk-exception");
const error_tip_1 = require("../api/error-tip");
/**
 * 解压后的Zip文件最大数量，含目录、文件、符号链接等
 */
const MAX_ZIP_ENTRY_NUM = 100000;
/**
 * 解压后的zip文件总大小的最大size，20G
 */
const MAX_ZIP_SIZE = 20 * 1024 * 1024 * 1024;
const unzip = (sourcePath, dest) => __awaiter(void 0, void 0, void 0, function* () {
    if (!fs_1.default.existsSync(sourcePath)) {
        throw new sdk_exception_1.SdkException('unzip failed', new error_tip_1.ErrorTip('', `Unable to find file ${sourcePath}`, ''));
    }
    const buffer = fs_1.default.readFileSync(sourcePath);
    if (buffer.length > MAX_ZIP_SIZE) {
        throw new sdk_exception_1.SdkException('unzip failed', new error_tip_1.ErrorTip('', `Zip size is too big, exceed ${MAX_ZIP_SIZE} bytes`, ''));
    }
    const jsZip = yield jszip_1.default.loadAsync(buffer, { createFolders: true });
    return saveZipFiles(dest, jsZip.files);
});
exports.unzip = unzip;
const saveZipFiles = (savePath, files) => __awaiter(void 0, void 0, void 0, function* () {
    if (Object.keys(files).length > MAX_ZIP_ENTRY_NUM) {
        throw new sdk_exception_1.SdkException('unzip failed', new error_tip_1.ErrorTip('', `Zip entry is too many, exceed max ${MAX_ZIP_ENTRY_NUM}`, ''));
    }
    try {
        // 获取解压后的文件
        for (const filename of Object.keys(files)) {
            const dest = path_1.default.join(savePath, filename);
            // 如果该文件为目录需先创建文件夹
            if (files[filename].dir) {
                const unixPermissions = files[filename].unixPermissions;
                const options = {
                    recursive: true,
                };
                if (unixPermissions) {
                    options.mode = unixPermissions;
                }
                fs_1.default.mkdirSync(dest, options);
            }
            else {
                // 把每个文件buffer写到硬盘中
                const content = yield files[filename].async('nodebuffer');
                // 设置权限，保持权限不变
                const unixPermissions = files[filename].unixPermissions;
                if (!unixPermissions) {
                    fs_1.default.writeFileSync(dest, content);
                    continue;
                }
                // true, this is a symlink
                if (typeof unixPermissions === 'number' && (unixPermissions & 0o170000) === 0o120000) {
                    const relativePath = path_1.default.relative(path_1.default.resolve(dest, '..'), path_1.default.resolve(dest, '..', content.toString()));
                    fs_1.default.symlinkSync(relativePath, dest);
                }
                else {
                    fs_1.default.writeFileSync(dest, content, { mode: unixPermissions });
                }
            }
        }
    }
    catch (error) {
        throw new sdk_exception_1.SdkException('unzip failed', new error_tip_1.ErrorTip('', error === null || error === void 0 ? void 0 : error.message, ''));
    }
});
