"use strict";
/**
 * 创建一个会缓存 func 结果的函数。 如果提供了 resolver ，就用 resolver 的返回值作为 key 缓存函数的结果。
 * 默认情况下用第一个参数作为缓存的 key。 func 在调用时 this 会绑定在缓存函数上。
 * 注意: 缓存会暴露在缓存函数的 cache 上。 它是可以定制的，只要替换了 _.memoize.Cache 构造函数，或实现了Map 的 delete, get, has, 和 set方法。
 * 参数
 *   1.func (Function): 需要缓存化的函数.
 *   2.[resolver] (Function): 这个函数的返回值作为缓存的 key。
 * 返回
 *   (Function): 返回缓存化后的函数。
 *
 * @category Function
 * @param func
 * @param {Function} [resolver] 用于指定缓存key的函数
 * @returns {Function} 返回已经缓存的函数
 * @example
 *
 * var object = { 'a': 1, 'b': 2 };
 * var other = { 'c': 3, 'd': 4 };
 * var values = _.memoize(_.values);
 * values(object);
 * // => [1, 2]
 * values(other);
 * // => [3, 4]
 * object.a = 2;
 * values(object);
 * // => [1, 2]
 * // 修改结果缓存。
 * values.cache.set(object, ['a', 'b']);
 * values(object);
 * // => ['a', 'b']
 * // 替换 `_.memoize.Cache`。
 * _.memoize.Cache = WeakMap;
 */
Object.defineProperty(exports, "__esModule", { value: true });
function memoize(func, resolver) {
    if (typeof func !== 'function' || (resolver != null && typeof resolver !== 'function')) {
        throw new TypeError('Expected a function');
    }
    function memoized() {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        var key = resolver ? resolver.apply(this, args) : args[0];
        var cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var res = func.apply(this, args);
        memoized.cache = cache.set(key, res) || cache;
        return res;
    }
    memoized.cache = new (memoize.Cache || Map);
    return memoized;
}
memoize.Cache = Map;
exports.default = memoize;
