"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instrumenter = void 0;
const core_1 = require("@babel/core");
const schema_1 = require("@istanbuljs/schema");
const visitor_1 = require("./visitor");
const readCoverage_1 = require("./readCoverage");
/**
 * Instrumenter类是该库的一个公共API
 * @param {Object} opts optional.
 * @param {string} [opts.coverageVariable=__coverage__] name of global coverage variable.
 * @param {boolean} [opts.compact=true] generate compact code.
 * @param {boolean} [opts.esModules=false] set to true to instrument ES6 modules.
 * @param {boolean} [opts.reportLogic=false] report boolean value of logical expressions.
 * @param {boolean} [opts.preserveComments=false] preserve comments in output.
 * @param {boolean} [opts.autoWrap=false] set to true to allow `return` statements outside of functions..
 * @param {array} [opts.parserPlugins] - set babel parser plugins, see @istanbuljs/schema for defaults.
 * @param {string} [opts.coverageGlobalScope=this] the global coverage variable scope.
 * @param {Array} [opts.ignoreClassMethods=[]] set to array of class method names to ignore for coverage.
 * @param {Function} [opts.sourceMapUrlCallback=null] a callback function that is called when a source map URL
 *     is found in the original code. This function is called with the source file name and the source map URL.
 * @param {boolean} [opts.debug=false] - turn debugging on.
 * @param {boolean} [opts.produceSourceMap=false] set to true to produce a source map for the instrumented code
 * @param {boolean} [opts.coverageGlobalScopeFunc=true] use an evaluated function to find coverageGlobalScope.
 */
class Instrumenter {
    constructor(opts = {}) {
        this.opts = {
            ...schema_1.defaults.instrumenter,
            ...opts,
        };
        this.fileCoverage = null;
        this.sourceMap = null;
        this.instrumentSync = this.instrumentSync.bind(this);
    }
    instrumentSync(code, filename, inputSourceMap) {
        if (typeof code !== 'string') {
            throw new Error('Code must be a string');
        }
        filename = filename || String(new Date().getTime()) + '.js';
        const { opts } = this;
        const ret = this.getBabelOpts(code, filename, opts, inputSourceMap);
        const res = ret.res;
        const transformRes = ret.transformRes;
        if (!res || !res.fileCoverage) {
            const initialCoverage = (0, readCoverage_1.readInitialCoverage)(res.ast) || {};
            this.fileCoverage = initialCoverage.coverageData;
            this.sourceMap = inputSourceMap;
            return code;
        }
        this.fileCoverage = res.fileCoverage;
        this.sourceMap = transformRes.map;
        return transformRes.code;
    }
    getBabelOpts(code, filename, opts, inputSourceMap) {
        let res = {};
        const babelOpts = {
            filename: filename || String(new Date().getTime()) + '.js',
            configFile: false,
            babelrc: false,
            ast: true,
            compact: opts.compact,
            comments: opts.preserveComments,
            inputSourceMap,
            sourceMaps: opts.produceSourceMap,
            parserOpts: {
                allowReturnOutsideFunction: opts.autoWrap,
                sourceType: opts.esModules ? 'module' : 'script',
                plugins: opts.parserPlugins,
            },
            plugins: [
                [
                    ({ types }) => {
                        const pv = (0, visitor_1.programVisitor)(types, filename, {
                            coverageVariable: opts.coverageVariable,
                            reportLogic: opts.reportLogic,
                            coverageGlobalScope: opts.coverageGlobalScope,
                            coverageGlobalScopeFunc: opts.coverageGlobalScopeFunc,
                            ignoreClassMethods: opts.ignoreClassMethods,
                            inputSourceMap,
                        });
                        return {
                            visitor: {
                                Program: {
                                    enter: pv.enter,
                                    exit(path) {
                                        res = pv.exit(path);
                                    },
                                },
                            },
                        };
                    },
                ],
            ],
        };
        const transformRes = (0, core_1.transformSync)(code, babelOpts);
        return { res, transformRes };
    }
    /**
     * 返回最近一次的文件覆盖率对象
     * @returns the file coverage object.
     */
    lastFileCoverage() {
        return this.fileCoverage;
    }
    /**
     * 返回最近一次的文件的sourcemap
     * @returns the source map object.
     */
    lastSourceMap() {
        return this.sourceMap;
    }
}
exports.Instrumenter = Instrumenter;
