"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommonSdkInfoHandler = void 0;
const path_and_apiversion_1 = require("./path-and-apiversion");
const sdk_exception_1 = require("./sdk-exception");
const error_tip_1 = require("./error-tip");
const os_1 = __importDefault(require("os"));
const path_and_api_map_1 = require("../util/path-and-api-map");
const component_contants_1 = require("../constants/component-contants");
class CommonSdkInfoHandler {
    constructor(sdkSettings) {
        this.sdkSettings = sdkSettings;
    }
    getLocalSdks() {
        const localComponents = this.getLocalComponents();
        if (localComponents === null) {
            return new Map();
        }
        return this.mapComponents(localComponents);
    }
    getOrDownload(pathAndApiVersions) {
        const localSdks = this.getLocalSdks();
        return this.doGetOrDownload(pathAndApiVersions, localSdks);
    }
    /**
     * doGetOrDownload
     *
     * @param pathAndApiVersions
     * @param localSdks
     * @param downConsumer
     * @protected
     */
    doGetOrDownload(pathAndApiVersions, localSdks, downConsumer) {
        return __awaiter(this, void 0, void 0, function* () {
            const localComponents = this.mapComponents(this.getLocalComponents());
            const existRequiredSdks = this._filterRequiredSdks(localComponents, pathAndApiVersions);
            const missingPathAndApiVersions = this._findMissingComponents(existRequiredSdks, pathAndApiVersions);
            if (missingPathAndApiVersions.length === 0) {
                return Promise.resolve().then(() => existRequiredSdks);
            }
            const missingComponents = yield this._downloadComponents(missingPathAndApiVersions, this.getProgress(), downConsumer);
            return this._mergeComponents(existRequiredSdks, missingComponents);
        });
    }
    _filterRequiredSdks(localComponents, pathAndApiVersions) {
        const resultMap = new path_and_api_map_1.PathAndApiMap();
        for (const pathAndApiVersion of pathAndApiVersions) {
            const component = localComponents.get(pathAndApiVersion);
            if (!component) {
                continue;
            }
            resultMap.set(pathAndApiVersion, component);
        }
        return resultMap;
    }
    _findMissingComponents(localComponents, expectedComponents) {
        return expectedComponents.filter(pathAndApiVersion => !localComponents.has(pathAndApiVersion));
    }
    _downloadComponents(missingPathAndApiVersions, progress, downConsumer) {
        return __awaiter(this, void 0, void 0, function* () {
            const remoteComponents = this.mapComponents(yield this.getRemoteComponents());
            const componentsToBeInstalled = [];
            const notFoundComponents = [];
            for (const pathAndApiVersion of missingPathAndApiVersions) {
                const remote = remoteComponents.get(pathAndApiVersion);
                if (remote) {
                    downConsumer === null || downConsumer === void 0 ? void 0 : downConsumer.call(this, remote);
                    componentsToBeInstalled.push(remote);
                }
                else {
                    notFoundComponents.push(pathAndApiVersion);
                }
            }
            if (notFoundComponents.length !== 0) {
                return this._throwNotFoundException(notFoundComponents);
            }
            yield this._checkLicense(componentsToBeInstalled);
            yield this._installComponents(componentsToBeInstalled, progress);
            return this.mapComponents(componentsToBeInstalled);
        });
    }
    _checkLicense(componentsToBeInstalled) {
        return __awaiter(this, void 0, void 0, function* () {
            const unacceptedLicenses = yield this.findUnacceptedLicenses(componentsToBeInstalled);
            if (unacceptedLicenses === undefined) {
                return Promise.reject(new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip('', 'Unable to obtain the license.', 'Check the network and HTTP proxy settings, and then try again.')));
            }
            if (unacceptedLicenses.length >= 1) {
                const license = unacceptedLicenses[0]; // 由于license相同，因此只取一个license就可以了
                const fullNames = componentsToBeInstalled
                    .filter(component => component.getLicense() === license.id)
                    .map(component => `${component.getFullName()}`);
                return Promise.reject(new sdk_exception_1.SdkException('', this.getLicenseNotAcceptedMessage(fullNames)));
            }
        });
    }
    _installComponents(componentsToBeInstalled, progress) {
        return __awaiter(this, void 0, void 0, function* () {
            for (const component of componentsToBeInstalled) {
                progress.info(`Installing ${component.getFullName()}...`);
                try {
                    yield this.createInstallComponentTask(component).execute(progress);
                }
                catch (e) {
                    if (e instanceof sdk_exception_1.SdkException) {
                        return Promise.reject(new sdk_exception_1.SdkException(`Install ${component.getFullName()} failed`, e.getErrorTip()));
                    }
                    else {
                        return Promise.reject(new sdk_exception_1.SdkException(`Install ${component.getFullName()} failed`, new error_tip_1.ErrorTip(e === null || e === void 0 ? void 0 : e.message, e === null || e === void 0 ? void 0 : e.message, '')));
                    }
                }
            }
        });
    }
    _throwNotFoundException(notFoundComponents) {
        let msg = '';
        for (const pathAndApiVersion of notFoundComponents) {
            const path = this._decoratePath(pathAndApiVersion.getPath());
            if (pathAndApiVersion.getApiVersion() !== 0) {
                msg += `${os_1.default.EOL}\t\t${path}:${pathAndApiVersion.getApiVersion()}`;
            }
            else {
                msg += `${os_1.default.EOL}\t\t${path}`;
            }
        }
        const sdkException = new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip('', `Unable to find the following components:${msg}`, `
\t\t${this.getNotFoundException()}
\t\t2.Alternatively, modify the compileSdkVersion settings in the project-level build-profile.json5 file.`));
        return Promise.reject(sdkException);
    }
    getNotFoundException() {
        return '1.Go to Tools > SDK Manager > SDK > OpenHarmony, download the components, and sync the project. Open SDK' +
            ' Manager';
    }
    _decoratePath(path) {
        if (component_contants_1.ComponentPath.ETS === path) {
            return 'ArkTS'; // lib库显示ets的地方要换个名称显示
        }
        return path;
    }
    _mergeComponents(existRequiredSdks, missingComponents) {
        for (const entry of existRequiredSdks.entries()) {
            const key = entry[0];
            const value = entry[1];
            if (!missingComponents.has(key)) {
                missingComponents.set(key, value);
            }
        }
        return missingComponents;
    }
    mapComponents(components) {
        const map = components.map(value => {
            const tuple = [
                new path_and_apiversion_1.PathAndApiVersion(value.getPath(), value.getApiVersion()),
                value,
            ];
            return tuple;
        });
        return new path_and_api_map_1.PathAndApiMap(map);
    }
}
exports.CommonSdkInfoHandler = CommonSdkInfoHandler;
