'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var process = require('process');
var worker_threads = require('worker_threads');
var smob = require('smob');
var terser$1 = require('terser');
var url = require('url');
var os = require('os');
var events = require('events');
var serializeJavascript = require('serialize-javascript');

/**
 * Duck typing worker context.
 *
 * @param input
 */
function isWorkerContextSerialized(input) {
    return (smob.isObject(input) &&
        smob.hasOwnProperty(input, 'code') &&
        typeof input.code === 'string' &&
        smob.hasOwnProperty(input, 'options') &&
        typeof input.options === 'string');
}
async function runWorker() {
    if (worker_threads.isMainThread || !worker_threads.parentPort || !isWorkerContextSerialized(worker_threads.workerData)) {
        return;
    }
    try {
        // eslint-disable-next-line no-eval
        const eval2 = eval;
        const options = eval2(`(${worker_threads.workerData.options})`);
        const result = await terser$1.minify(worker_threads.workerData.code, options);
        const output = {
            code: result.code || worker_threads.workerData.code,
            nameCache: options.nameCache
        };
        if (typeof result.map === 'string') {
            output.sourceMap = JSON.parse(result.map);
        }
        if (smob.isObject(result.map)) {
            output.sourceMap = result.map;
        }
        worker_threads.parentPort.postMessage(output);
    }
    catch (e) {
        process.exit(1);
    }
}

const symbol = Symbol.for('FreeWoker');
class WorkerPool extends events.EventEmitter {
    constructor(options) {
        super();
        this.tasks = [];
        this.workers = 0;
        this.maxInstances = options.maxWorkers || os.cpus().length;
        this.filePath = options.filePath;
        this.on(symbol, () => {
            if (this.tasks.length > 0) {
                this.run();
            }
        });
    }
    add(context, cb) {
        this.tasks.push({
            context,
            cb
        });
        if (this.workers >= this.maxInstances) {
            return;
        }
        this.run();
    }
    async addAsync(context) {
        return new Promise((resolve, reject) => {
            this.add(context, (err, output) => {
                if (err) {
                    reject(err);
                    return;
                }
                if (!output) {
                    reject(new Error('The output is empty'));
                    return;
                }
                resolve(output);
            });
        });
    }
    run() {
        if (this.tasks.length === 0) {
            return;
        }
        const task = this.tasks.shift();
        if (typeof task === 'undefined') {
            return;
        }
        this.workers += 1;
        let called = false;
        const callCallback = (err, output) => {
            if (called) {
                return;
            }
            called = true;
            this.workers -= 1;
            task.cb(err, output);
            this.emit(symbol);
        };
        const worker = new worker_threads.Worker(this.filePath, {
            workerData: {
                code: task.context.code,
                options: serializeJavascript(task.context.options)
            }
        });
        worker.on('message', (data) => {
            callCallback(null, data);
        });
        worker.on('error', (err) => {
            callCallback(err);
        });
        worker.on('exit', (code) => {
            if (code !== 0) {
                callCallback(new Error(`Minify worker stopped with exit code ${code}`));
            }
        });
    }
}

function terser(input = {}) {
    const { maxWorkers, ...options } = input;
    const workerPool = new WorkerPool({
        filePath: url.fileURLToPath((typeof document === 'undefined' ? new (require('u' + 'rl').URL)('file:' + __filename).href : (document.currentScript && document.currentScript.src || new URL('index.js', document.baseURI).href))),
        maxWorkers
    });
    return {
        name: 'terser',
        async renderChunk(code, chunk, outputOptions) {
            const defaultOptions = {
                sourceMap: outputOptions.sourcemap === true || typeof outputOptions.sourcemap === 'string'
            };
            if (outputOptions.format === 'es') {
                defaultOptions.module = true;
            }
            if (outputOptions.format === 'cjs') {
                defaultOptions.toplevel = true;
            }
            try {
                const { code: result, nameCache, sourceMap } = await workerPool.addAsync({
                    code,
                    options: smob.merge({}, options || {}, defaultOptions)
                });
                if (options.nameCache && nameCache) {
                    let vars = {
                        props: {}
                    };
                    if (smob.hasOwnProperty(options.nameCache, 'vars') && smob.isObject(options.nameCache.vars)) {
                        vars = smob.merge({}, options.nameCache.vars || {}, vars);
                    }
                    if (smob.hasOwnProperty(nameCache, 'vars') && smob.isObject(nameCache.vars)) {
                        vars = smob.merge({}, nameCache.vars, vars);
                    }
                    // eslint-disable-next-line no-param-reassign
                    options.nameCache.vars = vars;
                    let props = {};
                    if (smob.hasOwnProperty(options.nameCache, 'props') && smob.isObject(options.nameCache.props)) {
                        // eslint-disable-next-line prefer-destructuring
                        props = options.nameCache.props;
                    }
                    if (smob.hasOwnProperty(nameCache, 'props') && smob.isObject(nameCache.props)) {
                        props = smob.merge({}, nameCache.props, props);
                    }
                    // eslint-disable-next-line no-param-reassign
                    options.nameCache.props = props;
                }
                if ((!!defaultOptions.sourceMap || !!options.sourceMap) && smob.isObject(sourceMap)) {
                    return {
                        code: result,
                        map: sourceMap
                    };
                }
                return result;
            }
            catch (e) {
                return Promise.reject(e);
            }
        }
    };
}

runWorker();

exports.default = terser;
module.exports = Object.assign(exports.default, exports);
//# sourceMappingURL=index.js.map
