"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var eq_1 = __importDefault(require("./eq"));
var isObjectLike_1 = __importDefault(require("./isObjectLike"));
var isArray_1 = __importDefault(require("./isArray"));
var getTag_1 = __importDefault(require("./common/getTag"));
var isTypedArray_1 = __importDefault(require("./isTypedArray"));
var getAllKeys_1 = __importDefault(require("./common/getAllKeys"));
var toNumber_1 = __importDefault(require("./toNumber"));
var Stack_1 = require("./common/Stack");
var baseFindIndex_1 = require("./common/baseFindIndex");
var SetCache_1 = require("./common/SetCache");
var dataViewTag = '[object DataView]';
var arrayBufferTag = '[object ArrayBuffer]';
var mapTag = '[object Map]';
var setTag = '[object Set]';
var arrayTag = '[object Array]';
var argsTag = '[object Arguments]';
var COMPARE_PARTIAL_FLAG = 1;
var COMPARE_UNORDERED_FLAG = 2;
/**
 * 将value转为二进制字符串
 */
var getBinaryString = function (value) {
    var result = (0, toNumber_1.default)(value).toString(2);
    while (result.length < 64) {
        result = "0".concat(result);
    }
    return result;
};
/**
 * 模拟与运算
 */
var checkAnd = function (val1, val2) {
    if (val1 === undefined) {
        return 0;
    }
    var binaryVal1 = getBinaryString(val1);
    var binaryVal2 = getBinaryString(val2);
    var result = 0;
    for (var i = 0; i < 64; i++) {
        if (binaryVal1[i] === binaryVal2[i] && binaryVal1[i] === '1') {
            result = 1;
        }
    }
    return result;
};
/**
 * 模拟或运算
 */
var checkOr = function (val1, val2) {
    if (val1 === undefined) {
        return 1;
    }
    var binaryVal1 = getBinaryString(val1);
    var binaryVal2 = getBinaryString(val2);
    var result = 0;
    for (var i = 0; i < 64; i++) {
        if (binaryVal1[i] !== binaryVal2[i]) {
            result = 1;
        }
    }
    return result;
};
/**
 * 将Map转为数组，对结果排序后返回
 */
var switchMapToArray = function (map) {
    var index = 0;
    var result = Array(map.size);
    map.forEach(function (value, key) {
        result[index] = [key, value];
        index += 1;
    });
    return result.sort();
};
/**
 * 将集合转为数组，对结果排序后返回
 */
var switchSetToArray = function (set) {
    var index = 0;
    var result = Array(set.size);
    set.forEach(function (value) {
        result[index] = value;
        index += 1;
    });
    return result.sort();
};
/**
 * 检查数组元素，满足predicate条件时，返回true，否则返回false
 */
var arraySome = function (array, predicate) {
    for (var i = 0; i < (array === null ? 0 : array.length); i++) {
        if (predicate(array[i], i, array)) {
            return true;
        }
    }
    return false;
};
/**
 * 检查DataView
 */
var checkDateView = function (object, other) { return !((object.byteLength !== other.byteLength)
    || (object.byteOffset !== other.byteOffset)); };
/**
 * 检查ArrayBuffer
 */
var checkArrayBuffer = function (object, other, equalFunc) { return !((object.byteLength !== other.byteLength)
    || !equalFunc(new Uint8Array(object), new Uint8Array(other))); };
/**
 * 检查Map、Set
 */
var checkMapOrSet = function (param) {
    var bitmask = param.bitmask, tag = param.tag, object = param.object, other = param.other, stack = param.stack, customizer = param.customizer, equalFunc = param.equalFunc;
    var result = true;
    var bitmaskParam = bitmask;
    var isPartial = checkAnd(bitmask, COMPARE_PARTIAL_FLAG);
    var convert = tag === '[object Set]' ? switchSetToArray : switchMapToArray;
    if (object.size !== other.size && !isPartial) {
        return false;
    }
    var stacked = stack.get(object);
    if (stacked) {
        return stacked === other;
    }
    bitmaskParam = checkOr(bitmaskParam, COMPARE_UNORDERED_FLAG);
    stack.set(object, other);
    result = checkEqualByArray(convert(object), convert(other), bitmaskParam, customizer, equalFunc, stack);
    stack.delete(object);
    return result;
};
/**
 * 检查一些公共的对象，其Tag类型如下：'[object Boolean]', '[object Date]', '[object Number]', '[object RegExp]',
 * '[object String]', '[object Error]', '[object Symbol]'
 */
var checkForCommonTag = function (object, other, tag) {
    var commonTagList = ['[object Boolean]', '[object Date]', '[object Number]'];
    var stringTagList = ['[object RegExp]', '[object String]'];
    var errorTagList = ['[object Error]'];
    var symbolTagList = ['[object Symbol]'];
    if (commonTagList.includes(tag)) {
        return (0, eq_1.default)(+object, +other);
    }
    if (stringTagList.includes(tag)) {
        return "".concat(object) === "".concat(other);
    }
    if (errorTagList.includes(tag)) {
        return object.name === other.name && object.message === other.message;
    }
    if (symbolTagList.includes(tag) && Symbol.prototype.valueOf) {
        return Symbol.prototype.valueOf.call(object) === Symbol.prototype.valueOf.call(other);
    }
    return false;
};
/**
 * 根据Tag类型，判断两个对象是否相等
 */
function checkEqualByTag(objParam, otherParam, tag, bitmask, customizer, equalFunc, stack) {
    var object = objParam;
    var other = otherParam;
    var result;
    switch (tag) {
        case dataViewTag:
            result = checkDateView(object, other);
            if (result) {
                object = object.buffer;
                other = other.buffer;
            }
            break;
        case arrayBufferTag:
            result = checkArrayBuffer(object, other, equalFunc);
            break;
        case mapTag:
        case setTag:
            result = checkMapOrSet({
                bitmask: bitmask,
                tag: tag,
                object: object,
                other: other,
                stack: stack,
                customizer: customizer,
                equalFunc: equalFunc,
            });
            break;
        default:
            result = checkForCommonTag(object, other, tag);
            break;
    }
    return result;
}
/**
 * 按照数组判断是相等
 */
function checkEqualByArray(array, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = checkAnd(bitmask, COMPARE_PARTIAL_FLAG);
    var arrLength = array.length;
    var othLength = other.length;
    if (arrLength !== othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked === other && othStacked === array;
    }
    stack.set(array, other);
    stack.set(other, array);
    var index = 0;
    var result = true;
    var seen = checkAnd(bitmask, COMPARE_UNORDERED_FLAG) ? new SetCache_1.SetCache([]) : undefined;
    var _loop_1 = function () {
        var arrValue = array[index];
        var othValue = other[index];
        index += 1;
        if (seen) {
            var checkCacheHas = function (otherVal, othIndex) {
                if (!(0, baseFindIndex_1.cacheHas)(seen, othIndex)
                    && (arrValue === otherVal || equalFunc(arrValue, otherVal, bitmask, customizer, stack))) {
                    seen.push(othIndex);
                    return seen;
                }
                return undefined;
            };
            if (!arraySome(other, checkCacheHas)) {
                result = false;
                return "break";
            }
        }
        if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result = false;
            return "break";
        }
    };
    while (index < arrLength) {
        var state_1 = _loop_1();
        if (state_1 === "break")
            break;
    }
    stack.delete(array);
    stack.delete(other);
    return result;
}
/**
 * 按照对象判断是否相等
 */
function checkEqualObjects(object, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = checkAnd(bitmask, COMPARE_PARTIAL_FLAG);
    var objProps = (0, getAllKeys_1.default)(object);
    var othProps = (0, getAllKeys_1.default)(other);
    if (objProps.length !== othProps.length && !isPartial) {
        return false;
    }
    var index = objProps.length;
    while (index--) {
        var key_1 = objProps[index];
        if (!(isPartial ? key_1 in other : Object.prototype.hasOwnProperty.hasOwnProperty.call(other, key_1))) {
            return false;
        }
    }
    var objStacked = stack.get(object);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked === other && othStacked === object;
    }
    var result = true;
    stack.set(object, other);
    stack.set(other, object);
    var skipCtor = isPartial;
    var key;
    while (++index < objProps.length) {
        key = objProps[index];
        var objValue = object[key];
        var othValue = other[key];
        if (!(objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack))) {
            result = false;
            break;
        }
        skipCtor = skipCtor || (skipCtor = key === 'constructor');
    }
    if (result && !skipCtor) {
        var objCtor = object.constructor;
        var othCtor = other.constructor;
        if (objCtor !== othCtor && ('constructor' in object && 'constructor' in other)
            && !(typeof objCtor === 'function' && objCtor instanceof objCtor
                && typeof othCtor === 'function' && othCtor instanceof othCtor)) {
            result = false;
        }
    }
    stack.delete(object);
    stack.delete(other);
    return result;
}
var getObjectTag = function (objIsArray, object) {
    var resTag = objIsArray ? arrayTag : (0, getTag_1.default)(object);
    resTag = resTag === argsTag ? '[object Object]' : resTag;
    return resTag;
};
var checkByTagOrArray = function (param) {
    var objIsArr = param.objIsArr, object = param.object, other = param.other, objTag = param.objTag, bitmask = param.bitmask, customizer = param.customizer, equalFunc = param.equalFunc, stack = param.stack;
    var eqByArray = objIsArr || (0, isTypedArray_1.default)(object);
    return eqByArray ? checkEqualByArray(object, other, bitmask, customizer, equalFunc, stack)
        : checkEqualByTag(object, other, objTag, bitmask, customizer, equalFunc, stack);
};
/**
 * 对象的深度比较
 */
function isEqualDeepCore(object, other, otherParam) {
    if (object === void 0) { object = undefined; }
    if (other === void 0) { other = undefined; }
    var bitmask = otherParam.bitmask, customizer = otherParam.customizer, equalFunc = otherParam.equalFunc, _a = otherParam.stack, stack = _a === void 0 ? new Stack_1.Stack(undefined) : _a;
    var objIsArr = (0, isArray_1.default)(object);
    var objTag = getObjectTag(objIsArr, object);
    var othTag = getObjectTag((0, isArray_1.default)(other), other);
    // 当object和other的Tag类型一致，但不是Object对象类型时，进入Tag/Array的比较逻辑
    if (objTag === othTag && objTag !== '[object Object]') {
        return checkByTagOrArray({
            objIsArr: objIsArr,
            object: object,
            other: other,
            objTag: objTag,
            bitmask: bitmask,
            customizer: customizer,
            equalFunc: equalFunc,
            stack: stack,
        });
    }
    // 其他情况，进入对象的比较逻辑
    return checkEqualObjects(object, other, bitmask, customizer, equalFunc, stack);
}
/**
 * isEqual方法的主入口
 */
function isEqualCore(value, other, bitmask, customizer, stack) {
    if (value === void 0) { value = undefined; }
    if (other === void 0) { other = undefined; }
    var copyVal = value;
    var copyOth = other;
    if (copyVal === copyOth) {
        return true;
    }
    if (value === null || other === null || (!(0, isObjectLike_1.default)(value) && !(0, isObjectLike_1.default)(other))) {
        return copyVal !== value && copyOth !== other;
    }
    var restParam = {
        bitmask: bitmask,
        customizer: customizer,
        equalFunc: isEqualCore,
        stack: stack,
    };
    return isEqualDeepCore(copyVal, copyOth, restParam);
}
/**
 * 执行深比较来确定两者的值是否相等
 * 这个方法支持比较 arrays, array buffers, booleans, date objects, error objects, maps, numbers, Object objects, regexes,
 * sets, strings, symbols, 以及 typed arrays. Object 对象值比较自身的属性，不包括继承的和可枚举的属性。 不支持函数和DOM节点比较。
 * 如果两个值完全相同，那么返回 true，否则返回 false；当处理逻辑异常时，返回false
 * @param value
 * @param other
 */
function isEqual(value, other) {
    if (value === void 0) { value = undefined; }
    if (other === void 0) { other = undefined; }
    try {
        return isEqualCore(value, other);
    }
    catch (e) {
        return false;
    }
}
exports.default = isEqual;
