"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractLicenseManager = void 0;
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const utils_1 = require("../util/utils");
const crypto_1 = __importDefault(require("crypto"));
const LICENSE_DIR = 'licenses';
const HASH_SUFFIX = '.sha256';
class AbstractLicenseManager {
    findUnacceptedLicenses(licenseIds, progress) {
        return __awaiter(this, void 0, void 0, function* () {
            const licenses = yield this.retrieveLicenses(licenseIds, progress);
            if (licenses.size === 0) {
                return Promise.resolve(undefined);
            }
            this.checkLicense(licenses, licenseIds, progress);
            const unacceptedLicenses = [];
            for (const license of licenses.values()) {
                if (!this.hasAccepted(license, progress)) {
                    unacceptedLicenses.push(license);
                }
            }
            return unacceptedLicenses;
        });
    }
    checkLicense(licenses, licenseIds, progress) {
        for (const licenseId of licenseIds) {
            if (!licenses.get(licenseId)) {
                progress.warn(`Unable to obtain the SDK license [${licenseId}]`);
            }
        }
    }
    hasAccepted(license, progress) {
        const localHash = this.readLocalHash(license.id, progress);
        if (localHash === undefined) {
            return false;
        }
        return localHash === this.hashContent(license.content);
    }
    hashContent(content) {
        return crypto_1.default.createHash('sha256').update(content).digest('hex');
    }
    readLocalHash(licenseId, progress) {
        const licenseHashPath = path_1.default.resolve(this.getLicensesLocation(), licenseId + HASH_SUFFIX);
        if (!fs_1.default.existsSync(licenseHashPath)) {
            return undefined;
        }
        try {
            const licenseHash = fs_1.default.readFileSync(licenseHashPath, { encoding: 'utf-8' });
            if ((0, utils_1.isEmpty)(licenseHash)) {
                progress.warn(this.getReadLocalHashErrorMessage(this.getLicensesLocation()));
                return undefined;
            }
            return licenseHash;
        }
        catch (e) {
            progress.warn(this.getReadLocalHashErrorMessage(this.getLicensesLocation()));
            return undefined;
        }
    }
    getLicensesLocation() {
        return path_1.default.resolve(this.getLicensesRootLocation(), LICENSE_DIR);
    }
}
exports.AbstractLicenseManager = AbstractLicenseManager;
