"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.instrument = void 0;
const fs = require("fs");
const path = require("path");
const log4js = require("log4js");
const os = require("os");
const filterKey = 'use concurrent';
const filterIstanbulKey = '/* instrument ignore next */';
const logger = log4js.getLogger();
const ohosTest = 'ohosTest';
const srcOhosTest = `src${path.sep}ohosTest`;
const entryTest = '.test';
const unitTest = 'unitTest';
const ohosTestSourcePath = '.test/default/intermediates/ohosTest/source';
const unitTestSourcePath = '.test/default/intermediates/test/source';
const ohosTestInstrumentPath = '.test/default/intermediates/ohosTest/instrument';
const unitTestInstrumentPath = '.test/default/intermediates/test/instrument';
const ohosTestInitPath = '.test/default/intermediates/ohosTest/init_coverage.json';
const unitTestInitPath = '.test/default/intermediates/test/init_coverage.json';
const ohosTestSourceMapPath = '.test/default/intermediates/ohosTest/coverage_sourcemap';
const unitTestSourceMapPath = '.test/default/intermediates/test/coverage_sourcemap';
const ohosModule = 'oh_modules';
const test = `src${path.sep}test`;
const etsSuffix = '.ets';
const tsSuffix = '.ts';
const jsSuffix = '.js';
const index_1 = require("./instrument/index");
const instrumenter = (0, index_1.createInstrumenter)({
    compact: false,
    esModules: true
});
/**
 * 功能：代码插桩
 * @param {string} projectPath 插桩文件路径
 * @param {string} modulePath 模块路径
 * @param {string} source 源码
 * @param {sourceMap} sourceMap 转换生成js文件对应的sourceMap文件
 * @param {testFrameworkParam} 跑测试用例涉及的参数对象
 * @returns {string}
 */
const instrument = (projectPath, modulePath, source, sourceMap, testFrameworkParam) => {
    try {
        // map文件中保存的文件路径
        const mapFilePath = sourceMap.sources[0];
        const mapFileDirPath = path.dirname(mapFilePath);
        let coverageType = testFrameworkParam.testMode;
        if (modulePath.indexOf(srcOhosTest) !== -1) {
            throw new Error('enter instrument ohosTest.');
        }
        if (mapFileDirPath.includes(ohosModule)) {
            return source;
        }
        let sourceMapPath = coverageType == ohosTest
            ? path.join(modulePath, ohosTestSourceMapPath, 'ohosTestSourceMap.map')
            : path.join(modulePath, unitTestSourceMapPath, 'unitTestSourceMap.map');
        if (!fs.existsSync(sourceMapPath)) {
            fs.mkdirSync(path.dirname(sourceMapPath), { recursive: true });
        }
        let sourcePath = '';
        let instrumentPath = '';
        if (coverageType.includes(ohosTest)) {
            sourcePath = path.join(modulePath, ohosTestSourcePath);
            instrumentPath = path.join(modulePath, ohosTestInstrumentPath);
        }
        else if (coverageType.includes(unitTest)) {
            sourcePath = path.join(modulePath, unitTestSourcePath);
            instrumentPath = path.join(modulePath, unitTestInstrumentPath);
        }
        if (mapFileDirPath.includes(srcOhosTest) || mapFileDirPath.includes(test) || mapFileDirPath.includes(entryTest)) {
            fs.appendFileSync(sourceMapPath, '\"' + mapFilePath.replace(/\\/g, '\\\\') + '\":' + JSON.stringify(sourceMap) + ',');
            return source;
        }
        let paramMap = {
            'projectPath': projectPath,
            'modulePath': modulePath,
            'sourcePath': sourcePath,
            'instrumentPath': instrumentPath,
            'mapFileDirPath': mapFileDirPath,
            'mapFilePath': mapFilePath,
            'sourceMapPath': sourceMapPath,
        };
        source = instrumentPart2(source, sourceMap, coverageType, paramMap);
    }
    catch (err) {
        logger.error('instrument failed, error:' + err);
    }
    return source;
};
exports.instrument = instrument;
function instrumentPart2(source, sourceMap, coverageType, paramMap) {
    // 拼接.test文件夹下插桩文件和源文件的文件路径,路径和模块下的相对文件路径保持一致
    let projectPath = paramMap['projectPath'].concat(path.sep);
    let fileRelPath = paramMap['mapFileDirPath'].replace(projectPath, '');
    const sourceFilePath = paramMap['mapFilePath']
        .replace(paramMap['mapFileDirPath'], path.resolve(paramMap['sourcePath'], fileRelPath))
        .replace(etsSuffix, jsSuffix)
        .replace(tsSuffix, jsSuffix);
    const instrumentFilePath = paramMap['mapFilePath']
        .replace(paramMap['mapFileDirPath'], path.resolve(paramMap['instrumentPath'], fileRelPath))
        .replace(etsSuffix, jsSuffix)
        .replace(tsSuffix, jsSuffix);
    // 跳过插桩@concurrent注解标识的方法
    if (source.search(filterKey) != -1) {
        source = instrumentFilter(source);
    }
    if (path.dirname(sourceFilePath) != path.dirname(instrumentFilePath)) {
        const [, sourcePath] = paramMap['mapFilePath'].split(projectPath);
        sourceMap.sources[0] = sourcePath;
        // 在模块.test下创建临时文件夹，保存插件文件和源文件
        fsTransform(sourceFilePath, instrumentFilePath, paramMap['sourceMapPath'], source, sourceMap);
        // 将插桩后的文件内容更新到内存中
        source = fs.readFileSync(instrumentFilePath, 'utf-8');
        if (source.indexOf('// instrument ignore file') === -1 && source.indexOf('/* instrument ignore file */') === -1) {
            initCoverageData(source, paramMap['modulePath'], coverageType);
        }
        fs.unlinkSync(instrumentFilePath);
        fs.writeFileSync(instrumentFilePath, source);
    }
    return source;
}
/**
 * 功能：在模块.test下创建临时文件夹，保存插件文件和源文件
 * @param sourceFilePath
 * @param instrumentFilePath
 * @param sourceMapPath
 * @param source
 * @param sourceMap
 */
function fsTransform(sourceFilePath, instrumentFilePath, sourceMapPath, source, sourceMap) {
    // 在模块.test下创建临时文件夹，保存插件文件和源文件
    fs.mkdirSync(path.dirname(sourceFilePath), { recursive: true });
    fs.mkdirSync(path.dirname(instrumentFilePath), { recursive: true });
    fs.writeFileSync(sourceFilePath, source);
    const instrumentSource = instrumenter.instrumentSync(source, sourceFilePath, sourceMap);
    let lastSourceMap = instrumenter.lastSourceMap();
    fs.appendFileSync(sourceMapPath, '\"' + sourceFilePath.replace(/\\/g, '\\\\') + '\":' + JSON.stringify(lastSourceMap) + ',');
    fs.writeFileSync(instrumentFilePath, instrumentSource);
    fs.writeFileSync(`${sourceFilePath}.map`, JSON.stringify(sourceMap));
}
/**
 * 功能：对源文件进行过滤，如果方法中包含use concurrent则将在此方法加上nyc注释跳过插桩
 * @param {string} source js文件内容
 * @returns {string}
 */
function instrumentFilter(source) {
    let lineData = source.split(os.EOL);
    let length = lineData.length;
    let filterSource = '';
    for (let i = 0; i < length; i++) {
        if (lineData[i].includes(filterKey) && i - 1 >= 0) {
            let previousOneLine = lineData[i - 1];
            lineData[i - 1] = filterIstanbulKey + previousOneLine;
        }
    }
    for (let j = 0; j < lineData.length; j++) {
        filterSource = filterSource.concat(lineData[j]).concat(os.EOL);
    }
    return filterSource;
}
/**
 * 初始化覆盖率数据
 *
 * @param source 插桩后的源码
 * @param modulePath 模块路径
 * @param coverageType ohosTest/unitTest
 */
function initCoverageData(source, modulePath, coverageType) {
    const coverageDataIdentifies = 'coverageData = ';
    // 获取插装源码文件中的覆盖率数据在文件中的起止位置
    const index = source.indexOf(coverageDataIdentifies);
    const coverageDataStartIndex = index + coverageDataIdentifies.length;
    const coverageDataEndIndex = source.indexOf('var coverage =');
    let tempSource = source.substring(coverageDataStartIndex, coverageDataEndIndex);
    let tempIndex = tempSource.indexOf('\n  };');
    let sourceResult = tempSource.substring(0, tempIndex).concat(os.EOL).concat('  }');
    //匹配json中未带双引号的key
    sourceResult = sourceResult.replace(/(\w+)\s*:\s/g, '\"$1\": ');
    sourceResult = sourceResult.replace(/"sourcesContent":.*/g, '\"sourcesContent\":[],');
    const sourceJson = JSON.parse(sourceResult);
    const sourcePath = sourceJson.path;
    const sourceKey = sourcePath.replace(/\\/g, '\\\\');
    let initCoveragePath = '';
    if (coverageType.includes(ohosTest)) {
        initCoveragePath = path.join(modulePath, ohosTestInitPath);
    }
    else if (coverageType.includes(unitTest)) {
        initCoveragePath = path.join(modulePath, unitTestInitPath);
    }
    let formatSource = '';
    // 拼接成可merge的json格式
    mergeJson(initCoveragePath, formatSource, sourceKey, sourceResult);
}
function mergeJson(initCoveragePath, formatSource, sourceKey, sourceResult) {
    if (fs.existsSync(initCoveragePath)) {
        formatSource = `,${os.EOL}"${sourceKey}":${sourceResult}`;
    }
    else {
        formatSource = `"${sourceKey}":${sourceResult}`;
    }
    fs.appendFileSync(initCoveragePath, formatSource);
}
