"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.programVisitor = void 0;
const crypto_1 = require("crypto");
const core_1 = require("@babel/core");
const schema_1 = require("@istanbuljs/schema");
const log4js = require("log4js");
const sourceCoverage_1 = require("./sourceCoverage");
const constants_1 = require("./constants");
const logger = log4js.getLogger();
// pattern for instrument to ignore a section
const COMMENT_BRANCH_RE = /^\s*instrument\s+ignore\s+(if|else|next)(?=\W|$)/;
// pattern for instrument to ignore the whole file
const COMMENT_FILE_RE = /^\s*instrument\s+ignore\s+(file)(?=\W|$)/;
// source map URL pattern
const SOURCEMAP_RE = /[#@]\s*sourceMappingURL=(.*)\s*$/m;
// generate a variable name from hashing the supplied file path
function generateVarName(filename) {
    const hash = (0, crypto_1.createHash)(constants_1.Encryption.method);
    hash.update(filename);
    return 'cov_' + parseInt(hash.digest('hex').substr(0, 12), 16).toString(36);
}
// VisitorState holds the state of the visitor, provides helper functions
// and is the `this` for the individual coverage visitors.
// 记录每一个独立的覆盖率“访问者”的状态
class VisitorState {
    constructor(types, sourceFilePath, inputSourceMap, ignoreClassMethods = [], reportLogic = false) {
        this.varName = generateVarName(sourceFilePath);
        this.nextIgnore = null;
        this.srcCov = new sourceCoverage_1.SrcCoverage(sourceFilePath);
        if (typeof inputSourceMap !== 'undefined') {
            this.srcCov.getInputSource(inputSourceMap);
        }
        this.ignoreClassMethods = ignoreClassMethods;
        this.types = types;
        this.sourceMappingURL = null;
        this.reportLogic = reportLogic;
    }
    // 提取ignore注释提示(next|if|else)或null
    getHint(node) {
        let hint = null;
        if (node.leadingComments) {
            node.leadingComments.forEach((c) => {
                const val = (c.value ? c.value : '').trim();
                const groups = val.match(COMMENT_BRANCH_RE);
                if (groups) {
                    hint = groups[1];
                }
            });
        }
        return hint;
    }
    // should we ignore the node? Yes, if specifically ignoring
    // or if the node is generated.
    shouldIgnore(path) {
        return this.nextIgnore || !path.node.loc;
    }
    // 获取注释中的sourceMappingURL
    getAssignSourceMapURL(node) {
        const getURL = (comments) => {
            if (!comments) {
                return;
            }
            comments.forEach((c) => {
                const val = (c.value ? c.value : '').trim();
                const groups = val.match(SOURCEMAP_RE);
                if (groups) {
                    this.sourceMappingURL = groups[1];
                }
            });
        };
        getURL(node.leadingComments);
        getURL(node.trailingComments);
    }
    // for these expressions the statement counter needs to be hoisted, so
    // function name inference can be preserved
    counterNeedHoisting(path) {
        return (path.isFunctionExpression() ||
            path.isArrowFunctionExpression() ||
            path.isClassExpression());
    }
    // all the generic stuff that needs to be done on enter for every node
    onEnter(path) {
        const n = path.node;
        this.getAssignSourceMapURL(n);
        // if already ignoring, nothing more to do
        if (this.nextIgnore !== null) {
            return;
        }
        // check hint to see if ignore should be turned on
        const hint = this.getHint(n);
        if (hint === 'next') {
            this.nextIgnore = n;
            return;
        }
        // else check custom node attribute set by a prior visitor
        if (this.getProp(path.node, 'skip-all') !== null) {
            this.nextIgnore = n;
        }
        // else check for ignored class methods
        if (path.isFunctionExpression() &&
            this.ignoreClassMethods.some((name) => path.node.id && name === path.node.id.name)) {
            this.nextIgnore = n;
            return;
        }
        if (path.isClassMethod() &&
            this.ignoreClassMethods.some((name) => name === path.node.key.name)) {
            this.nextIgnore = n;
            return;
        }
    }
    // all the generic stuff on exit of a node,
    // including reseting ignores and custom node attrs
    onExit(path) {
        // restore ignore status, if needed
        if (path.node === this.nextIgnore) {
            this.nextIgnore = null;
        }
        // nuke all attributes for the node
        delete path.node.__cov__;
    }
    // 为所提供的节点设置节点属性
    setProp(node, name, val) {
        node.__cov__ = node.__cov__ || {};
        node.__cov__[name] = val;
    }
    // retrieve a node attribute for the supplied node or null
    getProp(node, name) {
        const cov = node.__cov__;
        if (!cov) {
            return null;
        }
        return cov[name];
    }
    // 修改原AST，生成cov_xxx()[]++的AST
    increase(type, id, index) {
        const T = this.types;
        // 如果存在`index`，将`x`转换为`x[index]'。
        const wrap = index !== null ? (x) => T.memberExpression(x, T.numericLiteral(index), true) : (x) => x;
        return T.updateExpression('++', wrap(T.memberExpression(T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(type)), T.numericLiteral(id), true)));
    }
    validateTrueNonTrivial(T, tempName) {
        return T.logicalExpression('&&', T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)), T.logicalExpression('&&', this.validateOrNonExpression(T, tempName), this.validateNoNonExpression(T, tempName)));
    }
    validateNoNonExpression(T, tempName) {
        return T.parenthesizedExpression(T.logicalExpression('||', T.binaryExpression('!==', T.callExpression(T.memberExpression(T.identifier('Object'), T.identifier('getPrototypeOf')), [
            T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)),
        ]), T.memberExpression(T.identifier('Object'), T.identifier('prototype'))), T.memberExpression(T.callExpression(T.memberExpression(T.identifier('Object'), T.identifier('values')), [
            T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)),
        ]), T.identifier('length'))));
    }
    validateOrNonExpression(T, tempName) {
        return T.parenthesizedExpression(T.logicalExpression('||', T.unaryExpression('!', T.callExpression(T.memberExpression(T.identifier('Array'), T.identifier('isArray')), [
            T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)),
        ])), T.memberExpression(T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)), T.identifier('length'))));
    }
    // Reads the logic expression conditions and conditionally increments truthy counter.
    increaseTrue(type, id, index, node) {
        const T = this.types;
        const tempName = `${this.varName}_temp`;
        return T.sequenceExpression([
            T.assignmentExpression('=', T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)), node // Only evaluates once.
            ),
            T.parenthesizedExpression(T.conditionalExpression(this.validateTrueNonTrivial(T, tempName), this.increase(type, id, index), T.nullLiteral())),
            T.memberExpression(T.callExpression(T.identifier(this.varName), []), T.identifier(tempName)),
        ]);
    }
    insertCounter(path, increment) {
        const T = this.types;
        if (path.isBlockStatement()) {
            path.node.body.unshift(T.expressionStatement(increment));
        }
        else if (path.isStatement()) {
            path.insertBefore(T.expressionStatement(increment));
        }
        else if (this.counterNeedHoisting(path) &&
            T.isVariableDeclarator(path.parentPath)) {
            // make an attempt to hoist the statement counter, so that
            // function names are maintained.
            const parent = path.parentPath.parentPath;
            if (parent && T.isExportNamedDeclaration(parent.parentPath)) {
                parent.parentPath.insertBefore(T.expressionStatement(increment));
            }
            else if (parent &&
                (T.isProgram(parent.parentPath) ||
                    T.isBlockStatement(parent.parentPath))) {
                parent.insertBefore(T.expressionStatement(increment));
            }
            else {
                path.replaceWith(T.sequenceExpression([increment, path.node]));
            }
        } /* instrument ignore else: not expected */
        else if (path.isExpression()) {
            path.replaceWith(T.sequenceExpression([increment, path.node]));
        }
        else {
            logger.error('Unable to insert counter for node type:', path.node.type);
        }
    }
    insertStatementCounter(path) {
        // instrument ignore if: paranoid check
        if (!(path.node && path.node.loc)) {
            return;
        }
        const index = this.srcCov.newStatement(path.node.loc);
        const increment = this.increase('s', index, null);
        this.insertCounter(path, increment);
    }
    insertFunctionCounter(path) {
        const T = this.types;
        // instrument ignore if: paranoid check
        if (!(path.node && path.node.loc)) {
            return;
        }
        const n = path.node;
        let floc = null;
        // get location for declaration
        switch (n.type) {
            case 'FunctionDeclaration':
            case 'FunctionExpression':
                // instrument ignore else: paranoid check
                if (n.id) {
                    floc = n.id.loc;
                }
                break;
        }
        if (!floc) {
            floc = {
                start: n.loc.start,
                end: { line: n.loc.start.line, column: n.loc.start.column + 1 },
            };
        }
        const name = path.node.id ? path.node.id.name : path.node.name;
        const index = this.srcCov.newFunction(name, floc, path.node.body.loc);
        const increment = this.increase('f', index, null);
        const body = path.get('body');
        // instrument ignore else: not expected
        if (body.isBlockStatement()) {
            body.node.body.unshift(T.expressionStatement(increment));
        }
        else {
            logger.error('Unable to process function body node type:', path.node.type);
        }
    }
    getBranchIncrement(branchName, loc) {
        const index = this.srcCov.addBranchPath(branchName, loc);
        return this.increase('b', branchName, index);
    }
    getBranchLogicIncrement(path, branchName, loc) {
        const index = this.srcCov.addBranchPath(branchName, loc);
        return [
            this.increase('b', branchName, index),
            this.increaseTrue('bT', branchName, index, path.node),
        ];
    }
    insertBranchCounter(path, branchName, loc) {
        const increment = this.getBranchIncrement(branchName, loc || path.node.loc);
        this.insertCounter(path, increment);
    }
    findNodeLeaves(node, accumulators, parent, property) {
        if (!node) {
            return;
        }
        if (node.type === 'LogicalExpression') {
            const hint = this.getHint(node);
            if (hint !== 'next') {
                this.findNodeLeaves(node.left, accumulators, node, 'left');
                this.findNodeLeaves(node.right, accumulators, node, 'right');
            }
        }
        else {
            accumulators.push({
                node,
                parent,
                property,
            });
        }
    }
}
/**
 * 接受一组访问者方法,返回具有`enter`和`exit`属性的访问者对象
 * @param enter
 * @returns
 */
function entries(...enter) {
    // the enter function
    const wrappedEntry = function (path, node) {
        this.onEnter(path);
        // 只有当忽略不生效时，才会调用提供的访问者
        if (this.shouldIgnore(path)) {
            return;
        }
        enter.forEach((e) => {
            e.call(this, path, node);
        });
    };
    const exit = function (path, node) {
        this.onExit(path, node);
    };
    return {
        enter: wrappedEntry,
        exit,
    };
}
function coverageStatement(path) {
    this.insertStatementCounter(path);
}
/* instrument ignore next: no node.js support */
function coverAssignmentPattern(path) {
    const n = path.node;
    const b = this.srcCov.newBranch('default-arg', n.loc);
    this.insertBranchCounter(path.get('right'), b);
}
function coverageFunction(path) {
    this.insertFunctionCounter(path);
}
function coverVariableDeclarator(path) {
    this.insertStatementCounter(path.get('init'));
}
function coverClassPropDeclarator(path) {
    this.insertStatementCounter(path.get('value'));
}
function madeBlock(propPath) {
    const type = this.types;
    if (!propPath.node) {
        propPath.replaceWith(type.blockStatement([]));
    }
    if (!propPath.isBlockStatement()) {
        propPath.replaceWith(type.blockStatement([propPath.node]));
        propPath.node.loc = propPath.node.body[0].loc;
        propPath.node.body[0].leadingComments = propPath.node.leadingComments;
        propPath.node.leadingComments = undefined;
    }
}
function blockProp(prop) {
    return function (path) {
        madeBlock.call(this, path.get(prop));
    };
}
function madeBracketedExpressionForNonIdentifier(path) {
    const T = this.types;
    if (path.node && !path.isIdentifier()) {
        path.replaceWith(T.parenthesizedExpression(path.node));
    }
}
function bracketedExpressionProp(prop) {
    return function (path) {
        madeBracketedExpressionForNonIdentifier.call(this, path.get(prop));
    };
}
function convertArrowExpression(path) {
    const node = path.node;
    const type = this.types;
    if (!type.isBlockStatement(node.body)) {
        const bloc = node.body.loc;
        if (node.expression === true) {
            node.expression = false;
        }
        node.body = type.blockStatement([type.returnStatement(node.body)]);
        // restore body location
        node.body.loc = bloc;
        // set up the location for the return statement so it gets
        // instrumented
        node.body.body[0].loc = bloc;
    }
}
function coverIfBranches(path) {
    const n = path.node;
    const hint = this.getHint(n);
    const ignoreIf = hint === 'if';
    const ignoreElse = hint === 'else';
    const branch = this.srcCov.newBranch('if', n.loc);
    if (ignoreIf) {
        this.setProp(n.consequent, 'skip-all', true);
    }
    else {
        this.insertBranchCounter(path.get('consequent'), branch, n.loc);
    }
    if (ignoreElse) {
        this.setProp(n.alternate, 'skip-all', true);
    }
    else {
        this.insertBranchCounter(path.get('alternate'), branch, n.loc);
    }
}
function createSwitchBranch(path) {
    const b = this.srcCov.newBranch('switch', path.node.loc);
    this.setProp(path.node, 'branchName', b);
}
function coverSwitchCase(path) {
    const T = this.types;
    const b = this.getProp(path.parentPath.node, 'branchName');
    // instrument ignore if: paranoid check 
    if (b === null) {
        throw new Error('Unable to get switch branch name');
    }
    const increment = this.getBranchIncrement(b, path.node.loc);
    path.node.consequent.unshift(T.expressionStatement(increment));
}
function coverTernary(path) {
    const n = path.node;
    const branch = this.srcCov.newBranch('cond-expr', path.node.loc);
    const cHint = this.getHint(n.consequent);
    const aHint = this.getHint(n.alternate);
    if (cHint !== 'next') {
        this.insertBranchCounter(path.get('consequent'), branch);
    }
    if (aHint !== 'next') {
        this.insertBranchCounter(path.get('alternate'), branch);
    }
}
function coverLogicalExpression(path) {
    const type = this.types;
    if (path.parentPath.node.type === 'LogicalExpression') {
        return; // already processed
    }
    const leaves = [];
    this.findNodeLeaves(path.node, leaves);
    const b = this.srcCov.newBranch('binary-expr', path.node.loc, this.reportLogic);
    for (let i = 0; i < leaves.length; i += 1) {
        const leaf = leaves[i];
        const hint = this.getHint(leaf.node);
        if (hint === 'next') {
            continue;
        }
        if (this.reportLogic) {
            const increment = this.getBranchLogicIncrement(leaf, b, leaf.node.loc);
            if (!increment[0]) {
                continue;
            }
            leaf.parent[leaf.property] = type.sequenceExpression([
                increment[0],
                increment[1],
            ]);
            continue;
        }
        const increment = this.getBranchIncrement(b, leaf.node.loc);
        if (!increment) {
            continue;
        }
        leaf.parent[leaf.property] = type.sequenceExpression([
            increment,
            leaf.node,
        ]);
    }
}
const codeVisitor = {
    ArrowFunctionExpression: entries(convertArrowExpression, coverageFunction),
    AssignmentPattern: entries(coverAssignmentPattern),
    BlockStatement: entries(),
    ExportDefaultDeclaration: entries(),
    ExportNamedDeclaration: entries(),
    ClassMethod: entries(coverageFunction),
    ExpressionStatement: entries(coverageStatement),
    BreakStatement: entries(coverageStatement),
    ContinueStatement: entries(coverageStatement),
    TryStatement: entries(coverageStatement),
    VariableDeclaration: entries(),
    VariableDeclarator: entries(coverVariableDeclarator),
    ClassDeclaration: entries(bracketedExpressionProp('superClass')),
    ClassProperty: entries(coverClassPropDeclarator),
    ClassPrivateProperty: entries(coverClassPropDeclarator),
    ObjectMethod: entries(coverageFunction),
    DebuggerStatement: entries(coverageStatement),
    ReturnStatement: entries(coverageStatement),
    ThrowStatement: entries(coverageStatement),
    IfStatement: entries(blockProp('consequent'), blockProp('alternate'), coverageStatement, coverIfBranches),
    ForStatement: entries(blockProp('body'), coverageStatement),
    ForInStatement: entries(blockProp('body'), coverageStatement),
    ForOfStatement: entries(blockProp('body'), coverageStatement),
    WhileStatement: entries(blockProp('body'), coverageStatement),
    DoWhileStatement: entries(blockProp('body'), coverageStatement),
    SwitchStatement: entries(createSwitchBranch, coverageStatement),
    SwitchCase: entries(coverSwitchCase),
    WithStatement: entries(blockProp('body'), coverageStatement),
    FunctionDeclaration: entries(coverageFunction),
    FunctionExpression: entries(coverageFunction),
    LabeledStatement: entries(coverageStatement),
    ConditionalExpression: entries(coverTernary),
    LogicalExpression: entries(coverLogicalExpression),
};
const globalTemplateAlteredFunction = (0, core_1.template)(`
        var Function = (function(){}).constructor;
        var global = (new Function(GLOBAL_COVERAGE_SCOPE))();
`);
const globalTemplateFunction = (0, core_1.template)(`
        var global = (new Function(GLOBAL_COVERAGE_SCOPE))();
`);
const globalTemplateVariable = (0, core_1.template)(`
        var global = GLOBAL_COVERAGE_SCOPE;
`);
const templateStr = `
function COVERAGE_FUNCTION () {
    var nPath = PATH;
    var hash = HASH;
    GLOBAL_COVERAGE_TEMPLATE
    var gcv = GLOBAL_COVERAGE_VAR;
    var coverageData = INITIAL;
    var coverage = global[gcv] || (global[gcv] = {});
    if (!coverage[nPath] || coverage[nPath].hash !== hash) {
        coverage[nPath] = coverageData;
    }

    var actualCoverage = coverage[nPath];
    {
        // @ts-ignore
        COVERAGE_FUNCTION = function () {
            global[gcv][nPath] = actualCoverage
            return actualCoverage;
        }
    }

    return actualCoverage;
}
`;
// the template to insert at the top of the program.
const coverageTemplate = (0, core_1.template)(templateStr, { preserveComments: true });
// the rewire plugin (and potentially other babel middleware)
// may cause files to be instrumented twice, see:
// https://github.com/istanbuljs/babel-plugin-istanbul/issues/94
// we should only instrument code for coverage the first time
// it's run through istanbul-lib-instrument.
function alreadyInstrumented(path, visitState) {
    return path.scope.hasBinding(visitState.varName);
}
function shouldIgnoreFile(programNode) {
    return (programNode.parent &&
        programNode.parent.comments.some((c) => COMMENT_FILE_RE.test(c.value)));
}
const getOptsAndState = (opt, types, sourceFilePath) => {
    const opts = {
        ...schema_1.defaults.instrumentVisitor,
        ...opt,
    };
    const visitorState = new VisitorState(types, sourceFilePath, opts.inputSourceMap, opts.ignoreClassMethods, opts.reportLogic);
    return {
        opts,
        visitorState,
    };
};
function getTemplate(opts, path, T) {
    if (opts.coverageGlobalScopeFunc) {
        if (path.scope.getBinding('Function')) {
            return globalTemplateAlteredFunction({
                GLOBAL_COVERAGE_SCOPE: T.stringLiteral('return ' + opts.coverageGlobalScope),
            });
        }
        else {
            return globalTemplateFunction({
                GLOBAL_COVERAGE_SCOPE: T.stringLiteral('return ' + opts.coverageGlobalScope),
            });
        }
    }
    else {
        return globalTemplateVariable({
            GLOBAL_COVERAGE_SCOPE: opts.coverageGlobalScope,
        });
    }
}
function addTemplate(path, T, visitorState, countFunction) {
    path.node.body.unshift(T.expressionStatement(T.callExpression(T.identifier(visitorState.varName), [])));
    path.node.body.unshift(countFunction);
}
/**
 * programVisitor 是用于instrumentation的“babel”适配器.
 * 它返回一个含有`enter`和`exit`两个方法的对象。
 * 在babel visitor中，这些函数应分配给“程序”进入和退出函数，或从“程序”进入和退出函数调用
 * 这些函数不对Babel设置的状态进行假设，因此可以在Babel插件以外的上下文中使用。
 *
 * exit函数返回一个当前具有以下键的对象：
 * `fileCoverage` -为源文件创建的文件覆盖对象。
 * `sourceMappingURL` -处理文件时找到的任何source mapping URL 。
 *
 * @param {Object} types - an instance of babel-types.
 * @param {string} sourceFilePath - the path to source file.
 * @param {Object} opts - additional options.
 * @param {string} [opts.coverageVariable=__coverage__] the global coverage variable name.
 * @param {boolean} [opts.reportLogic=false] report boolean value of logical expressions.
 * @param {string} [opts.coverageGlobalScope=this] the global coverage variable scope.
 * @param {boolean} [opts.coverageGlobalScopeFunc=true] use an evaluated function to find coverageGlobalScope.
 * @param {Array} [opts.ignoreClassMethods=[]] names of methods to ignore by default on classes.
 * @param {object} [opts.inputSourceMap=undefined] the input source map, that maps the uninstrumented code back to the
 * original code.
 */
const programVisitor = (types, sourceFilePath = 'unknown.js', opts = {}) => {
    const T = types;
    const res = getOptsAndState(opts, types, sourceFilePath);
    opts = res.opts;
    const visitorState = res.visitorState;
    function exit(path) {
        if (alreadyInstrumented(path, visitorState)) {
            return;
        }
        visitorState.srcCov.deleteBranch();
        const coverageData = visitorState.srcCov.toJSON();
        if (shouldIgnoreFile(path.find((p) => p.isProgram()))) {
            return {
                fileCoverage: coverageData,
                sourceMappingURL: visitorState.sourceMappingURL,
            };
        }
        coverageData[constants_1.Encryption.key] = constants_1.Encryption.val;
        const { hashStr, coverageNode } = getHashAndNode(coverageData, T);
        let gvTemplate = getTemplate(opts, path, T);
        const countFunction = coverageTemplate({
            GLOBAL_COVERAGE_VAR: T.stringLiteral(opts.coverageVariable),
            GLOBAL_COVERAGE_TEMPLATE: gvTemplate,
            COVERAGE_FUNCTION: T.identifier(visitorState.varName),
            PATH: T.stringLiteral(sourceFilePath),
            INITIAL: coverageNode,
            HASH: T.stringLiteral(hashStr),
        });
        // 显式调用this.varName以确保覆盖始终初始化
        addTemplate(path, T, visitorState, countFunction);
        return {
            fileCoverage: coverageData,
            sourceMappingURL: visitorState.sourceMappingURL,
        };
    }
    return {
        enter(path) {
            if (shouldIgnoreFile(path.find((p) => p.isProgram()))) {
                return;
            }
            if (alreadyInstrumented(path, visitorState)) {
                return;
            }
            // 遍历当前节点的子节点
            path.traverse(codeVisitor, visitorState);
        },
        exit(path) {
            return exit(path);
        },
    };
};
exports.programVisitor = programVisitor;
function getHashAndNode(coverageData, T) {
    const hashStr = (0, crypto_1.createHash)(constants_1.Encryption.method)
        .update(JSON.stringify(coverageData))
        .digest('hex');
    coverageData.hash = hashStr;
    if (coverageData.inputSourceMap &&
        Object.getPrototypeOf(coverageData.inputSourceMap) !== Object.prototype) {
        coverageData.inputSourceMap = {
            ...coverageData.inputSourceMap,
        };
    }
    const coverageNode = T.valueToNode(coverageData);
    delete coverageData[constants_1.Encryption.key];
    delete coverageData.hash;
    return { hashStr, coverageNode };
}
