"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArkUIXLocalComponentLoader = void 0;
const abstract_local_component_loader_1 = require("../../core/loader/abstract-local-component-loader");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const oh_component_dto_1 = require("../../ohos/loader/oh-component-dto");
const component_contants_1 = require("../../core/constants/component-contants");
const abstract_component_dto_1 = require("../../core/loader/abstract-component-dto");
const utils_1 = require("../../core/util/utils");
const arkuix_component_constants_1 = require("../constants/arkuix-component-constants");
class ArkUIXLocalComponentLoader extends abstract_local_component_loader_1.AbstractLocalComponentLoader {
    constructor(sdkRoot, progress) {
        super(ArkUIXLocalComponentLoader._MAX_SCAN_DEPTH, sdkRoot, progress);
        this._inCompatibleComponents = new Array();
    }
    /**
     * parseSdks
     * @param potentialUniPackages 扫描出来的元数据文件
     * @protected
     * @return Component[] Component数组
     */
    parseSdks(potentialUniPackages) {
        const arkuixSdkMap = new Map();
        for (const uniPackage of potentialUniPackages) {
            try {
                const buffer = fs_1.default.readFileSync(uniPackage, { encoding: 'utf-8' });
                const parse = JSON.parse(buffer);
                const component = this.convertComponent(parse, uniPackage);
                if (!this.isValidComponent(component)) {
                    continue;
                }
                if (this._isDifferentMetaVersion(component)) {
                    continue;
                }
                if (this._isComponentInWrongPlace(component)) {
                    this.progress.debug(`Component ${component.getFullName()} has been placed in the wrong place,
             expect ${component.getLocation()}, actual ${path_1.default.resolve(uniPackage, '..')}`);
                    continue;
                }
                this.addToMap(arkuixSdkMap, component);
            }
            catch (e) {
                this.progress.warn(`Failed to get ${arkuix_component_constants_1.ARKUIX_UNI_PACKAGE} info, ${e.message}`);
            }
        }
        const sdks = [];
        arkuixSdkMap.forEach(value => sdks.push(value));
        return sdks;
    }
    /**
     * 将json字符串转换为ComponentDto类
     * @param jsonObject
     * @param uniPackage
     * @protected
     */
    convertComponent(jsonObject, uniPackage) {
        const componentDto = new oh_component_dto_1.OhComponentDto();
        componentDto.path = jsonObject.path;
        componentDto.releaseType = jsonObject.releaseType;
        componentDto.version = jsonObject.version;
        componentDto.displayName = jsonObject.displayName;
        const apiVersion = Number(jsonObject.apiVersion);
        if (!Number.isNaN(apiVersion)) {
            componentDto.apiVersion = apiVersion;
        }
        this.configMeta(jsonObject, componentDto);
        componentDto.displayName = this.convertDisplayName(componentDto);
        componentDto.location = path_1.default.resolve(uniPackage, '..');
        return componentDto;
    }
    configMeta(jsonObject, componentDto) {
        var _a;
        const metaDto = new abstract_component_dto_1.MetaDto();
        const metaVersion = (_a = jsonObject.meta) === null || _a === void 0 ? void 0 : _a.metaVersion;
        if ((0, utils_1.isEmpty)(metaVersion)) {
            metaDto.metaVersion = arkuix_component_constants_1.ARKUIX_META_VERSION;
        }
        else {
            metaDto.metaVersion = metaVersion;
        }
        componentDto.meta = metaDto;
    }
    _isDifferentMetaVersion(dto) {
        var _a;
        const metaVersion = (_a = dto.meta) === null || _a === void 0 ? void 0 : _a.getMetaVersion();
        if ((0, utils_1.isEmpty)(metaVersion) || !component_contants_1.META_VERSION_PATTERN.test(metaVersion)) {
            this.progress.debug(`Illegal component ${dto.displayName}:${dto.version}, Incorrect meta version ${metaVersion}`);
            return true;
        }
        const compareResult = (0, utils_1.compareVersion)(metaVersion, arkuix_component_constants_1.ARKUIX_META_VERSION);
        if (compareResult !== 0) {
            this._inCompatibleComponents.push(dto);
            return true;
        }
        return false;
    }
    _isComponentInWrongPlace(component) {
        const actualLocation = component.getLocation();
        if (actualLocation !== this._getExpectedLocation(component)) {
            this._inCompatibleComponents.push(component);
            return true;
        }
        return false;
    }
    _getExpectedLocation(component) {
        return this.getLocation(component, this.getSdkRoot());
    }
    getUniPackageName() {
        return arkuix_component_constants_1.ARKUIX_UNI_PACKAGE;
    }
    getSdkRoot() {
        return this.sdkRoot;
    }
    /**
     * 获取不兼容的sdk
     */
    getInCompatibleComponents() {
        return this._inCompatibleComponents;
    }
}
exports.ArkUIXLocalComponentLoader = ArkUIXLocalComponentLoader;
/**
 * 扫描文件夹深度的最大值
 * @private
 */
ArkUIXLocalComponentLoader._MAX_SCAN_DEPTH = 3;
