"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOsArch = exports.getEnvPathSeparator = exports.getEnvPathVariable = exports.getOsType = exports.isMac = exports.isLinux = exports.isWindows = void 0;
const os_1 = __importDefault(require("os"));
const path_1 = __importDefault(require("path"));
const child_process_1 = require("child_process");
const utils_1 = require("./utils");
function isWindows() {
    return os_1.default.platform() === 'win32';
}
exports.isWindows = isWindows;
function isLinux() {
    return os_1.default.platform() === 'linux';
}
exports.isLinux = isLinux;
function isMac() {
    return os_1.default.platform() === 'darwin';
}
exports.isMac = isMac;
const getOsType = () => {
    if (isLinux()) {
        return 'linux';
    }
    else if (isWindows()) {
        return 'windows';
    }
    else if (isMac()) {
        return 'darwin';
    }
    else {
        return 'other';
    }
};
exports.getOsType = getOsType;
function getEnvPathVariable() {
    return isWindows() ? 'Path' : 'PATH';
}
exports.getEnvPathVariable = getEnvPathVariable;
function getEnvPathSeparator() {
    return path_1.default.delimiter;
}
exports.getEnvPathSeparator = getEnvPathSeparator;
var CPUArch;
(function (CPUArch) {
    CPUArch["X64"] = "x64";
    CPUArch["ARM64"] = "arm64";
    CPUArch["UNKNOWN"] = "";
    CPUArch["X86_64"] = "x86_64";
    CPUArch["AMD64"] = "amd64";
    CPUArch["AARCH64"] = "aarch64";
})(CPUArch || (CPUArch = {}));
const isIntel64 = (osArch) => {
    return CPUArch.X86_64 === osArch || CPUArch.AMD64 === osArch || CPUArch.X64 === osArch;
};
const isArm64 = (osArch) => {
    return CPUArch.AARCH64 === osArch || CPUArch.ARM64 === osArch;
};
const getOsArchFromCommand = () => {
    var _a;
    const spawnSyncReturns = (0, child_process_1.spawnSync)('uname', ['-m'], {
        encoding: 'utf-8',
        shell: true,
        windowsHide: true,
    });
    let osArchFromCommand = spawnSyncReturns.stdout;
    const commandErrorInfo = (_a = spawnSyncReturns.error) === null || _a === void 0 ? void 0 : _a.message;
    if ((0, utils_1.isEmpty)(osArchFromCommand) && (0, utils_1.isNotEmpty)(commandErrorInfo)) {
        osArchFromCommand = commandErrorInfo;
    }
    return osArchFromCommand;
};
const osArch = getOsArchFromCommand();
const getOsArchFromMac = () => {
    for (const cpuInfo of os_1.default.cpus()) {
        if (cpuInfo.model.includes('Intel')) {
            return CPUArch.X64;
        }
        if (cpuInfo.model.includes('Apple')) {
            return CPUArch.ARM64;
        }
    }
    // 兜底返回x64
    return CPUArch.X64;
};
/**
 * 获取CPU架构信息
 * @param progress
 */
const getOsArch = (progress) => {
    if (isWindows()) {
        return CPUArch.X64;
    }
    if (isMac()) {
        return getOsArchFromMac();
    }
    if (isIntel64(osArch)) {
        return CPUArch.X64;
    }
    else if (isArm64(osArch)) {
        return CPUArch.ARM64;
    }
    else {
        progress === null || progress === void 0 ? void 0 : progress.info(`CPU Architecture: ${osArch}`);
        return CPUArch.X64;
    }
};
exports.getOsArch = getOsArch;
