"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OhLocalComponentLoader = void 0;
const abstract_local_component_loader_1 = require("../../core/loader/abstract-local-component-loader");
const utils_1 = require("../../core/util/utils");
const component_contants_1 = require("../../core/constants/component-contants");
const abstract_component_dto_1 = require("../../core/loader/abstract-component-dto");
const oh_component_constants_1 = require("../const/oh-component-constants");
const path_1 = __importDefault(require("path"));
const oh_component_dto_1 = require("./oh-component-dto");
const fs_1 = __importDefault(require("fs"));
class OhLocalComponentLoader extends abstract_local_component_loader_1.AbstractLocalComponentLoader {
    constructor(sdkRoot, progress) {
        super(OhLocalComponentLoader._MAX_SCAN_DEPTH, sdkRoot, progress);
        this._inCompatibleComponents = new Array();
    }
    /**
     * 将扫描到的文件转换为Component
     * @param potentialUniPackages
     * @protected
     */
    parseSdks(potentialUniPackages) {
        const sdkMap = new Map();
        for (const uniPackage of potentialUniPackages) {
            try {
                const buffer = fs_1.default.readFileSync(uniPackage, { encoding: 'utf-8' });
                const parse = JSON.parse(buffer);
                const component = this.convertComponent(parse, uniPackage);
                if (!this.isValidComponent(component)) {
                    continue;
                }
                if (!this._isSameMetaVersion(component)) {
                    continue;
                }
                if (!this._isComponentInRightPlace(component)) {
                    this.progress.debug(`Component ${component.getFullName()} has been placed in the wrong place,
             expect ${component.getLocation()}, actual ${path_1.default.resolve(uniPackage, '..')}`);
                    continue;
                }
                this.addToMap(sdkMap, component);
            }
            catch (e) {
                this.progress.warn(`Failed to get oh-uni-package info, ${e.message}`);
            }
        }
        const sdks = [];
        sdkMap.forEach(value => sdks.push(value));
        return sdks;
    }
    /**
     * 将json字符串转换为ComponentDto类
     * @param jsonObject
     * @protected
     */
    convertComponent(jsonObject, uniPackage) {
        const componentDto = new oh_component_dto_1.OhComponentDto();
        componentDto.path = jsonObject.path;
        componentDto.version = jsonObject.version;
        componentDto.releaseType = jsonObject.releaseType;
        componentDto.displayName = jsonObject.displayName;
        const apiVersion = Number(jsonObject.apiVersion);
        if (!Number.isNaN(apiVersion)) {
            componentDto.apiVersion = apiVersion;
        }
        if (component_contants_1.ComponentPath.TOOLCHAINS === componentDto.getPath() &&
            oh_component_constants_1.TOOLCHAINS_3_0_0_0_VERSION === componentDto.getVersion()) {
            componentDto.apiVersion = oh_component_constants_1.TOOLCHAINS_3_0_0_0_API7_MAPPING; // 需要迁移的toolchains的apiversion固定设置为7
        }
        this.configMeta(jsonObject, componentDto);
        componentDto.displayName = this.convertDisplayName(componentDto);
        componentDto.location = path_1.default.resolve(uniPackage, '..');
        return componentDto;
    }
    /**
     * 配置json里面的meta字段信息,需要子类重写
     * @param jsonObject
     * @param dto OhComponentDto
     * @protected
     */
    configMeta(jsonObject, dto) {
        var _a;
        const metaDto = new abstract_component_dto_1.MetaDto();
        const metaVersion = (_a = jsonObject.meta) === null || _a === void 0 ? void 0 : _a.metaVersion;
        if ((0, utils_1.isEmpty)(metaVersion)) {
            metaDto.metaVersion = oh_component_constants_1.SUPPORT_META_VERSION;
        }
        else {
            metaDto.metaVersion = metaVersion;
        }
        dto.meta = metaDto;
    }
    _isSameMetaVersion(dto) {
        var _a;
        const metaVersion = (_a = dto.meta) === null || _a === void 0 ? void 0 : _a.getMetaVersion();
        if ((0, utils_1.isEmpty)(metaVersion) || !component_contants_1.META_VERSION_PATTERN.test(metaVersion)) {
            this.progress.debug(`Illegal component ${dto.displayName}:${dto.version}, Incorrect meta version ${metaVersion}`);
            return false;
        }
        if (oh_component_constants_1.IGNORE_SDK_VERSIONS.includes(dto.version)) {
            this._inCompatibleComponents.push(dto);
            return false;
        }
        const compareResult = (0, utils_1.compareVersion)(metaVersion, oh_component_constants_1.SUPPORT_META_VERSION);
        if (compareResult !== 0) {
            this._inCompatibleComponents.push(dto);
            return false;
        }
        return true;
    }
    /**
     * 校验sdk的路径
     *
     * @param component
     * @protected
     */
    _isComponentInRightPlace(component) {
        const actualLocation = component.getLocation();
        if (actualLocation !== this._getExpectedLocation(component)) {
            this._inCompatibleComponents.push(component);
            return false;
        }
        return true;
    }
    _getExpectedLocation(component) {
        return this.getLocation(component, this.getSdkRoot());
    }
    getUniPackageName() {
        return oh_component_constants_1.OH_UNI_PACKAGE_NAME;
    }
    getSdkRoot() {
        return this.sdkRoot;
    }
    /**
     * 获取不兼容的sdk
     */
    getInCompatibleComponents() {
        return this._inCompatibleComponents;
    }
}
exports.OhLocalComponentLoader = OhLocalComponentLoader;
OhLocalComponentLoader._MAX_SCAN_DEPTH = 5;
