"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from, pack) {
    if (pack || arguments.length === 2) for (var i = 0, l = from.length, ar; i < l; i++) {
        if (ar || !(i in from)) {
            if (!ar) ar = Array.prototype.slice.call(from, 0, i);
            ar[i] = from[i];
        }
    }
    return to.concat(ar || Array.prototype.slice.call(from));
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var root_1 = __importDefault(require("./common/root"));
/**
 * 防抖管理类，生成防抖函数
 */
var DebounceMgr = /** @class */ (function () {
    function DebounceMgr(options) {
        var _this = this;
        // 是否指定在延迟开始前调用
        this.isLeadingEnabled = false;
        // 是否指定在延迟结束后调用
        this.isTrailingEnabled = true;
        // 是否开启最大等待时间功能
        this.isMaxWaitEnabled = false;
        // 记录最后一次 func 调用的时间
        this.lastInvokeTime = 0;
        /**
         * 获取 debounced 方法，DebounceMgr 唯一对外暴露的 API
         *
         * @param context 执行 debounced 方法时的 上下文 this
         * @param arg 执行 debounced 方法时的入参
         */
        this.debounced = function (context) {
            var arg = [];
            for (var _i = 1; _i < arguments.length; _i++) {
                arg[_i - 1] = arguments[_i];
            }
            var time = Date.now();
            var isInvoking = _this.shouldInvoke(time);
            // 每次执行 debounced 更新入参、上下文、debounced 执行的时间
            _this.lastArgs = arg;
            _this.lastThis = context;
            _this.lastCallTime = time;
            // 是否可以执行 func
            if (isInvoking) {
                if (_this.timerId === undefined) {
                    return _this.invokeLeading(_this.lastCallTime);
                }
                /**
                 * 对应 `should support `maxWait` in a tight loop` 测试用例，在短时间快速调用 debounced 函数，
                 * 由于 setTimeout 需等待主线程的代码执行完毕，防止下面条件里的 invokeFunc 和 invokeTrailing
                 * 里的invokeFunc 相继调用
                 */
                if (_this.isMaxWaitEnabled) {
                    clearTimeout(_this.timerId);
                    _this.timerId = _this.startTimer(_this.scheduleTimer, _this.wait);
                    return _this.invokeFunc(_this.lastCallTime);
                }
            }
            if (_this.timerId === undefined) {
                _this.timerId = _this.startTimer(_this.scheduleTimer, _this.wait);
            }
            return _this.debouncedResult;
        };
        /**
         * 立即执行 func，不需要等待 timer 时间到后再触发
         */
        this.flush = function () {
            if (_this.timerId === undefined) {
                // 计时器已经执行完毕，直接返回上次的执行结果
                return _this.debouncedResult;
            }
            // 传入当前时间执行 invokeTrailing
            return _this.invokeTrailing(Date.now());
        };
        /**
         * 取消计时器，并且对变量等进行重置
         */
        this.cancel = function () {
            if (_this.timerId !== undefined) {
                // 计时器还未执行，终止计数器
                _this.cancelTimer(_this.timerId);
            }
            // 重置变量参数
            _this.lastInvokeTime = 0;
            _this.lastCallTime = undefined;
            _this.lastArgs = undefined;
            _this.lastThis = undefined;
            _this.timerId = undefined;
        };
        /**
         * 定时器是否还未执行，状态是否处于 pending 状态
         */
        this.pending = function () {
            return _this.timerId !== undefined;
        };
        this.initData = function (options) {
            var func = options.func, wait = options.wait, _a = options.leading, leading = _a === void 0 ? false : _a, _b = options.trailing, trailing = _b === void 0 ? true : _b, maxWait = options.maxWait;
            /**
             * 是否使用 RAF，即使用 requestAnimationFrame API，当传入的 wait 为假值，并且没有指定为 0，
             * 并且当前环境支持 requestAnimationFrame 的情况下，才使用 requestAnimationFrame，
             * 该 API 会在浏览器重绘之前调用，性能比 setTimeout 更好。
             */
            _this.isUsingRAF = _this.wait === undefined && typeof root_1.default.requestAnimationFrame === 'function';
            _this.func = func;
            _this.wait = wait !== null && wait !== void 0 ? wait : 0;
            // 只有设置了最大等待时间，才允许开启最大等待模式
            _this.isMaxWaitEnabled = maxWait !== undefined;
            _this.maxWait = _this.isMaxWaitEnabled ? Math.max(maxWait !== null && maxWait !== void 0 ? maxWait : 0, wait) : maxWait;
            // 默认延迟结束后调用 func
            _this.isLeadingEnabled = leading;
            _this.isTrailingEnabled = trailing;
        };
        /**
         * 判断是否需要执行
         * @param time
         */
        this.shouldInvoke = function (time) {
            var timeSinceLastCall = time - _this.lastCallTime;
            var timeSinceLastInvoke = time - _this.lastInvokeTime;
            /**
             * 1、this.lastCallTime 为 undefined 时，即第一次调用 debounced 函数，
             * 2、timeSinceLastCall 大于等于 wait 时，即事件触发的间隔超过 wait 时，
             * 3、timeSinceLastCall 小于 0 时，怎么会在下次检测的时候，time 会比 lastCallTime 还要小的情况呢？
             *   其实这种应该算是边缘情况，例如在修改系统时间的情况下
             * 4、开启了最大等待时间功能，并且距离上次 func 执行的时间间隔大于 maxWait 时
             * 上面四种场景都需要执行 func 函数
             */
            return (_this.lastCallTime === undefined || (timeSinceLastCall >= _this.wait) || (timeSinceLastCall < 0)
                || (_this.isMaxWaitEnabled && timeSinceLastInvoke >= _this.maxWait));
        };
        this.invokeFunc = function (time) {
            var args = _this.lastArgs;
            var thisArg = _this.lastThis;
            _this.lastArgs = undefined;
            _this.lastThis = undefined;
            _this.lastInvokeTime = time;
            _this.debouncedResult = _this.func.apply(thisArg, args);
            return _this.debouncedResult;
        };
        /**
         * 每轮 debounce 开始时调用执行
         * @param time 当前时间戳
         * @private
         */
        this.invokeLeading = function (time) {
            _this.lastInvokeTime = time;
            // 启动计时器
            _this.timerId = _this.startTimer(_this.scheduleTimer, _this.wait);
            // 如果设置延迟前调用，则立即调用 invokeFunc 函数
            return _this.isLeadingEnabled ? _this.invokeFunc(time) : _this.debouncedResult;
        };
        /**
         * 计时器结束后执行
         * @param time 当前时间戳
         * @private
         */
        this.invokeTrailing = function (time) {
            _this.timerId = undefined;
            // 只有在 trailing 为 true 且 lastArgs 不为 undefined 时调用 invokeFunc 函数
            if (_this.isTrailingEnabled && _this.lastArgs) {
                return _this.invokeFunc(time);
            }
            _this.lastArgs = undefined;
            _this.lastThis = undefined;
            return _this.debouncedResult;
        };
        /**
         * 通过 shouldInvoke 函数判断是否执行 func 函数
         * 执行：调用 invokeTrailing 函数
         * 不执行：调用 startTimer 函数重新开始 timer，wait 值通过 calcRemainingWait 函数计算
         */
        this.scheduleTimer = function () {
            var time = Date.now();
            if (_this.shouldInvoke(time)) {
                // 定时时间到，执行 func
                _this.invokeTrailing(time);
                return;
            }
            // 使用 calcRemainingWait 函数重新计算等待时间，再次调用 startTimer
            _this.timerId = _this.startTimer(_this.scheduleTimer, _this.calcRemainingWait(time));
        };
        /**
         * 启用定时器，这里做了点优化，根据条件判断是使用 setTimeout 还是 requestAnimationFrame
         * @param pendingFunc
         * @param wait
         */
        this.startTimer = function (pendingFunc, wait) {
            if (_this.isUsingRAF) {
                // 使用 RAF
                root_1.default.cancelAnimationFrame(_this.timerId);
                return requestAnimationFrame(pendingFunc);
            }
            return setTimeout(pendingFunc, wait);
        };
        /**
         * 取消计时器
         * @param timerId 计时器ID
         */
        this.cancelTimer = function (timerId) {
            if (_this.isUsingRAF) {
                root_1.default.cancelAnimationFrame(timerId);
                return;
            }
            clearTimeout(timerId);
        };
        /**
         * 根据当前时间计算计时器剩余的时间
         * @param time 当前时间戳
         */
        this.calcRemainingWait = function (time) {
            // 计算距离上一次 debounced 函数调用的时间差
            var timeSinceLastCall = time - _this.lastCallTime;
            // 计算距离上一次 func 函数调用的时间差
            var timeSinceLastInvoke = time - _this.lastInvokeTime;
            // 计算没有开启最大等待时间项时的剩余等待时间
            var timeWaiting = _this.wait - timeSinceLastCall;
            return _this.isMaxWaitEnabled ? Math.min(timeWaiting, _this.maxWait - timeSinceLastInvoke) : timeWaiting;
        };
        this.initData(options);
    }
    return DebounceMgr;
}());
/**
 * 防抖 API，使用如下：
 * 1、生成防抖函数 debounced：debounce(log, 250, { maxWait: 1000 });
 * 2、事件绑定到防抖函数上：addEventListener('resize', debounced);
 * 上面 debounced 在事件每次触发时都会执行，log 是待执行的实际方法 func，在等待时间超过设置的时间下执行，达到防抖作用
 *
 * @param func 待执行的实际方法
 * @param wait 等待时间
 * @param options 设置项，参考 Options 接口定义
 */
function debounce(func, wait, options) {
    if (options === void 0) { options = {}; }
    if (typeof func !== 'function') {
        throw new TypeError('Expected a function');
    }
    var debounceMgrIns = new DebounceMgr(__assign(__assign({}, options), { func: func, wait: wait }));
    function debounced() {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return debounceMgrIns.debounced.apply(debounceMgrIns, __spreadArray([this], __read(args), false));
    }
    // 在 debounced 上提供三个 API
    debounced.flush = debounceMgrIns.flush;
    debounced.cancel = debounceMgrIns.cancel;
    debounced.pending = debounceMgrIns.pending;
    return debounced;
}
exports.default = debounce;
