"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getObjValidPathFromGeneralPath = void 0;
var utils_1 = require("./utils");
/**
 * 将指定的字符串类型的path（对象path），转换成数组类型。
 * @param strPath 字符串类型的path
 * @returns 按照path规则将字符串拆分成数组
 * @example
 * const stringPaths = 'a[-1.23]["[\\"b\\"]"].c[\'[\\\'d\\\']\'][\ne\n][f].g';
 * const paths = stringToPaths(stringPaths);
 * // => ['a', '-1.23', '"[\\"b\\"]"', 'c', '\'[\\\'d\\\']\'', '\ne\n', 'f', 'g']
 */
function stringToPaths(strPath) {
    if (strPath.trim() === '') {
        return [strPath];
    }
    var regex = /(?:\[('|")((?:\\[\s\S]|(?!\1)[^\\])+)\1\]|\[(-?\d+(?:\.\d+)?)\]|\[((?:\\[\s\S]|[^[\]])*?)\]|(\w+))/g;
    var arrayPaths = [];
    var match;
    while ((match = regex.exec(strPath)) !== null) {
        if (match[2]) {
            // 带有转义字符的带引号的字符
            arrayPaths.push("".concat(match[1]).concat(match[2]).concat(match[1]));
        }
        else if (match[3]) {
            // 数字匹配，含小数、正负数
            arrayPaths.push(match[3]);
        }
        else if (match[4] || match[4] === '') {
            // 匹配其他类型的字符，含空字符串
            arrayPaths.push("".concat(match[4]));
        }
        else if (match[5]) {
            // 纯字符串匹配，一个或多个字母、数字或下划线字符
            arrayPaths.push(match[5]);
        }
    }
    return arrayPaths;
}
exports.default = stringToPaths;
/**
 * 根据广义的路径得到对于obj而言合理的属性路径
 * @param obj，路径所属的obj
 * @param rawPath，广义的路径，可以是字符串或者数组
 * @return 对obj合理的属性路径
 */
function getObjValidPathFromGeneralPath(obj, rawPath) {
    if (typeof rawPath === 'symbol') {
        return [rawPath];
    }
    if (Array.isArray(rawPath)) {
        return rawPath.map(function (e) { return (0, utils_1.toStringWithZeroSign)(e); });
    }
    var generalPath = (0, utils_1.toStringWithZeroSign)(rawPath);
    if (obj !== undefined && obj !== null && generalPath in Object(obj)) {
        return [generalPath];
    }
    return stringToPaths(generalPath);
}
exports.getObjValidPathFromGeneralPath = getObjValidPathFromGeneralPath;
