"use strict";
/*
 * Copyright (c) Huawei Technologies Co., Ltd. 2023. All rights reserved.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var isNil_1 = __importDefault(require("./isNil"));
var toString_1 = __importDefault(require("./toString"));
var baseIsRegExp_1 = __importDefault(require("./common/baseIsRegExp"));
var isString_1 = __importDefault(require("./isString"));
var toInteger_1 = __importDefault(require("./toInteger"));
var has_1 = __importDefault(require("./has"));
var defaultOptions = {
    length: 30,
    omission: '...',
};
function findLastSeparatorIndex(validLenStr, opt) {
    var index = validLenStr.length;
    if (!(0, isNil_1.default)(opt.separator)) {
        var separator = (0, isString_1.default)(opt.separator) ? RegExp(opt.separator) : opt.separator;
        if (!(0, baseIsRegExp_1.default)(separator)) {
            separator = RegExp((0, toString_1.default)(opt.separator));
        }
        if (!separator.global) {
            separator = RegExp(separator.source, 'g');
        }
        var strSep = validLenStr.matchAll(separator);
        var matchedItem = strSep.next();
        // 找最后一个截断点 or 第一个截断点？
        while (!matchedItem.done) {
            index = matchedItem.value.index;
            matchedItem = strSep.next();
        }
    }
    return index;
}
function setDefaultIfNotExist(opt, result) {
    if (!(0, has_1.default)(opt, 'omission')) {
        result.omission = defaultOptions.omission;
    }
    else {
        result.omission = opt.omission;
    }
    if (!(0, has_1.default)(opt, 'length')) {
        result.length = defaultOptions.length;
    }
    else {
        result.length = opt.length;
    }
    if ((0, has_1.default)(opt, 'separator')) {
        result.separator = opt.separator;
    }
}
function coerceOptions(opt) {
    var result = {};
    setDefaultIfNotExist(opt, result);
    if (result.omission === null) {
        result.omission = 'null';
    }
    if (result.omission === undefined) {
        result.omission = 'undefined';
    }
    if (result.length === undefined) {
        result.length = defaultOptions.length;
    }
    result.length = (0, toInteger_1.default)(result.length);
    if (result.length < 0) {
        result.length = 0;
    }
    return result;
}
/**
 * 截断string字符串，如果字符串超出了限定的最大值。 被截断的字符串后面会以 omission 代替，omission 默认是 "..."。
 * @param str 要截断的字符串
 * @param options 为 Object 对象，截断的选项。<pre>
 *   options.length 默认值`30`，整形，允许的最大长度。
 *   options.omission 默认值 `...`，字符串类型，超出后的代替字符串
 *   options.separator 无默认值， RegExp | string 类型，截断点。
 *   </pre>
 * @returns 按照要求截断后的字符串
 * @example
 * ```ts
 * _.truncate('hi-diddly-ho there, neighborino');
 * // => 'hi-diddly-ho there, neighbo...'
 *
 * _.truncate('hi-diddly-ho there, neighborino', {
 *   'length': 24,
 *   'separator': ' '
 * });
 * // => 'hi-diddly-ho there,...'
 *
 * _.truncate('hi-diddly-ho there, neighborino', {
 *   'length': 24,
 *   'separator': /,? +/
 * });
 * // => 'hi-diddly-ho there...'
 *
 * _.truncate('hi-diddly-ho there, neighborino', {
 *   'omission': ' [...]'
 * });
 * // => 'hi-diddly-ho there, neig [...]'
 * ```
 */
function truncate(str, options) {
    if (str === void 0) { str = ''; }
    var opt = (0, isNil_1.default)(options) ? defaultOptions : options;
    // 将 nullish omission 转换成字符串，将 length 转换成 number ，负数 length 转成 0
    opt = coerceOptions(opt);
    var strValue = (0, toString_1.default)(str);
    if (strValue.length <= opt.length) {
        return str;
    }
    var validLenStr = strValue.substring(0, opt.length);
    var index = findLastSeparatorIndex(validLenStr, opt);
    // 缩略语加上来之后比总长度大时，截断以满足最大长度
    if (index + opt.omission.length > validLenStr.length) {
        var newIndex = 2 * validLenStr.length - opt.omission.length - index;
        return validLenStr.substring(0, newIndex) + opt.omission;
    }
    return validLenStr.substring(0, index) + opt.omission;
}
exports.default = truncate;
