"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractLocalComponentLoader = void 0;
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const simple_progress_1 = require("../log/simple-progress");
const abstract_component_loader_1 = require("./abstract-component-loader");
const file_util_1 = require("../util/file-util");
class AbstractLocalComponentLoader extends abstract_component_loader_1.AbstractComponentLoader {
    constructor(maxScanDepth, sdkRoot, progress = new simple_progress_1.SimpleProgress()) {
        super(progress);
        /**
         * 临时文件夹
         * @private
         */
        this._TEMP_PREFIX = ['.temp', '.delete'];
        this._maxScanDepth = maxScanDepth;
        this.sdkRoot = sdkRoot;
    }
    /**
     * 加载本地的sdk
     */
    load() {
        const potentialUniPackages = this.findPotentialSdks(this.getSdkRoot());
        return this.parseSdks(potentialUniPackages);
    }
    /**
     * 使用非递归方式扫描指定sdk根路径下潜在的sdk元数据文件
     * @param root
     * @protected
     */
    findPotentialSdks(root) {
        const potentialUniPackage = [];
        if (root === null || root === undefined) {
            return potentialUniPackage;
        }
        const stats = (0, file_util_1.fsStateSync)(root);
        if (stats === undefined || !stats.isDirectory()) {
            return potentialUniPackage;
        }
        if (!this._needScan(path_1.default.basename(root))) {
            return potentialUniPackage;
        }
        const depthMap = new Map();
        const stack = [];
        stack.push(root);
        depthMap.set(root, 1);
        while (stack.length !== 0) {
            const rootPath = stack.pop();
            const depth = depthMap.get(rootPath);
            if (depth > this._maxScanDepth || !this._needScan(path_1.default.basename(rootPath))) {
                continue;
            }
            // fs.readdirSync返回的路径仅仅是文件夹的名称,不是绝对路径
            for (const dirent of fs_1.default.readdirSync(rootPath, { withFileTypes: true })) {
                const childDirName = dirent.name;
                if (!this._needScan(childDirName)) {
                    continue;
                }
                const uniPackageFile = path_1.default.resolve(rootPath, childDirName, this.getUniPackageName());
                if (fs_1.default.existsSync(uniPackageFile)) {
                    potentialUniPackage.push(uniPackageFile);
                    continue;
                }
                if (!dirent.isDirectory()) {
                    continue;
                }
                const childDirPath = path_1.default.resolve(rootPath, childDirName);
                stack.push(childDirPath);
                depthMap.set(childDirPath, depth + 1);
            }
        }
        return potentialUniPackage;
    }
    _needScan(dirName) {
        for (const tempPrefix of this._TEMP_PREFIX) {
            if (dirName.startsWith(tempPrefix)) {
                return false;
            }
        }
        return true;
    }
}
exports.AbstractLocalComponentLoader = AbstractLocalComponentLoader;
