"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractInstallTask = void 0;
const fs_1 = __importDefault(require("fs"));
const crypto_1 = __importDefault(require("crypto"));
const path_1 = __importDefault(require("path"));
const zip_util_1 = require("../util/zip-util");
const sdk_exception_1 = require("../api/sdk-exception");
const error_tip_1 = require("../api/error-tip");
const os_1 = __importDefault(require("os"));
const file_util_1 = require("../util/file-util");
const abstract_component_task_1 = require("./abstract-component-task");
class AbstractInstallTask extends abstract_component_task_1.AbstractComponentTask {
    constructor(component, sdkSettings, netClient) {
        super(component);
        /**
         * 清除残留文件
         * @param filePath
         */
        this._cleanLegacyFile = (filePath) => __awaiter(this, void 0, void 0, function* () {
            try {
                yield fs_1.default.promises.rm(filePath, { force: true, recursive: true }); // 下载前先删除之前可能残留的文件
            }
            catch (e) {
                throw new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip(`${e === null || e === void 0 ? void 0 : e.message}`, `Unable to delete ${filePath}. A file may be open by another program.`, 'Close the open file, delete the folder, and then download the SDK again.'));
            }
        });
        this._verifyComponent = (downloadLocation, progress) => {
            var _a, _b, _c, _d;
            progress.debug('Verifying file...');
            const buffer = fs_1.default.readFileSync(downloadLocation);
            if (buffer.length !== ((_a = this.component.getArchive()) === null || _a === void 0 ? void 0 : _a.getSize())) {
                throw new sdk_exception_1.SdkException(`Failed to download SDK, size is incorrect, expected is ${(_b = this.component.getArchive()) === null || _b === void 0 ? void 0 : _b.getSize()}, actual is ${buffer.length}`, new error_tip_1.ErrorTip('', 'The network is unstable. SDK package integrity check failed.', `${os_1.default.EOL}1. Connect to a stable network and download the file again.\
          ${os_1.default.EOL}2. If the issue persists, choose Help > Show Log in Explorer to obtain logs, \
          and then choose Help > Contact Support for technical assistance.`));
            }
            const fsHash = crypto_1.default.createHash('sha256');
            fsHash.update(buffer);
            const md5 = fsHash.digest('hex');
            if (md5 !== ((_c = this.component.getArchive()) === null || _c === void 0 ? void 0 : _c.getChecksum())) {
                throw new sdk_exception_1.SdkException(`Failed to download SDK, checksum is incorrect, expected is ${(_d = this.component
                    .getArchive()) === null || _d === void 0 ? void 0 : _d.getChecksum()}, actual is ${md5}.`, new error_tip_1.ErrorTip('', 'Unreliable network environment. The SDK package file has been tampered with.', `${os_1.default.EOL}1. Connect to a trusted network and download the file again.\
          ${os_1.default.EOL}2. If the issue persists, choose Help > Show Log in Explorer to obtain logs, \
          and then choose Help > Contact Support for technical assistance.`));
            }
        };
        this._unzip = (downloadLocation, progress) => __awaiter(this, void 0, void 0, function* () {
            progress.info(`Unzipping ${downloadLocation}...`);
            const dest = this._getTempInstallLocation();
            yield this._cleanLegacyFile(dest);
            return (0, zip_util_1.unzip)(downloadLocation, dest);
        });
        this.netClient = netClient;
        this.sdkSettings = sdkSettings;
    }
    getSdkSettings() {
        return this.sdkSettings;
    }
    getComponent() {
        return this.component;
    }
    getSdkRoot() {
        return this.sdkSettings.getLocation();
    }
    prepare(progress) {
        return __awaiter(this, void 0, void 0, function* () {
            const downloadLocation = yield this.download(progress);
            this._verifyComponent(downloadLocation, progress);
            return this._unzip(downloadLocation, progress);
        });
    }
    complete(progress) {
        return __awaiter(this, void 0, void 0, function* () {
            return this._moveToTarget(progress);
        });
    }
    downloadHelper(downloadUrl, downloadLocation, progress) {
        return __awaiter(this, void 0, void 0, function* () {
            (0, file_util_1.ensureParentDir)(downloadLocation);
            yield this._cleanLegacyFile(downloadLocation);
            const isDownloadSuccess = yield this.netClient.doGetDownload(downloadUrl, downloadLocation, progress);
            if (!isDownloadSuccess) {
                throw new sdk_exception_1.SdkException('Failed to download sdk', new error_tip_1.ErrorTip('', 'The network or server is abnormal', `${os_1.default.EOL}1. Check the network connection and download the file again.\
            ${os_1.default.EOL}2. If the issue persists, choose Help > Show Log in Explorer to obtain logs, \
          and then choose Help > Contact Support for technical assistance.`));
            }
        });
    }
    getFileNameFromUrl(url) {
        const index = url.lastIndexOf('/');
        if (index === -1) {
            return url;
        }
        return url.substring(index + 1);
    }
    _moveToTarget(progress) {
        return __awaiter(this, void 0, void 0, function* () {
            progress.info('Moving the SDK...');
            const src = this.getTempComponentPath();
            const des = this.component.getLocation();
            if (!des) {
                throw new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip('', `Illegal component ${this.getComponent().getFullName()}, location is undefined`, ''));
            }
            yield this._cleanLegacyFile(des);
            (0, file_util_1.ensureParentDir)(des);
            try {
                yield fs_1.default.promises.rename(src, des);
            }
            catch (e) {
                progress.info('Copying the SDK...');
                yield fs_1.default.promises.cp(src, des, { recursive: true });
                return fs_1.default.promises.rm(src, { recursive: true, force: true }).catch(reason => {
                    progress.warn(`Unable to delete the temporary folder in the ${src} directory. You can delete it manually.`);
                    progress.warn(reason.message);
                });
            }
        });
    }
    getWorkDir() {
        return path_1.default.resolve(super.getWorkDir(), 'install');
    }
    _getTempInstallLocation() {
        return path_1.default.resolve(this.getWorkDir(), 'unzip');
    }
    getTempComponentPath() {
        const tempInstallLocation = this._getTempInstallLocation();
        const dirs = fs_1.default.readdirSync(tempInstallLocation);
        if (dirs.length !== 1) {
            throw new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip('', `Illegal component ${this.getComponent().getFullName()}`, 'Choose Help > Show Log in Explorer to obtain logs, and then choose Help > Contact Support ' +
                'for technical assistance.'));
        }
        const isExistUniPackageFile = fs_1.default.existsSync(path_1.default.resolve(tempInstallLocation, dirs[0], this.getUniPackageName()));
        if (!isExistUniPackageFile) {
            throw new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip('', `Illegal component ${this.getComponent().getFullName()}, 
        failed to find meta data.`, 'Choose Help > Show Log in Explorer to obtain logs, and then choose Help > Contact Support ' +
                'for technical assistance.'));
        }
        return path_1.default.resolve(tempInstallLocation, dirs[0]);
    }
}
exports.AbstractInstallTask = AbstractInstallTask;
