"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmInstallTask = void 0;
const sdk_exception_1 = require("../api/sdk-exception");
const error_tip_1 = require("../api/error-tip");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const abstract_component_task_1 = require("./abstract-component-task");
const url_util_1 = require("../util/url-util");
const os_type_util_1 = require("../util/os-type-util");
const utils_1 = require("../util/utils");
const child_process_1 = require("child_process");
const url_contants_1 = require("../constants/url-contants");
const file_util_1 = require("../util/file-util");
const NPM_ADD_COMPATIBLE_CONFIG_LOWEST_VERSION = '7.0.0';
const NPM_VERSION_PATTERN = /^(\d+)(\.\d+){2}$/;
class NpmInstallTask extends abstract_component_task_1.AbstractComponentTask {
    constructor(installTask) {
        super(installTask.getComponent());
        this._env = process.env;
        this._needNpmInstall = () => __awaiter(this, void 0, void 0, function* () {
            const acePath = this.getRunNpmInstallPath();
            if (!fs_1.default.existsSync(acePath)) { // 闭源SDK没有acePath,不需要执行npm install
                return Promise.resolve(false);
            }
            const nodeModulesPath = path_1.default.resolve(acePath, 'node_modules');
            if (fs_1.default.existsSync(nodeModulesPath)) { // acePath下面如果存在node_modules文件夹，说明已经有依赖了，就不需要执行npm install，新的SDK发布时自带了依赖
                const nodeModulesStats = yield fs_1.default.promises.stat(nodeModulesPath); // 当文件不存在时，获取stats对象会抛异常，所以先判断是否存在，再判断是否是文件夹
                if (nodeModulesStats.isDirectory()) {
                    return Promise.resolve(false);
                }
            }
            return Promise.resolve(true);
        });
        this._doNpmInstall = (progress) => __awaiter(this, void 0, void 0, function* () {
            return this._installDependencies(progress);
        });
        this._installDependencies = (progress) => __awaiter(this, void 0, void 0, function* () {
            progress.info(`Installing ${this.getComponent().getDisplayName()} dependencies...`);
            const acePath = this.getRunNpmInstallPath();
            this._verifySdkJs(acePath);
            return this._runCmd(acePath, progress);
        });
        this._verifySdkJs = (acePath) => {
            const packagePath = path_1.default.resolve(acePath, 'package.json');
            if (fs_1.default.existsSync(packagePath) && !fs_1.default.statSync(packagePath).isDirectory()) {
                return;
            }
            throw new sdk_exception_1.SdkException('Unable to execute the npm install command.', new error_tip_1.ErrorTip('', 'The package.json file cannot be found.', 'Choose Help > Show Log in Explorer to obtain logs, ' +
                'and then choose Help > Contact Support for technical assistance.'));
        };
        this._runCmd = (acePath, progress) => __awaiter(this, void 0, void 0, function* () {
            progress.info('Running \'npm install\'...');
            this._addCommonNpmCmdEnv();
            this._addNpmBinPathToEnv();
            this._addNpmRegistryIfNeed();
            yield this._addCompatibleLaterNodeVersionConfig(acePath);
            const args = [
                'install',
                '--fetch-retry-mintimeout=10000',
                '--fetch-retry-maxtimeout=15000',
                '--fetch-retries=1',
            ];
            return this.npmInstall(acePath, args, progress);
        });
        this._addCommonNpmCmdEnv = () => {
            // npm包update-notifier的配置，关闭更新检查，每次启动会打开更新检查，没网情况下，启动cmd时会一直不退出，导致js安装失败
            this._env['NO_UPDATE_NOTIFIER'] = 'true';
        };
        this._addNpmBinPathToEnv = () => {
            const npmDirPath = path_1.default.dirname(process.execPath);
            const envPath = this._env[(0, os_type_util_1.getEnvPathVariable)()];
            this._env[(0, os_type_util_1.getEnvPathVariable)()] = npmDirPath + (0, os_type_util_1.getEnvPathSeparator)() + envPath;
        };
        this._addNpmRegistryIfNeed = () => {
            const stringSpawnSyncReturns = (0, child_process_1.spawnSync)(`"${this.getNpmBinPath()}"`, ['config', 'get', '@ohos:registry'], {
                encoding: 'utf-8',
                env: this._env,
                shell: true,
                windowsHide: true,
            });
            const registry = stringSpawnSyncReturns.stdout.trim();
            if ((0, utils_1.isEmpty)(registry) || registry === 'undefined') {
                this._env['npm_config_@ohos:registry'] = `${url_contants_1.urls.harmonyOSRepo}/npm`;
            }
        };
        /**
         * npm -v命令有可能出现多行，而且最后一行才是版本号，如果npmrc里面有配置一些不兼容的参数，那么前面会出现告警提示，所以要取最后一行
         * @param npmVersion
         */
        this._processNpmVersion = (npmVersion) => {
            if ((0, utils_1.isEmpty)(npmVersion)) {
                return npmVersion;
            }
            const results = npmVersion.split('\n');
            if (results.length > 0) {
                return results[results.length - 1].trim();
            }
            return npmVersion;
        };
        this.getNpmBinPath = () => {
            return path_1.default.resolve(path_1.default.dirname(process.execPath), 'npm');
        };
        this.npmInstall = (_cwd, args, progress) => __awaiter(this, void 0, void 0, function* () {
            const npmHelpUrl = (0, url_util_1.getNpmHelpUrl)(yield this.getSdkSettings().getGrsService());
            return new Promise((resolve, reject) => {
                const installProcess = (0, child_process_1.spawn)(`"${this.getNpmBinPath()}"`, args, {
                    cwd: _cwd,
                    env: this._env,
                    shell: true,
                    windowsHide: true, // 当node以server模式运行的时候，使用spawn启动命令行在windows上面会出现cmd窗口一闪而逝的现象，加上这个参数可以使cmd窗口不弹出
                });
                installProcess.stdout.on('data', (data) => {
                    progress.info(`${String(data)}`);
                });
                installProcess.stderr.on('data', (data) => {
                    progress.info(`${String(data)}`);
                });
                installProcess.on('close', (code) => {
                    if (code === 0) { // exitCode为0才代表正常退出
                        progress.info('\'npm install\' executed');
                        resolve();
                    }
                    else {
                        reject(new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip(`Install ${this.getComponent().getDisplayName()} dependencies failed.`, 'Unable to run \'npm install\'.', `See ${npmHelpUrl}`)));
                    }
                });
                installProcess.on('error', (err) => {
                    progress.warn(err.message);
                    reject(new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip(`Install ${this.getComponent().getDisplayName()} dependencies failed.`, 'Unable to run \'npm install\'.', `See ${npmHelpUrl}`)));
                });
                process.on('exit', () => {
                    // 如果父级process退出了，那么子process也结束掉
                    if (!installProcess.killed) {
                        installProcess.kill('SIGINT');
                    }
                });
            });
        });
        this._installTask = installTask;
    }
    prepare(progress) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this._installTask.prepare(progress);
            if (yield this._needNpmInstall()) {
                return this._doNpmInstall(progress);
            }
        });
    }
    complete(progress) {
        return __awaiter(this, void 0, void 0, function* () {
            return this._installTask.complete(progress);
        });
    }
    /**
     * npm 7.x以上的版本执行npm install会有问题，需要做一些特殊处理
     * @private
     */
    _addCompatibleLaterNodeVersionConfig(acePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const stringSpawnSyncReturns = (0, child_process_1.spawnSync)(`"${this.getNpmBinPath()}"`, ['-v'], {
                encoding: 'utf-8',
                env: this._env,
                shell: true,
                windowsHide: true,
            });
            const stdout = stringSpawnSyncReturns.stdout.trim();
            const npmVersion = this._processNpmVersion(stdout);
            if ((0, utils_1.isNotEmpty)(npmVersion) && NPM_VERSION_PATTERN.test(npmVersion)
                && (0, utils_1.compareVersion)(npmVersion, NPM_ADD_COMPATIBLE_CONFIG_LOWEST_VERSION) > 0) {
                this._env['npm_config_legacy_peer_deps'] = 'true';
                yield this._processPackageLock(acePath);
            }
            return Promise.resolve();
        });
    }
    /**
     * SDK的package-lock.json里面有resolved字段，该字段描述的是某个npm包的下载地址
     * npm6.x版本在执行npm install的时候不会关注resolved字段，下载npm包是从配置的registry仓里面获取
     * npm7.x以上版本在执行npm install的时候会关注resolved字段，下载npm包从resolved字段获取，此时用户配置的registry没有生效
     * 由于地址写死成外网华为npm仓，用户无法通过切换registry来下载npm依赖，会导致npm install失败
     *
     * 这里采用的方案是在npm install之前将所有的package-lock.json文件里面的resolved字段值替换成false，
     * 这样npm就可以去用户配置的registry仓库下载依赖了
     * @param acePath 需要执行npm install的根目录
     * @private
     */
    _processPackageLock(acePath) {
        return __awaiter(this, void 0, void 0, function* () {
            for (const packageLock of yield (0, file_util_1.findFiles)(acePath, fileName => 'package-lock.json' === fileName)) {
                const oldContent = yield fs_1.default.promises.readFile(packageLock, { encoding: 'utf-8' });
                const newContent = oldContent.replace(/"resolved".+/g, '"resolved": false,');
                yield fs_1.default.promises.writeFile(packageLock, newContent, { encoding: 'utf-8' });
            }
        });
    }
    getComponent() {
        return this._installTask.getComponent();
    }
    getWorkDir() {
        return this._installTask.getWorkDir();
    }
    getRootWorkDir() {
        return this._installTask.getRootWorkDir();
    }
    getSdkSettings() {
        return this._installTask.getSdkSettings();
    }
    getSdkRoot() {
        return this._installTask.getSdkRoot();
    }
    getTempComponentPath() {
        return this._installTask.getTempComponentPath();
    }
    download(progress) {
        progress.debug('Nothing to do');
        return Promise.resolve('');
    }
}
exports.NpmInstallTask = NpmInstallTask;
