import process from 'process';
import { isMainThread, parentPort, workerData, Worker } from 'worker_threads';
import { isObject, hasOwnProperty, merge } from 'smob';
import { minify } from 'terser';
import { fileURLToPath } from 'url';
import { cpus } from 'os';
import { EventEmitter } from 'events';
import serializeJavascript from 'serialize-javascript';

/**
 * Duck typing worker context.
 *
 * @param input
 */
function isWorkerContextSerialized(input) {
    return (isObject(input) &&
        hasOwnProperty(input, 'code') &&
        typeof input.code === 'string' &&
        hasOwnProperty(input, 'options') &&
        typeof input.options === 'string');
}
async function runWorker() {
    if (isMainThread || !parentPort || !isWorkerContextSerialized(workerData)) {
        return;
    }
    try {
        // eslint-disable-next-line no-eval
        const eval2 = eval;
        const options = eval2(`(${workerData.options})`);
        const result = await minify(workerData.code, options);
        const output = {
            code: result.code || workerData.code,
            nameCache: options.nameCache
        };
        if (typeof result.map === 'string') {
            output.sourceMap = JSON.parse(result.map);
        }
        if (isObject(result.map)) {
            output.sourceMap = result.map;
        }
        parentPort.postMessage(output);
    }
    catch (e) {
        process.exit(1);
    }
}

const symbol = Symbol.for('FreeWoker');
class WorkerPool extends EventEmitter {
    constructor(options) {
        super();
        this.tasks = [];
        this.workers = 0;
        this.maxInstances = options.maxWorkers || cpus().length;
        this.filePath = options.filePath;
        this.on(symbol, () => {
            if (this.tasks.length > 0) {
                this.run();
            }
        });
    }
    add(context, cb) {
        this.tasks.push({
            context,
            cb
        });
        if (this.workers >= this.maxInstances) {
            return;
        }
        this.run();
    }
    async addAsync(context) {
        return new Promise((resolve, reject) => {
            this.add(context, (err, output) => {
                if (err) {
                    reject(err);
                    return;
                }
                if (!output) {
                    reject(new Error('The output is empty'));
                    return;
                }
                resolve(output);
            });
        });
    }
    run() {
        if (this.tasks.length === 0) {
            return;
        }
        const task = this.tasks.shift();
        if (typeof task === 'undefined') {
            return;
        }
        this.workers += 1;
        let called = false;
        const callCallback = (err, output) => {
            if (called) {
                return;
            }
            called = true;
            this.workers -= 1;
            task.cb(err, output);
            this.emit(symbol);
        };
        const worker = new Worker(this.filePath, {
            workerData: {
                code: task.context.code,
                options: serializeJavascript(task.context.options)
            }
        });
        worker.on('message', (data) => {
            callCallback(null, data);
        });
        worker.on('error', (err) => {
            callCallback(err);
        });
        worker.on('exit', (code) => {
            if (code !== 0) {
                callCallback(new Error(`Minify worker stopped with exit code ${code}`));
            }
        });
    }
}

function terser(input = {}) {
    const { maxWorkers, ...options } = input;
    const workerPool = new WorkerPool({
        filePath: fileURLToPath(import.meta.url),
        maxWorkers
    });
    return {
        name: 'terser',
        async renderChunk(code, chunk, outputOptions) {
            const defaultOptions = {
                sourceMap: outputOptions.sourcemap === true || typeof outputOptions.sourcemap === 'string'
            };
            if (outputOptions.format === 'es') {
                defaultOptions.module = true;
            }
            if (outputOptions.format === 'cjs') {
                defaultOptions.toplevel = true;
            }
            try {
                const { code: result, nameCache, sourceMap } = await workerPool.addAsync({
                    code,
                    options: merge({}, options || {}, defaultOptions)
                });
                if (options.nameCache && nameCache) {
                    let vars = {
                        props: {}
                    };
                    if (hasOwnProperty(options.nameCache, 'vars') && isObject(options.nameCache.vars)) {
                        vars = merge({}, options.nameCache.vars || {}, vars);
                    }
                    if (hasOwnProperty(nameCache, 'vars') && isObject(nameCache.vars)) {
                        vars = merge({}, nameCache.vars, vars);
                    }
                    // eslint-disable-next-line no-param-reassign
                    options.nameCache.vars = vars;
                    let props = {};
                    if (hasOwnProperty(options.nameCache, 'props') && isObject(options.nameCache.props)) {
                        // eslint-disable-next-line prefer-destructuring
                        props = options.nameCache.props;
                    }
                    if (hasOwnProperty(nameCache, 'props') && isObject(nameCache.props)) {
                        props = merge({}, nameCache.props, props);
                    }
                    // eslint-disable-next-line no-param-reassign
                    options.nameCache.props = props;
                }
                if ((!!defaultOptions.sourceMap || !!options.sourceMap) && isObject(sourceMap)) {
                    return {
                        code: result,
                        map: sourceMap
                    };
                }
                return result;
            }
            catch (e) {
                return Promise.reject(e);
            }
        }
    };
}

runWorker();

export { terser as default };
//# sourceMappingURL=index.js.map
