"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var isInteger_1 = __importDefault(require("./isInteger"));
var isNaN_1 = __importDefault(require("./isNaN"));
var isNil_1 = __importDefault(require("./isNil"));
var toNumber_1 = __importDefault(require("./toNumber"));
/**
 * 产生一个包括 lower 与 upper 之间的数。 如果只提供一个参数返回一个0到提供数之间的数。
 * 如果 floating 设为 true，或者 lower 或 upper 是浮点数，结果返回浮点数。
 *
 * <B>注意</B>: JavaScript 遵循 IEEE-754 标准处理无法预料的浮点数结果。
 * @since 1.0.0
 * @param [lower=0] (number): 下限。
 * @param [upper=1] (number): 上限。
 * @param [floating]  (boolean): 指定是否返回浮点数。
 * @example
 * _.random(0, 5);
 * // => an integer between 0 and 5
 *
 * _.random(5);
 * // => also an integer between 0 and 5
 *
 * _.random(5, true);
 * // => a floating-point number between 0 and 5
 *
 * _.random(1.2, 5.2);
 * // => a floating-point number between 1.2 and 5.2
 */
function random(lower, upper, floating) {
    var isFloat = floating;
    var min = lower;
    var max = upper;
    if ((0, isNil_1.default)(floating)) {
        if (typeof upper === 'boolean') {
            isFloat = upper;
            max = undefined;
        }
        else if (typeof lower === 'boolean') {
            isFloat = lower;
            min = undefined;
        }
    }
    // 先排除 nil
    min = (0, isNil_1.default)(min) ? 0 : min;
    max = (0, isNil_1.default)(max) ? 1 : max;
    // 再排除非 Number 的对象
    if (typeof min !== 'number') {
        min = (0, toNumber_1.default)(min);
    }
    if (typeof max !== 'number') {
        max = (0, toNumber_1.default)(max);
    }
    // 再排除 NaN
    if ((0, isNaN_1.default)(max)) {
        max = 0;
    }
    if ((0, isNaN_1.default)(min)) {
        min = 0;
    }
    // 最后按照大小排序
    var temp = min;
    if (min > max) {
        min = max;
        max = temp;
    }
    // 对 Infinity 做特殊处理：只要有一个操作数是 Infinity，结果就是 Infinity，使用最大整数表达
    if (min === Infinity || max === Infinity) {
        return Number.MAX_VALUE;
    }
    if (typeof isFloat !== 'boolean') {
        isFloat = false;
    }
    if (shouldBeFloating(min, max, isFloat)) {
        return Math.random() * (max - min) + min;
    }
    return getRandomInt(min, max);
}
function getRandomInt(lower, upper) {
    var min = Math.ceil(lower);
    var max = Math.floor(upper);
    return Math.floor(Math.random() * (max - min + 1) + min);
}
function shouldBeFloating(lower, upper, floating) {
    return floating || !(0, isInteger_1.default)(lower) || !(0, isInteger_1.default)(upper);
}
exports.default = random;
