"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.etsCollect = void 0;
const fs = require("fs");
const path = require("path");
const sourceMap = require("source-map");
const libCoverage = require("istanbul-lib-coverage");
const log4js = require("log4js");
const os = require("os");
const logger = log4js.getLogger();
/**
 * 功能：将js的覆盖率数据转换为ets的代码覆盖率数据
 * @param {string[]} coverageDataArrayPath 覆盖率数据路径组
 * @param {string} projectPath 项目路径
 */
async function etsCollect(coverageDataArrayPath, projectPath) {
    let etsCoverageData = {};
    try {
        const map = libCoverage.createCoverageMap();
        coverageDataArrayPath.forEach((element) => {
            // 对初始化覆盖率文件format
            if (element.indexOf('init_coverage.json') != -1) {
                let initCoverageData = fs.readFileSync(element);
                initCoverageData = '{' + os.EOL + initCoverageData + '}' + os.EOL;
                map.merge(JSON.parse(initCoverageData));
            }
            else {
                map.merge(JSON.parse(fs.readFileSync(element)));
            }
        });
        let coverageObject = JSON.stringify(map);
        let jsCoverageData = JSON.parse(coverageObject);
        let etsTempCoverageData = {};
        for (let key of Object.keys(jsCoverageData)) {
            let rawSourceMap = jsCoverageData[key].inputSourceMap;
            if (rawSourceMap === undefined) {
                continue;
            }
            let statementMap = jsCoverageData[key].statementMap;
            let fnMap = jsCoverageData[key].fnMap;
            let branchMap = jsCoverageData[key].branchMap;
            let source = rawSourceMap.sources[0];
            source = projectPath + path.sep + source;
            etsTempCoverageData[source] = jsCoverageData[key];
            etsTempCoverageData[source].path = source;
            await sourceMap.SourceMapConsumer.with(rawSourceMap, null, function (consumer) {
                // statementMap转换
                transFormStatementMap(statementMap, consumer, etsTempCoverageData, source);
                // fnMap转换
                transFormFnMap(fnMap, consumer, etsTempCoverageData, source);
                // branchMap转换
                transFormBranchMap(branchMap, consumer, etsTempCoverageData, source);
                filterCoverage(etsTempCoverageData, etsCoverageData);
            });
        }
    }
    catch (err) {
        logger.error('transform failed, error:' + err);
    }
    return etsCoverageData;
}
exports.etsCollect = etsCollect;
/**
 * 功能：statementMap转换
 * @param {statementMap} statementMap
 * @param {BasicSourceMapConsumer | IndexedSourceMapConsumer} consumer
 * @param {coverageMap} etsTempCoverageData
 * @param {string} source
 */
function transFormStatementMap(statementMap, consumer, etsTempCoverageData, source) {
    for (let i of Object.keys(statementMap)) {
        let statementStart = consumer.originalPositionFor(statementMap[i].start);
        statementMap[i].start.line = statementStart.line || 0;
        statementMap[i].start.column = statementStart.column || 0;
        let statementEnd = consumer.originalPositionFor(statementMap[i].end);
        statementMap[i].end.line = statementEnd.line || 0;
        statementMap[i].end.column = statementEnd.column || 0;
    }
    etsTempCoverageData[source].statementMap = statementMap;
}
/**
 * 功能：fnMap转换
 * @param {fnMap} fnMap
 * @param {BasicSourceMapConsumer | IndexedSourceMapConsumer} consumer
 * @param {coverageMap} etsTempCoverageData
 * @param {string} source
 */
function transFormFnMap(fnMap, consumer, etsTempCoverageData, source) {
    for (let i of Object.keys(fnMap)) {
        // decl
        let declStart = consumer.originalPositionFor(fnMap[i].decl.start);
        fnMap[i].decl.start.line = declStart.line || 0;
        fnMap[i].decl.start.column = declStart.column || 0;
        let declEnd = consumer.originalPositionFor(fnMap[i].decl.end);
        fnMap[i].decl.end.line = declEnd.line || 0;
        fnMap[i].decl.end.column = declEnd.column || 0;
        // loc
        let locStart = consumer.originalPositionFor(fnMap[i].loc.start);
        fnMap[i].loc.start.line = locStart.line || 0;
        fnMap[i].loc.start.column = locStart.column || 0;
        let locEnd = consumer.originalPositionFor(fnMap[i].loc.end);
        fnMap[i].loc.end.line = locEnd.line || 0;
        fnMap[i].loc.end.column = locEnd.column || 0;
        // line
        fnMap[i].line = declStart.line || 0;
    }
    etsTempCoverageData[source].fnMap = fnMap;
}
/**
 * 功能：branchMap转换
 * @param {branchMap} branchMap
 * @param {BasicSourceMapConsumer | IndexedSourceMapConsumer} consumer
 * @param {coverageMap} etsTempCoverageData
 * @param {string} source
 */
function transFormBranchMap(branchMap, consumer, etsTempCoverageData, source) {
    for (let i of Object.keys(branchMap)) {
        // locations
        let locations = branchMap[i].locations;
        for (let location of locations) {
            let locationStart = consumer.originalPositionFor(location.start);
            location.start.line = locationStart.line || 0;
            location.start.column = locationStart.column || 0;
            let locationEnd = consumer.originalPositionFor(location.end);
            location.end.line = locationEnd.line || 0;
            location.end.column = locationEnd.column || 0;
        }
        branchMap[i].locations = locations;
        // loc
        let locStart = consumer.originalPositionFor(branchMap[i].loc.start);
        branchMap[i].loc.start.line = locStart.line || 0;
        branchMap[i].loc.start.column = locStart.column || 0;
        let locEnd = consumer.originalPositionFor(branchMap[i].loc.end);
        branchMap[i].loc.end.line = locEnd.line || 0;
        branchMap[i].loc.end.column = locEnd.column || 0;
        // line
        branchMap[i].line = locStart.line || 0;
    }
    etsTempCoverageData[source].branchMap = branchMap;
}
/**
 * 过滤冗余数据
 * @param {coverageMap} etsTempCoverageData
 * @param {coverageMap} etsCoverageData
 * @param {string} etsCoveragePath
 */
function filterCoverage(etsTempCoverageData, etsCoverageData) {
    for (let key of Object.keys(etsTempCoverageData)) {
        // 文件路径设置
        etsCoverageData[key] = etsTempCoverageData[key];
        etsCoverageData[key].path = etsTempCoverageData[key].path;
        etsCoverageData[key].statementMap = etsTempCoverageData[key].statementMap;
        etsCoverageData[key].fnMap = etsTempCoverageData[key].fnMap;
        etsCoverageData[key].s = etsTempCoverageData[key].s;
        etsCoverageData[key].f = etsTempCoverageData[key].f;
        etsCoverageData[key].b = etsTempCoverageData[key].b;
        let statementMap = etsTempCoverageData[key].statementMap;
        let fnMap = etsTempCoverageData[key].fnMap;
        let branchMap = etsTempCoverageData[key].branchMap;
        // 过滤无效得到语句覆盖数据
        let statementCheckMap = {};
        for (let i in statementMap) {
            if (isInvalidStatementNode(statementMap[i]) ||
                isExistStatement(statementMap[i], statementCheckMap)) {
                delete etsCoverageData[key].statementMap[i];
                delete etsCoverageData[key].s[i];
                continue;
            }
            else {
                statementCheckMap[i] = statementMap[i];
            }
        }
        // 过滤无效的函数覆盖数据
        for (let i in fnMap) {
            if (isInvalidFunNode(fnMap[i])) {
                delete etsCoverageData[key].fnMap[i];
                delete etsCoverageData[key].f[i];
                continue;
            }
        }
        // 过滤无效的分支覆盖数据
        for (let i in branchMap) {
            if (branchMap[i].loc.start.line === undefined ||
                branchMap[i].loc.start.line === null) {
                delete etsCoverageData[key].branchMap[i];
                delete etsCoverageData[key].b[i];
                continue;
            }
        }
    }
}
/**
 * 功能：statementMap中元素的开始、结束行为null则为无效数据
 * @param {startEnd} statementMapNode
 * @returns {boolean}
 */
function isInvalidStatementNode(statementMapNode) {
    if (statementMapNode.start.line === undefined ||
        statementMapNode.start.line === null ||
        statementMapNode.end.line === undefined ||
        statementMapNode.end.line === null) {
        return true;
    }
    return false;
}
/**
 * 功能：funMapNode中元素的开始、结束行为null则为无效数据
 * @param {funMapNode} funMapNode
 * @returns {boolean}
 */
function isInvalidFunNode(funMapNode) {
    if (funMapNode.loc.start.line === undefined ||
        funMapNode.loc.start.line === null ||
        funMapNode.loc.end.line === undefined ||
        funMapNode.loc.end.line === null) {
        return true;
    }
    return false;
}
/**
 * 功能：判断statementMap中元素是否重复
 * @param statementMapIndex
 * @param statementCheckMap
 * @returns {boolean}
 */
function isExistStatement(statementMapIndex, statementCheckMap) {
    for (let i in statementCheckMap) {
        if (statementMapIndex.start.line == statementCheckMap[i].start.line) {
            return true;
        }
    }
    return false;
}
