"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArkUIXSdkInfoHandler = exports.getArkUIXMenuLocation = void 0;
const common_sdk_info_handler_1 = require("../../core/api/common-sdk-info-handler");
const simple_progress_1 = require("../../core/log/simple-progress");
const sdk_info_settings_1 = require("../../core/domain/sdk-info-settings");
const sdk_exception_1 = require("../../core/api/sdk-exception");
const error_tip_1 = require("../../core/api/error-tip");
const arkuix_local_component_loader_1 = require("../loader/arkuix-local-component-loader");
const arkuix_remote_component_loader_1 = require("../loader/arkuix-remote-component-loader");
const arkuix_license_manager_1 = require("../license/arkuix-license-manager");
const arkuix_task_factory_1 = require("../task/arkuix-task-factory");
const os_type_util_1 = require("../../core/util/os-type-util");
const getArkUIXMenuLocation = () => {
    if ((0, os_type_util_1.isMac)()) {
        return 'DevEco Studio > Preferences > ArkUI-X';
    }
    return 'File > Settings > ArkUI-X';
};
exports.getArkUIXMenuLocation = getArkUIXMenuLocation;
class ArkUIXSdkInfoHandler extends common_sdk_info_handler_1.CommonSdkInfoHandler {
    constructor(sdkInfoConfig, progress = new simple_progress_1.SimpleProgress()) {
        super(new sdk_info_settings_1.SdkInfoSettings(sdkInfoConfig));
        this._progress = progress;
        this._sdkInfoConfig = sdkInfoConfig;
    }
    getOrDownload(pathAndApiVersions) {
        const loader = new arkuix_local_component_loader_1.ArkUIXLocalComponentLoader(this.sdkSettings.getLocation(), this._progress);
        return this.doGetOrDownload(pathAndApiVersions, this.mapComponents(loader.load()), component => {
            const inCompatibleComponents = loader.getInCompatibleComponents();
            if (inCompatibleComponents.length > 0) {
                throw new sdk_exception_1.SdkException('', new error_tip_1.ErrorTip('', 'The SDK management mode has changed.', `
\t\tMethod 1: Update the SDKs by going to ${(0, exports.getArkUIXMenuLocation)()} and following the onscreen instructions. Open SDK Manager
\t\tMethod 2: Uninstall the SDKs in the following paths:
${inCompatibleComponents.map(component => `\t\t    ${component.getLocation()}`).join('\n')}\n`));
            }
        });
    }
    getLocalComponents() {
        return new arkuix_local_component_loader_1.ArkUIXLocalComponentLoader(this.sdkSettings.getLocation(), this._progress).load();
    }
    getRemoteComponents() {
        return new arkuix_remote_component_loader_1.ArkUIXRemoteComponentLoader(this.sdkSettings, this.sdkSettings.getNetClient(), this._progress).load();
    }
    getProgress() {
        return this._progress;
    }
    findUnacceptedLicenses(components) {
        const licenseIds = components.map(component => component.getLicense());
        return new arkuix_license_manager_1.ArkUIXLicenseManager(this.sdkSettings, this.sdkSettings.getNetClient()).findUnacceptedLicenses(licenseIds, this._progress);
    }
    getLicenseNotAcceptedMessage(fullNameList) {
        const sdkInfo = fullNameList.join(',');
        return new error_tip_1.ErrorTip('', 'The SDK license agreement is not accepted.', `Go to ${(0, exports.getArkUIXMenuLocation)()}, download the ${sdkInfo} SDK, and accept the license agreement. Open SDK Manager`);
    }
    createInstallComponentTask(component) {
        return (0, arkuix_task_factory_1.createArkUIXInstallTask)(component, this.sdkSettings, this.sdkSettings.getNetClient());
    }
    getNotFoundException() {
        return `1.Go to ${(0, exports.getArkUIXMenuLocation)()}, download the components, and sync the project. Open SDK Manager`;
    }
}
exports.ArkUIXSdkInfoHandler = ArkUIXSdkInfoHandler;
