"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractComponentLoader = void 0;
const utils_1 = require("../util/utils");
const component_contants_1 = require("../constants/component-contants");
const path_1 = __importDefault(require("path"));
class AbstractComponentLoader {
    constructor(progress) {
        this.ETS_DISPLAY_NAME = 'ArkTS';
        this.JS_DISPLAY_NAME = 'JS';
        this.progress = progress;
    }
    /**
     * 校验Component
     * @param component
     * @protected
     */
    isValidComponent(component) {
        if ((0, utils_1.isEmpty)(component.getPath())) {
            this.progress.warn('path is empty');
            return false;
        }
        if ((0, utils_1.isEmpty)(component.getDisplayName())) {
            this.progress.warn('displayName is empty');
            return false;
        }
        if ((0, utils_1.isEmpty)(component.getVersion())) {
            this.progress.warn('version is empty');
            return false;
        }
        if (!component_contants_1.VERSION_PATTERN.test(component.getVersion())) {
            this.progress.warn(`Illegal component ${component.getFullName()}, version is incorrect`);
            return false;
        }
        if (component.getApiVersion() === undefined || component.getApiVersion() < 0) {
            this.progress.warn(`Illegal component ${component.getFullName()}, 
        apiVersion should not be less than 0`);
            return false;
        }
        if ((0, utils_1.isEmpty)(component.getLocation())) {
            this.progress.warn(`Illegal component ${component.getFullName()}, location is null`);
            return false;
        }
        return true;
    }
    /**
     * 获取sdk的路径
     * @param component
     * @param sdkRoot
     * @protected
     */
    getLocation(component, sdkRoot) {
        const componentPath = component.path;
        if ((0, utils_1.isEmpty)(componentPath)) {
            return undefined;
        }
        if (component.apiVersion > 0) {
            return path_1.default.resolve(sdkRoot, `${component.apiVersion}`, this.resolveComponentPath(component.path));
        }
        return path_1.default.resolve(sdkRoot, this.resolveComponentPath(componentPath));
    }
    /**
     * 将path属性中的逗号替换成文件分隔符
     * @param componentPath
     * @protected
     */
    resolveComponentPath(componentPath) {
        return componentPath.replace(',', path_1.default.sep);
    }
    addToMap(sdkMap, component) {
        const key = this.genKey(component);
        const oldComponent = sdkMap.get(key);
        if (oldComponent === null || oldComponent === undefined || oldComponent.compareTo(component) < 0) {
            sdkMap.set(key, component);
        }
    }
    genKey(component) {
        if (component.hasApiVersion()) {
            return `${component.getPath()}#${component.getApiVersion()}`;
        }
        else {
            return component.getPath();
        }
    }
    convertDisplayName(component) {
        if (component.getPath() === component_contants_1.ComponentPath.ETS) {
            return this.ETS_DISPLAY_NAME;
        }
        if (component.getPath() === component_contants_1.ComponentPath.JS) {
            return this.JS_DISPLAY_NAME;
        }
        return component.getDisplayName();
    }
}
exports.AbstractComponentLoader = AbstractComponentLoader;
