"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmptyLocationValidator = exports.SystemRootFileValidator = exports.PathWritableValidator = exports.PathLengthValidator = exports.InvalidSlashesValidator = exports.IllegalCharacterValidator = exports.AsciiCharacterValidator = exports.validateSdkPath = void 0;
const path_1 = __importDefault(require("path"));
const os_type_util_1 = require("../util/os-type-util");
const fs_1 = __importDefault(require("fs"));
const file_util_1 = require("../util/file-util");
const utils_1 = require("../util/utils");
const validator_1 = require("./validator");
/**
 * 校验sdk路径合法性
 * @param sdkPath
 */
function validateSdkPath(sdkPath) {
    return validator_1.ValidatorChains.create()
        .addValidator(new EmptyLocationValidator())
        .addValidator(new SystemRootFileValidator())
        .addValidator(new AsciiCharacterValidator())
        .addValidator(new IllegalCharacterValidator())
        .addValidator(new InvalidSlashesValidator())
        .addValidator(new PathLengthValidator())
        .addValidator(new PathWritableValidator())
        .validator(sdkPath);
}
exports.validateSdkPath = validateSdkPath;
/**
 * 校验路径是否包含中文
 */
class AsciiCharacterValidator {
    validator(sdkPath) {
        const fileName = path_1.default.basename(sdkPath);
        if (this.containsNonAscii(fileName)) {
            if ((0, os_type_util_1.isWindows)()) {
                return validator_1.SdkValidation.error('Your sdk path contains non-ASCII characters.');
            }
            else {
                return validator_1.SdkValidation.warn('Your sdk path contains non-ASCII characters, which can cause problems. Proceed with caution');
            }
        }
        return validator_1.SdkValidation.SUCCESS;
    }
    containsNonAscii(string) {
        for (let i = 0; i < string.length; i++) {
            if (string.charAt(i) > '\u007f') {
                return true;
            }
        }
        return false;
    }
}
exports.AsciiCharacterValidator = AsciiCharacterValidator;
/**
 * 校验非法字符，只能是字母、数字、空格以及-_.
 */
class IllegalCharacterValidator {
    validator(sdkPath) {
        let sdkFilePath = sdkPath;
        while (true) {
            if (!IllegalCharacterValidator.LEGAL_FILE_NAME.test(path_1.default.basename(sdkFilePath))) {
                return validator_1.SdkValidation.error('Make sure the SDK path contains only letters, digits, spaces, and special characters: - _ .');
            }
            const parentPath = path_1.default.dirname(sdkFilePath);
            if (path_1.default.dirname(parentPath) === parentPath) {
                break;
            }
            sdkFilePath = parentPath;
        }
        return validator_1.SdkValidation.SUCCESS;
    }
}
exports.IllegalCharacterValidator = IllegalCharacterValidator;
IllegalCharacterValidator.LEGAL_FILE_NAME = /^[a-zA-Z0-9-_.\s]+$/;
/**
 * 校验文件路径分隔符
 */
class InvalidSlashesValidator {
    validator(sdkPath) {
        const normalizePath = path_1.default.resolve(sdkPath).normalize();
        if ((path_1.default.sep === '/' && normalizePath.includes('\\')) || (path_1.default.sep === '\\' && normalizePath.includes('/'))) {
            return validator_1.SdkValidation.error('Your sdk path contains incorrect slashes (\'\\\' vs \'/\')');
        }
        return validator_1.SdkValidation.SUCCESS;
    }
}
exports.InvalidSlashesValidator = InvalidSlashesValidator;
/**
 * 校验路径长度
 */
class PathLengthValidator {
    validator(sdkPath) {
        if ((0, os_type_util_1.isWindows)() && sdkPath.length > PathLengthValidator.WINDOWS_PATH_LENGTH_MAX) {
            return validator_1.SdkValidation.error('The path cannot exceed 100 characters.');
        }
        return validator_1.SdkValidation.SUCCESS;
    }
}
exports.PathLengthValidator = PathLengthValidator;
PathLengthValidator.WINDOWS_PATH_LENGTH_MAX = 100;
/**
 * 校验路径是否有写权限
 */
class PathWritableValidator {
    validator(sdkPath) {
        const parent = path_1.default.dirname(sdkPath);
        if (!fs_1.default.existsSync(sdkPath) && parent !== sdkPath && fs_1.default.existsSync(parent) && !(0, file_util_1.isWritable)(parent)) {
            return validator_1.SdkValidation.error(`The path ${parent} is not writable. Please choose a new location.`);
        }
        if (fs_1.default.existsSync(sdkPath) && !(0, file_util_1.isWritable)(sdkPath)) {
            return validator_1.SdkValidation.error(`The path ${sdkPath} is not writable. Please choose a new location.`);
        }
        return validator_1.SdkValidation.SUCCESS;
    }
}
exports.PathWritableValidator = PathWritableValidator;
/**
 * 校验sdk是否在磁盘根目录下，sdk路径不能为磁盘根目录
 */
class SystemRootFileValidator {
    validator(sdkPath) {
        if (path_1.default.dirname(sdkPath) === sdkPath) {
            return validator_1.SdkValidation.error('The sdk path can not be at the filesystem root');
        }
        return validator_1.SdkValidation.SUCCESS;
    }
}
exports.SystemRootFileValidator = SystemRootFileValidator;
/**
 * 校验sdk路径是否为空
 */
class EmptyLocationValidator {
    validator(sdkPath) {
        if ((0, utils_1.isEmpty)(sdkPath)) {
            return validator_1.SdkValidation.error('Specify the SDK location.');
        }
        return validator_1.SdkValidation.SUCCESS;
    }
}
exports.EmptyLocationValidator = EmptyLocationValidator;
