import 'dart:convert';
import 'package:crypto/crypto.dart';

class InteractiveUrlUtil {
  // 私有构造函数，防止实例化
  InteractiveUrlUtil._();

  static const String RTMP = "rtmp://";
  static const String HTTP = "http://";
  static const String ARTC = "artc://";
  static const String FLV = ".flv";

  ///连麦推拉流地址的二级域名（Second-level domain）
  static const String PULL_SLD = "play";
  static const String PUSH_SLD = "push";

  ///连麦推拉流地址的参数配置key
  static const String SDK_APP_ID = "sdkAppId";
  static const String USER_ID = "userId";
  static const String TIMESTAMP = "timestamp";
  static const String TOKEN = "token";

  static const String APP_NAME = "live";

  ///旁路混流地址用，纯音频--->audio，音视频--->camera
  static const String STREAM_TASK_TYPE_CAMERA = "camera";
  static const String STREAM_TASK_TYPE_AUDIO = "audio";

  ///固定字段
  static const String ALILIVE_INTERACTIVE_DOMAIN = "live.aliyun.com";

  static String _appId = "";
  static String _appKey = "";
  static String _playDomain = "";

  static void setAppInfo(String id, String key, String domain) {
    _appId = id;
    _appKey = key;
    _playDomain = domain;
  }

  static StringBuffer generateProtocol(int type) {
    StringBuffer stringBuffer = StringBuffer();
    if (type == 0) {
      stringBuffer.write(RTMP);
    } else if (type == 1) {
      stringBuffer.write(ARTC);
    } else if (type == 2) {
      stringBuffer.write(HTTP);
    }
    return stringBuffer;
  }

  ///根据 APPID/APPKEY/PLAY_DOMAIN 连麦配置信息，生成连麦推流地址
  ///<p>
  ///需提前配置好以下连麦配置：
  ///{@link AliLiveUserSigGenerate#ALILIVE_APPID}
  ///{@link AliLiveUserSigGenerate#ALILIVE_APPKEY}
  ///{@link AliLiveUserSigGenerate#ALILIVE_PLAY_DOMAIN}
  ///@param channelId 频道 ID
  ///@param userId    用户 ID
  ///@return 连麦推流地址
  static String generateInteractivePushUrl(String channelId, String userId) {
    return generateInteractiveUrl(channelId, userId, true);
  }

  ///根据 APPID/APPKEY/PLAY_DOMAIN 连麦配置信息，生成连麦拉流地址
  ///<p>
  ///需提前配置好以下连麦配置：
  ///{@link AliLiveUserSigGenerate#ALILIVE_APPID}
  ///{@link AliLiveUserSigGenerate#ALILIVE_APPKEY}
  ///{@link AliLiveUserSigGenerate#ALILIVE_PLAY_DOMAIN}///
  ///@param channelId 频道 ID
  ///@param userId    用户 ID
  ///@return 连麦拉流地址
  static String generateInteractivePullUrl(String channelId, String userId) {
    return generateInteractiveUrl(channelId, userId, false);
  }

  static String generateInteractiveUrl(
      String channelId, String userId, bool isPush) {
    String sld = isPush ? PUSH_SLD : PULL_SLD;
    int timestamp = getTimesTamp();
    String token = createToken(_appId, _appKey, channelId, userId, timestamp);

    StringBuffer stringBuffer = StringBuffer(ARTC);
    stringBuffer
      ..write(ALILIVE_INTERACTIVE_DOMAIN)
      ..write("/")
      ..write(sld)
      ..write("/")
      ..write(channelId)
      ..write("?")
      ..write(SDK_APP_ID)
      ..write("=")
      ..write(_appId)
      ..write("&")
      ..write(USER_ID)
      ..write("=")
      ..write(userId)
      ..write("&")
      ..write(TIMESTAMP)
      ..write("=")
      ..write(timestamp)
      ..write("&")
      ..write(TOKEN)
      ..write("=")
      ..write(token);

    return stringBuffer.toString();
  }

  ///根据 APPID/APPKEY/PLAY_DOMAIN 连麦配置信息，生成普通观众的CDN拉流地址
  ///<p>
  ///需提前配置好以下连麦配置：
  ///{@link AliLiveUserSigGenerate#ALILIVE_APPID}
  ///{@link AliLiveUserSigGenerate#ALILIVE_APPKEY}
  ///{@link AliLiveUserSigGenerate#ALILIVE_PLAY_DOMAIN}
  ///@param channelId 频道 ID
  ///@param userId    用户 ID
  ///@return 生成的拉流地址
  static String generateCDNPullUrl(
      String channelId, String userId, bool isAudioOnly) {
    ///建议将rtmp换成http-flv的形式。理由如下：
    ///在阿里云点播控制台生成地址时，会同时生成RTMP与HTTP-FLV的地址，这两个协议里包含的数据内容是一致的，只是网络协议通道不一样。
    ///<p>
    ///HTTP协议是互联网主要协议，CDN、运营商、中间网络设备等链路中都对HTTP有很长时间的网络优化，
    ///HTTP的默认80/443端口号也是常见白名单端口，不容易被禁用，而RTMP协议比较老，其默认端口号是1935有可能被防火墙等设备禁用，导致异常。
    ///因此在综合网络环境下，HTTP-FLV的稳定性、性能（卡顿、延时）会比RTMP更好。
    return generateCDNFlvPullUrl(channelId, userId, isAudioOnly);
  }

  ///HTTP-FLV格式的CDN旁路拉流地址（普通观众用）
  static String generateCDNFlvPullUrl(
      String channelId, String userId, bool isAudioOnly) {
    String streamTaskType =
        isAudioOnly ? STREAM_TASK_TYPE_AUDIO : STREAM_TASK_TYPE_CAMERA;

    StringBuffer stringBuilder = StringBuffer(HTTP);
    stringBuilder
      ..write(_playDomain)
      ..write("/")
      ..write(APP_NAME)
      ..write("/")
      ..write(_appId)
      ..write("_")
      ..write(channelId)
      ..write("_")
      ..write(userId)
      ..write("_")
      ..write(streamTaskType)
      ..write(FLV);
    return stringBuilder.toString();
  }

  ///RTMP格式的CDN旁路拉流地址（普通观众用）
  static String generateCDNRtmpPullUrl(
      String channelId, String userId, bool isAudioOnly) {
    String streamTaskType =
        isAudioOnly ? STREAM_TASK_TYPE_AUDIO : STREAM_TASK_TYPE_CAMERA;

    StringBuffer stringBuffer = StringBuffer(RTMP);
    stringBuffer
      ..write(_playDomain)
      ..write("/")
      ..write(APP_NAME)
      ..write("/")
      ..write(_appId)
      ..write("_")
      ..write(channelId)
      ..write("_")
      ..write(userId)
      ..write("_")
      ..write(streamTaskType);
    return stringBuffer.toString();
  }

  ///根据 appid，appkey，channelId，userId，nonc，timestamp 生层 token
  ///@param appid     应用ID。在控制台应用管理页面创建和查看。
  ///@param appkey    在控制台应用管理页面创建和查看。
  ///@param channelId 频道 ID
  ///@param userId    用户 ID
  ///@param timestamp 过期时间戳
  ///@return token
  static String createToken(String appid, String appkey, String channelId,
      String userId, int timestamp) {
    String stringBuilder =
        appid + appkey + channelId + userId + timestamp.toString();
    return getSHA256(stringBuilder);
  }

  ///字符串签名
  ///@param str 输入源
  ///@return 返回签名
  static String getSHA256(String str) {
    var bytes = utf8.encode(str);
    return sha256.convert(bytes).toString();

    // MessageDigest messageDigest;
    // String encodestr = "";
    // try {
    //   messageDigest = MessageDigest.getInstance("SHA-256");
    //   messageDigest.update(str.getBytes("UTF-8"));
    //   encodestr = byte2Hex(messageDigest.digest());
    // }
    // catch
    // (
    // NoSuchAlgorithmException
    // | UnsupportedEncodingException e) {
    // e.printStackTrace();
    // }
    // return encodestr;
  }

  ///过期时间
  ///时间单位秒，代表令牌有效时间。可设置最大范围是小于等于1天，建议不要设置的过短或超过1天，超过1天会不安全。
  ///默认时间1天。1天 = 60 x  60  x 24。
  static int getTimesTamp() {
    return (DateTime.now().millisecondsSinceEpoch / 1000 + 60 * 60 * 24)
        .round();
  }

// static String byte2Hex(byte[] bytes) {
//   StringBuffer stringBuffer = StringBuffer();
//   String temp;
//   for (byte aByte : bytes) {
//     temp = Integer.toHexString(aByte & 0xFF);
//     if (temp.length() == 1) {
//       stringBuffer.write("0");
//     }
//     stringBuffer.write(temp);
//   }
//   return stringBuffer.toString();
// }
}
