import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter_livepush_demo/manager/push_config_manager.dart';
import 'package:flutter_livepush_demo/page/camera_push/state.dart';
import 'package:flutter_livepush_demo/widget/icon_text.dart';
import 'package:flutter_livepush_demo/widget/camera_push_music_widget.dart';
import 'package:flutter_livepush_demo/widget/camera_push_setting_widget.dart';
import 'package:flutter_livepush_demo/widget/live_push_queen_view.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_config.dart';
import 'package:flutter_livepush_plugin/pusher/live_pusher_preview.dart';
import 'package:redux/redux.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';

import 'actions.dart';

/**
 * Copyright © 2025 Alibaba Cloud. All rights reserved.
 * @author junhuiYe
 * @date 2025/1/23 10:31
 * @brief
 */
class CameraPushPage extends StatefulWidget {
  final Store<CameraPushState> store;
  final ConfigData configData;
  final String pushURLKey;
  final AlivcLivePushConfig pushConfigKey; // 替换为实际配置类型
  final bool pushBeautyOnKey;
  final int pushMode; // 替换为实际类型

  // 构造函数
  CameraPushPage({
    required this.store,
    required this.configData,
    required this.pushURLKey,
    required this.pushConfigKey,
    required this.pushBeautyOnKey,
    required this.pushMode,
  });

  @override
  _CameraPushState createState() => _CameraPushState();
}

class _CameraPushState extends State<CameraPushPage> {
  @override
  void initState() {
    super.initState();
    widget.store.dispatch(CameraPushAction(CameraPushActionType.pushBeautyOnKey,
        payload: widget.pushBeautyOnKey));
    widget.store.dispatch(
        CameraPushAction(CameraPushActionType.initState, payload: context));
    widget.store.dispatch(CameraPushAction(CameraPushActionType.initListener,
        payload: widget.store));
  }

  @override
  Widget build(BuildContext context) {
    return OrientationBuilder(
      builder: (context, orientation) {
        return Scaffold(
          body: WillPopScope(
            child: Stack(
              fit: StackFit.expand,
              children: [
                StreamBuilder<CameraPushState>(
                  stream: widget.store.onChange,
                  initialData: widget.store.state,
                  builder: (context, snapshot) {
                    return _buildPreviewWidget();
                  },
                ),
                StreamBuilder<CameraPushState>(
                  stream: widget.store.onChange,
                  initialData: widget.store.state,
                  builder: (context, snapshot) {
                    return _buildBottomWidget();
                  },
                ),
                StreamBuilder<CameraPushState>(
                  stream: widget.store.onChange,
                  initialData: widget.store.state,
                  builder: (context, snapshot) {
                    return _buildRightWidget();
                  },
                ),
                StreamBuilder<CameraPushState>(
                  stream: widget.store.onChange,
                  initialData: widget.store.state,
                  builder: (context, snapshot) {
                    return _buildTopViewWidget();
                  },
                ),
                StreamBuilder<CameraPushState>(
                  stream: widget.store.onChange,
                  initialData: widget.store.state,
                  builder: (context, snapshot) {
                    return _buildQueenWidget();
                  },
                ),
              ],
            ),
            onWillPop: () async {
              return false;
            },
          ),
          bottomSheet: StreamBuilder<CameraPushState>(
            stream: widget.store.onChange,
            initialData: widget.store.state,
            builder: (context, snapshot) {
              return _buildBottomSheet();
            },
          ),
        );
      },
    );
  }

  Widget _buildPreviewWidget() {
    var x = 0.0;
    var y = 0.0;
    var width = MediaQuery.of(context).size.width;
    var height = MediaQuery.of(context).size.height;

    AlivcPusherPreviewType viewType;
    if (Platform.isAndroid) {
      if (widget.store.state.livePushMode == 0) {
        viewType = AlivcPusherPreviewType.base;
      } else {
        viewType = AlivcPusherPreviewType.push;
      }
    } else {
      viewType = AlivcPusherPreviewType.push;
    }

    AlivcPusherPreview pusherPreviewView = AlivcPusherPreview(
      viewType: viewType,
      onCreated: (id) async {
        await Future.delayed(const Duration(milliseconds: 500));
        widget.store
            .dispatch(CameraPushAction(CameraPushActionType.previewCreated));
      },
      x: x,
      y: y,
      width: width,
      height: height,
    );
    return Positioned(
      child: Container(
          color: Colors.black,
          width: width,
          height: height,
          child: pusherPreviewView),
    );
  }

  Widget _buildRightWidget() {
    var right = MediaQuery.of(context).padding.right;
    var height = MediaQuery.of(context).size.height;
    var childrenItemCount = 4;

    List<Widget> childrenView = [
      StreamBuilder<CameraPushState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return IconText(
            text: AppLocalizations.of(context)!.camerapush_music,
            onIconClick: (isSelected) {
              widget.store.dispatch(CameraPushBottomSheetAction(
                  CameraPushBottomSheetActionType.showMusic,
                  payload: CameraPushPageBottomSheetState.showMusic));
            },
            normalIcon: "assets/images/ic_music.png",
          );
        },
      ),
    ];
    if (widget.pushBeautyOnKey) {
      childrenView.addAll([
        const SizedBox(height: 20.0),
        StreamBuilder<CameraPushState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return StreamBuilder<CameraPushState>(
              stream: widget.store.onChange,
              initialData: widget.store.state,
              builder: (context, snapshot) {
                return IconText(
                  text: AppLocalizations.of(context)!.camerapush_beauty,
                  onIconClick: (isSelected) {
                    if (widget.store.state.showBeautyMenu) {
                      widget.store.dispatch(CameraPushAction(
                          CameraPushActionType.hideBeauty,
                          payload: false));
                    } else {
                      widget.store.dispatch(CameraPushAction(
                          CameraPushActionType.showBeauty,
                          payload: true));
                    }
                  },
                  isSelected: widget.store.state.showBeautyMenu,
                  normalIcon: "assets/images/ic_beauty.png",
                );
              },
            );
          },
        ),
      ]);
      childrenItemCount += 1;
    }
    childrenView.addAll([
      const SizedBox(height: 20.0),
      StreamBuilder<CameraPushState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return IconText(
            text: AppLocalizations.of(context)!.camerapush_flash,
            onIconClick: (isSelected) {
              widget.store.dispatch(CameraPushAction(
                  CameraPushActionType.clickFlashLamp,
                  payload: !isSelected));
            },
            isSelected: widget.store.state.isOpenFlashLamp,
            normalIcon: "assets/images/ic_flash_off.png",
            selectedIcon: "assets/images/ic_flash_on.png",
          );
        },
      ),
      const SizedBox(height: 20.0),
      IconText(
        text: AppLocalizations.of(context)!.camerapush_camera,
        onIconClick: (isSelected) {
          widget.store.dispatch(
              CameraPushAction(CameraPushActionType.clickSwitchCamera));
        },
        normalIcon: "assets/images/ic_camera_on.png",
      ),
      const SizedBox(height: 20.0),
      StreamBuilder<CameraPushState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return IconText(
            text: AppLocalizations.of(context)!.camerapush_snapshot,
            onIconClick: (isSelected) {
              widget.store.dispatch(
                  CameraPushAction(CameraPushActionType.clickSnapShot));
            },
            normalIcon: "assets/images/ic_screenshot.png",
          );
        },
      ),
    ]);

    return Positioned(
      right: right == 0 ? 20.0 : right,
      top: (height - 20.0 * (childrenItemCount - 1) - 45 * childrenItemCount) /
          2.0,
      child: Column(
        children: childrenView,
      ),
    );
  }

  Widget _buildTopViewWidget() {
    var x = 0.0;
    var y = MediaQuery.of(context).padding.top;
    var width = MediaQuery.of(context).size.width;
    var height = 44.0;
    return Positioned(
      left: x,
      top: y,
      width: width,
      height: height,
      child: Container(
          padding: const EdgeInsets.only(left: 8, right: 8),
          width: width,
          height: height,
          child: Row(
            children: [
              IconButton(
                icon: Image.asset(
                  "assets/images/ic_back.png",
                  width: 32,
                  height: 32,
                ),
                onPressed: () {
                  widget.store.dispatch(
                      CameraPushAction(CameraPushActionType.goBackPage));
                },
              ),
              const SizedBox(width: 8),
              Container(
                width: width - 64 * 2,
                child: Text(
                  widget.store.state.pushStatusTip,
                  textAlign: TextAlign.center,
                  style: TextStyle(color: Colors.white, fontSize: 16),
                ),
              ),
            ],
          )),
    );
  }

  Widget _buildBottomWidget() {
    return Positioned(
        bottom: 0.0,
        left: 0.0,
        right: 0.0,
        child: Container(
          color: const Color(0x66FCFCFD),
          child: SingleChildScrollView(
              scrollDirection: Axis.horizontal,
              padding: const EdgeInsets.only(
                  left: 20.0, bottom: 10.0, top: 10.0, right: 20.0),
              child: Row(
                children: [
                  const SizedBox(width: 10.0),
                  StreamBuilder<CameraPushState>(
                    stream: widget.store.onChange,
                    initialData: widget.store.state,
                    builder: (context, snapshot) {
                      return IconText(
                        text: AppLocalizations.of(context)!
                            .camerapush_start_preview,
                        onIconClick: (isSelected) {
                          widget.store.dispatch(CameraPushAction(
                              CameraPushActionType.clickPreview,
                              payload:
                                  CameraPushPagePreviewState.startPreview));
                        },
                        isSelected: widget.store.state.previewState ==
                                CameraPushPagePreviewState.startPreview &&
                            widget.store.state.pushState !=
                                CameraPushPagePushState.pausePush,
                        normalIcon: "assets/images/ic_preview_off.png",
                        selectedIcon: "assets/images/ic_preview_on.png",
                        selectedTextColor: const Color(0xFF4DCFE1),
                      );
                    },
                  ),
                  const SizedBox(width: 20.0),
                  StreamBuilder<CameraPushState>(
                    stream: widget.store.onChange,
                    initialData: widget.store.state,
                    builder: (context, snapshot) {
                      return IconText(
                        text: AppLocalizations.of(context)!
                            .camerapush_start_pushing,
                        onIconClick: (isSelected) {
                          widget.store.dispatch(CameraPushAction(
                              CameraPushActionType.clickePush,
                              payload: widget.pushURLKey));
                        },
                        isSelected: widget.store.state.pushState !=
                            CameraPushPagePushState.stopPush,
                        normalIcon: "assets/images/ic_push_off.png",
                        selectedIcon: "assets/images/ic_push_on.png",
                        selectedTextColor: const Color(0xFF4DCFE1),
                      );
                    },
                  ),
                  const SizedBox(width: 20.0),
                  StreamBuilder<CameraPushState>(
                    stream: widget.store.onChange,
                    initialData: widget.store.state,
                    builder: (context, snapshot) {
                      return IconText(
                        text: AppLocalizations.of(context)!.camerapush_pause,
                        onIconClick: (isSelected) {
                          widget.store.dispatch(CameraPushAction(
                              CameraPushActionType.clickPausePush));
                        },
                        isSelected: widget.store.state.pushState ==
                            CameraPushPagePushState.pausePush,
                        normalIcon: "assets/images/ic_push_pause.png",
                        selectedTextColor: const Color(0xFF4DCFE1),
                      );
                    },
                  ),
                  const SizedBox(width: 20.0),
                  StreamBuilder<CameraPushState>(
                    stream: widget.store.onChange,
                    initialData: widget.store.state,
                    builder: (context, snapshot) {
                      return IconText(
                        text: AppLocalizations.of(context)!
                            .camerapush_restart_pushing,
                        onIconClick: (isSelected) {
                          widget.store.dispatch(CameraPushAction(
                              CameraPushActionType.clickReStartPush));
                        },
                        normalIcon: "assets/images/ic_repush_off.png",
                      );
                    },
                  ),
                  const SizedBox(width: 20.0),
                  StreamBuilder<CameraPushState>(
                    stream: widget.store.onChange,
                    initialData: widget.store.state,
                    builder: (context, snapshot) {
                      return IconText(
                        text: AppLocalizations.of(context)!
                            .camerapush_more_setting,
                        onIconClick: (isSelected) {
                          widget.store.dispatch(CameraPushBottomSheetAction(
                              CameraPushBottomSheetActionType.showMoreSetting,
                              payload: CameraPushPageBottomSheetState
                                  .showMoreSetting));
                        },
                        normalIcon: "assets/images/ic_setting_more_on.png",
                      );
                    },
                  ),
                  const SizedBox(width: 20.0),
                  StreamBuilder<CameraPushState>(
                    stream: widget.store.onChange,
                    initialData: widget.store.state,
                    builder: (context, snapshot) {
                      return IconText(
                        text:
                            AppLocalizations.of(context)!.camerapush_more_data,
                        onIconClick: (isSelected) {},
                        normalIcon: "assets/images/ic_push_info_on.png",
                      );
                    },
                  ),
                ],
              )),
        ));
  }

  Widget _buildBottomSheet() {
    switch (widget.store.state.bottomSheetState) {
      case CameraPushPageBottomSheetState.showMoreSetting:
        return showSetting(context, widget.store.state, widget.store);
      case CameraPushPageBottomSheetState.showMusic:
        return bottomSheetMusic(context, widget.store.state, widget.store);
      case CameraPushPageBottomSheetState.nothing:
        return const SizedBox();
    }
  }

  Widget _buildQueenWidget() {
    if (widget.store.state.showBeautyMenu && Platform.isAndroid) {
      var x = 0.0;
      var y = 0.0;
      var width = MediaQuery.of(context).size.width;
      var height = MediaQuery.of(context).size.height / 3;
      return Positioned(
        left: 0,
        right: 0,
        bottom: 0,
        child: SizedBox(
          width: width,
          height: height,
          child: LivePushQueenView(
            x: x,
            y: y,
            width: width,
            height: height,
            onCreated: (id) {},
          ),
        ),
      );
    } else {
      return const SizedBox();
    }
  }
}
