import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:flutter/services.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/model/music_info.dart';
import 'package:flutter_livepush_demo/page/camera_push/state.dart';
import 'package:flutter_livepush_plugin/base/live_base.dart';
import 'package:flutter_livepush_plugin/base/live_base_def.dart';
import 'package:flutter_livepush_plugin/beauty/live_beauty.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_def.dart';
import 'package:flutter_livepush_plugin/pusher/live_pusher.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';
import 'package:fluttertoast/fluttertoast.dart';

import 'actions.dart';

/**
 * Copyright © 2025 Alibaba Cloud. All rights reserved.
 * @author junhuiYe
 * @date 2025/1/23 10:33
 * @brief
 */
CameraPushState cameraPushReducer(CameraPushState state, action) {
  if (action is CameraPushAction) {
    switch (action.type) {
      case CameraPushActionType.initState:
        state.mContext = action.payload;
        _init(state, action);
        break;
      case CameraPushActionType.initListener:
        _initListener(state, action);
        break;
      case CameraPushActionType.pushBeautyOnKey:
        state.beautyOn = action.payload;
        break;
      case CameraPushActionType.clickPreview:
        _clickPreview(state, action);
        break;
      case CameraPushActionType.previewCreated:
        _previewCreated(state, action);
        break;
      case CameraPushActionType.clickePush:
        _clickPushAction(state, action);
        break;
      case CameraPushActionType.clickPausePush:
        _clickPausePushAction(state, action);
        break;
      case CameraPushActionType.clickReStartPush:
        // TODO: Handle this case.
        break;
      case CameraPushActionType.showBeauty:
        _showBeauty(state, action);
        break;
      case CameraPushActionType.hideBeauty:
        _hindBeauty(state, action);
        break;
      case CameraPushActionType.clickFlashLamp:
        _clickFlashLamp(state, action);
        break;
      case CameraPushActionType.clickSwitchCamera:
        _clickSwitchCamera(state, action);
        break;
      case CameraPushActionType.clickSnapShot:
        _clickSnapShot(state, action);
        break;
      case CameraPushActionType.goBackPage:
        _goBackPage(state, action);
        break;
      case CameraPushActionType.updatePushStatusTip:
        state.pushStatusTip = action.payload;
        break;
    }
  }
  if (action is CameraPushBGMAction) {
    switch (action.type) {
      case CameraPushBGMActionType.changeEars:
        _changeEars(state, action);
        break;
      case CameraPushBGMActionType.changeAudioDenoise:
        _changeAudioDenoise(state, action);
        break;
      case CameraPushBGMActionType.changeMute:
        _changeMute(state, action);
        break;
      case CameraPushBGMActionType.changePlayState:
        _changePlayState(state, action);
        break;
      case CameraPushBGMActionType.changeLoop:
        _changeLoop(state, action);
        break;
      case CameraPushBGMActionType.changeBGMVolume:
        _changeBGMVolume(state, action);
        break;
      case CameraPushBGMActionType.changeCaptureVolume:
        _changeCaptureVolume(state, action);
        break;
      case CameraPushBGMActionType.changeBGMSource:
        _changeBGMSource(state, action);
        break;
      case CameraPushBGMActionType.updateBGMProgress:
        _updateBGMProgress(state, action);
        break;
      case CameraPushBGMActionType.updateBGMDuration:
        _updateBGMDuration(state, action);
        break;
    }
  }
  if (action is CameraPushBottomSheetAction) {
    switch (action.type) {
      case CameraPushBottomSheetActionType.showMusic:
        _showMusic(state, action);
        break;
      case CameraPushBottomSheetActionType.hideMusic:
        _hindMusic(state, action);
        break;
      case CameraPushBottomSheetActionType.showMoreSetting:
        _showMoreSetting(state, action);
        break;
      case CameraPushBottomSheetActionType.hideMoreSetting:
        _hideMoreSetting(state, action);
        break;
    }
  }

  if (action is CameraPushConfigAction) {
    switch (action.type) {
      case CameraPushConfigActionType.changePushMirror:
        _changePushMirror(state, action);
        break;
      case CameraPushConfigActionType.changePreviewMirror:
        _changePreviewMirror(state, action);
        break;
      case CameraPushConfigActionType.setPreviewDisplayMode:
        _setPreviewDisplayMode(state, action);
        break;
      case CameraPushConfigActionType.setTargetVideoBitrate:
        _setTargetVideoBitrate(state, action);
        break;
      case CameraPushConfigActionType.setMinVideoBitrate:
        _setMinVideoBitrate(state, action);
        break;
    }
  }
  return state;
}

late AlivcLiveBase _alivcBase;
late AlivcLivePusher _alivcLivePusher;
late AlivcLiveBeautyManager _beautyManager;

Future<void> _init(CameraPushState state, CameraPushAction action) async {
  _setPageOrientation(state, action);
  AlivcLiveBase.registerSDK();
  _setLivePusher(state, action);
}

Future<void> _setPageOrientation(
    CameraPushState state, CameraPushAction action) async {
  AlivcLivePushOrientation currentOrientation =
      await state.pusherConfig.getOrientation();
  if (currentOrientation == AlivcLivePushOrientation.landscape_home_left) {
    SystemChrome.setPreferredOrientations([
      DeviceOrientation.landscapeLeft,
    ]);
  } else if (currentOrientation ==
      AlivcLivePushOrientation.landscape_home_right) {
    SystemChrome.setPreferredOrientations([
      DeviceOrientation.landscapeRight,
    ]);
  }
}

Future<void> _setLivePusher(
    CameraPushState state, CameraPushAction action) async {
  _alivcLivePusher = AlivcLivePusher.init();
  _alivcLivePusher.initLivePusher();
  _alivcLivePusher.setErrorDelegate();
  _alivcLivePusher.setInfoDelegate();
  _alivcLivePusher.setNetworkDelegate();
  _alivcLivePusher.setCustomFilterDelegate();
  _alivcLivePusher.setCustomDetectorDelegate();
  _alivcLivePusher.setBGMDelegate();
  if (state.beautyOn) {
    _setBeauty(state, action);
  }
}

// 美颜初始化
Future<void> _setBeauty(CameraPushState state, CameraPushAction action) async {
  _alivcLivePusher.useBeauty();
  _beautyManager = AlivcLiveBeautyManager.init();
  String sdkVersion = await AlivcLiveBeautyManager.getQueenSDKVersion();
  print("queen sdk version:${sdkVersion}");
}

Future<void> _previewCreated(
    CameraPushState state, CameraPushAction action) async {
  _alivcLivePusher.startPreview();
  _beautyManager.setupBeauty();
  action.payload.dispatch((CameraPushPagePreviewState.startPreview));
}

// 摄像头切换
Future<void> _clickSwitchCamera(
    CameraPushState state, CameraPushAction action) async {
  _alivcLivePusher.switchCamera();
}

// 闪光灯开关
Future<void> _clickFlashLamp(
    CameraPushState state, CameraPushAction action) async {
  bool flashLamp = action.payload ?? false;
  state.isOpenFlashLamp = flashLamp;
  _alivcLivePusher.setFlash(flashLamp);
}

// 截图
Future<void> _clickSnapShot(
    CameraPushState state, CameraPushAction action) async {
  if (Platform.isIOS) {
    _alivcLivePusher.snapshot(1, 0, "snapshot",
        dirTypeForIOS: AlivcLiveSnapshotDirType.document);
  } else {
    CommonUtil.getSystemPath(DirType.externalFile).then((value) {
      _alivcLivePusher.snapshot(1, 0, value);
    });
  }
  _alivcLivePusher.setSnapshotDelegate();
}

// 美颜显示
Future<void> _showBeauty(CameraPushState state, CameraPushAction action) async {
  if (state.beautyOn) {
    state.showBeautyMenu = action.payload;
    _beautyManager.showPanel();
  }
}

// 美颜隐藏
Future<void> _hindBeauty(CameraPushState state, CameraPushAction action) async {
  if (state.beautyOn) {
    state.showBeautyMenu = action.payload;
    _beautyManager.hidePanel();
  }
}

// 开始推流
Future<void> _clickPushAction(
    CameraPushState state, CameraPushAction action) async {
  String pushURL = action.payload ?? "";
  CameraPushPagePushState pushState = state.pushState;
  if (pushState != CameraPushPagePushState.stopPush) {
    pushState = CameraPushPagePushState.stopPush;
    _alivcLivePusher.stopPush();
  } else {
    pushState = CameraPushPagePushState.startPush;
    _alivcLivePusher.startPushWithURL(pushURL);
  }
  state.pushState = pushState;
}

// 暂停推流
Future<void> _clickPausePushAction(
    CameraPushState state, CameraPushAction action) async {
  CameraPushPagePushState pushState = state.pushState;
  if (pushState != CameraPushPagePushState.pausePush) {
    pushState = CameraPushPagePushState.pausePush;
    _alivcLivePusher.pause();
  } else {
    pushState = CameraPushPagePushState.startPush;
    _alivcLivePusher.resume();
  }
  state.pushState = pushState;
}

// 返回
Future<void> _goBackPage(CameraPushState state, CameraPushAction action) async {
  _destoryPusher(state, action);
  _resetPageOrientation(state, action);
  Navigator.of(state.mContext).pop();
}

Future<void> _destoryPusher(
    CameraPushState state, CameraPushAction action) async {
  _alivcLivePusher.stopPush();
  _alivcLivePusher.stopPreview();
  _alivcLivePusher.destroy();
  if (state.beautyOn) {
    _beautyManager.destroyBeauty();
  }
}

Future<void> _resetPageOrientation(
    CameraPushState state, CameraPushAction action) async {
  AlivcLivePushOrientation currentOrientation =
      await state.pusherConfig.getOrientation();
  if (currentOrientation != AlivcLivePushOrientation.portrait) {
    SystemChrome.setPreferredOrientations([
      DeviceOrientation.portraitUp,
    ]);
  }
}

Future<void> _clickPreview(
    CameraPushState state, CameraPushAction action) async {
  CameraPushPagePreviewState previewState = state.previewState;
  if (previewState == CameraPushPagePreviewState.startPreview) {
    previewState = CameraPushPagePreviewState.stopPreview;
    _alivcLivePusher.stopPreview();
  } else {
    previewState = CameraPushPagePreviewState.startPreview;
    _alivcLivePusher.startPreview();
  }
  state.previewState = previewState;
}

// ----------------- setting panel ---------------
Future<void> _setPreviewDisplayMode(
    CameraPushState state, CameraPushConfigAction action) async {
  AlivcPusherPreviewDisplayMode previewMode =
      AlivcPusherPreviewDisplayMode.values[action.payload ?? 1];
  _alivcLivePusher.setPreviewDisplayMode(previewMode);
  int modeIndex = action.payload ?? 0;
  state.pushSettingState.previewModelIndex = modeIndex;
}

Future<void> _setTargetVideoBitrate(
    CameraPushState state, CameraPushConfigAction action) async {
  if (action.payload != null) {
    int videoBitrate = action.payload ?? 1400;
    _alivcLivePusher.setTargetVideoBitrate(videoBitrate);
    state.pushSettingState.videoTargetBitrate = videoBitrate;
  }
}

Future<void> _setMinVideoBitrate(
    CameraPushState state, CameraPushConfigAction action) async {
  int videoBitrate = action.payload ?? 600;
  _alivcLivePusher.setMinVideoBitrate(videoBitrate);
  state.pushSettingState.videoMinBitrate = videoBitrate;
}

Future<void> _changePushMirror(
    CameraPushState state, CameraPushConfigAction action) async {
  bool isMirror = action.payload ?? false;
  _alivcLivePusher.setPushMirror(isMirror);
  state.pushSettingState.isOpenPushMirror = isMirror;
}

Future<void> _changePreviewMirror(
    CameraPushState state, CameraPushConfigAction action) async {
  bool isMirror = action.payload ?? false;
  _alivcLivePusher.setPreviewMirror(isMirror);
  state.pushSettingState.isOpenPreviewMirror = isMirror;
}

//------------------ show music or setting panel -----------------
Future<void> _showMusic(
    CameraPushState state, CameraPushBottomSheetAction action) async {
  state.bottomSheetState = action.payload;
}

Future<void> _hindMusic(
    CameraPushState state, CameraPushBottomSheetAction action) async {
  state.bottomSheetState = action.payload;
}

Future<void> _showMoreSetting(
    CameraPushState state, CameraPushBottomSheetAction action) async {
  state.bottomSheetState = action.payload;
}

Future<void> _hideMoreSetting(
    CameraPushState state, CameraPushBottomSheetAction action) async {
  state.bottomSheetState = action.payload;
}
//------------------ show music or setting panel end --------------

//------------------ music panel ------------------
Future<void> _changeEars(
    CameraPushState state, CameraPushBGMAction action) async {
  bool enableEars = action.payload ?? false;
  _alivcLivePusher.setBGMEarsBack(enableEars);
  state.bgmState.isOpenBGMEarsBack = enableEars;
}

Future<void> _changeAudioDenoise(
    CameraPushState state, CameraPushBGMAction action) async {
  bool enableAudioDenoise = action.payload ?? false;
  _alivcLivePusher.setAudioDenoise(enableAudioDenoise);
  state.bgmState.isOpenAudioDenoise = enableAudioDenoise;
}

Future<void> _changeMute(
    CameraPushState state, CameraPushBGMAction action) async {
  bool enableMute = action.payload ?? false;
  _alivcLivePusher.setMute(enableMute);
  state.bgmState.isBGMMute = enableMute;
}

Future<void> _updateBGMProgress(
    CameraPushState state, CameraPushBGMAction action) async {
  state.bgmState.bgmProgress = action.payload ?? 0;
}

Future<void> _updateBGMDuration(
    CameraPushState state, CameraPushBGMAction action) async {
  state.bgmState.bgmDuration = action.payload ?? 0;
}

Future<void> _changePlayState(
    CameraPushState state, CameraPushBGMAction action) async {
  bool isPlaying = action.payload ?? false;
  if (isPlaying) {
    _alivcLivePusher.pauseBGM();
  } else {
    _alivcLivePusher.resumeBGM();
  }
  state.bgmState.isBGMPlaying = isPlaying;
}

Future<void> _changeLoop(
    CameraPushState state, CameraPushBGMAction action) async {
  bool enableLoop = action.payload ?? false;
  _alivcLivePusher.setBGMLoop(enableLoop);
  state.bgmState.isBGMLoop = enableLoop;
}

Future<void> _changeBGMVolume(
    CameraPushState state, CameraPushBGMAction action) async {
  int bgmVolume = action.payload ?? 50;
  _alivcLivePusher.setBGMVolume(bgmVolume);
  state.bgmState.bgmVolume = bgmVolume;
}

Future<void> _changeCaptureVolume(
    CameraPushState state, CameraPushBGMAction action) async {
  int captureVolume = action.payload ?? 50;
  _alivcLivePusher.setCaptureVolume(captureVolume);
  state.bgmState.bgmCaptureVolume = captureVolume;
}

Future<void> _changeBGMSource(
    CameraPushState state, CameraPushBGMAction action) async {
  print("BGMSOURCE");
  MusicInfo musicInfo = action.payload;

  String path = musicInfo.path;
  if (path.isEmpty) {
    _alivcLivePusher.stopBGMAsync();
  } else if (path.startsWith("http://") || path.startsWith("https://")) {
    _alivcLivePusher.startBGMWithMusicPathAsync(musicInfo.path);
  } else {
    DirType dirType;
    if (Platform.isAndroid) {
      dirType = DirType.externalFile;
    } else {
      dirType = DirType.document;
    }
    var split = musicInfo.path.split("/");
    CommonUtil.getSaveDir(dirType, split[2]).then((value) {
      musicInfo.path = value.path;
      _alivcLivePusher.startBGMWithMusicPathAsync(musicInfo.path);
    });
  }
  state.bgmState.bgmSelectedIndex = musicInfo.index;
}
//------------------ music panel end --------------

Future<void> _initListener(
    CameraPushState state, CameraPushAction action) async {
  AlivcLiveBase.setListener(AlivcLiveBaseListener(
    onLicenceCheck: (AlivcLiveLicenseCheckResultCode result, String reason) {
      if (result != AlivcLiveLicenseCheckResultCode.success) {
        Fluttertoast.showToast(msg: reason, gravity: ToastGravity.CENTER);
        print("[MSG: $reason]");
      }
    },
  ));
  _alivcLivePusher.setOnSDKError((errorCode, errorDescription) {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!.camerapush_sdk_error,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 系统错误回调
  _alivcLivePusher.setOnSystemError((errorCode, errorDescription) {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!.camerapush_system_error,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 开始预览回调
  _alivcLivePusher.setOnPreviewStarted(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_start_preview_log));
  });

  /// 停止预览回调
  _alivcLivePusher.setOnPreviewStoped(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_stop_preview_log));
  });

  /// 渲染第一帧回调
  _alivcLivePusher.setOnFirstFramePreviewed(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_first_frame_log));
  });

  /// 推流开始回调
  _alivcLivePusher.setOnPushStarted(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_start_push_log));
  });

  /// 摄像头推流暂停回调
  _alivcLivePusher.setOnPushPaused(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_pause_push_log));
  });

  /// 摄像头推流恢复回调
  _alivcLivePusher.setOnPushResumed(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_resume_push_log));
  });

  /// 重新推流回调
  _alivcLivePusher.setOnPushRestart(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_restart_push_log));
  });

  /// 推流停止回调
  _alivcLivePusher.setOnPushStoped(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_stop_push_log));
  });

  /// 推流链接失败
  _alivcLivePusher.setOnConnectFail((errorCode, errorDescription) {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!.camerapush_connect_failed_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 网络恢复
  _alivcLivePusher.setOnConnectRecovery(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload: AppLocalizations.of(state.mContext)!
            .camerapush_connect_recovery_log));
  });

  /// 连接被断开
  _alivcLivePusher.setOnConnectionLost(() {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!.camerapush_connection_lost_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 网络差回调
  _alivcLivePusher.setOnNetworkPoor(() {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!.camerapush_network_slow_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 重连失败回调
  _alivcLivePusher.setOnReconnectError((errorCode, errorDescription) {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!.camerapush_reconnect_fail_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 重连开始回调
  _alivcLivePusher.setOnReconnectStart(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload: AppLocalizations.of(state.mContext)!
            .camerapush_reconnect_start_log));
  });

  /// 重连成功回调
  _alivcLivePusher.setOnReconnectSuccess(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload: AppLocalizations.of(state.mContext)!
            .camerapush_reconnect_success_log));
  });

  /// 发送数据超时
  _alivcLivePusher.setOnSendDataTimeout(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload: AppLocalizations.of(state.mContext)!
            .camerapush_send_data_timeout_log));
  });

  /// 背景音乐播放完毕
  _alivcLivePusher.setOnBGMCompleted(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload: AppLocalizations.of(state.mContext)!
            .camerapush_bgm_play_complete_log));
  });

  /// 背景音乐下载播放超时
  _alivcLivePusher.setOnBGMDownloadTimeout(() {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!
          .camerapush_bgm_download_timeout_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 背景音乐开启失败
  _alivcLivePusher.setOnBGMOpenFailed(() {
    Fluttertoast.showToast(
      msg: AppLocalizations.of(state.mContext)!
          .camerapush_bgm_file_open_failed_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 背景音乐暂停播放
  _alivcLivePusher.setOnBGMPaused(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_bgm_pause_log));
  });

  /// 背景音乐当前播放进度
  _alivcLivePusher.setOnBGMProgress((progress, duration) {
    action.payload.dispatch(CameraPushBGMAction(
        CameraPushBGMActionType.updateBGMDuration,
        payload: duration));
    action.payload.dispatch(CameraPushBGMAction(
        CameraPushBGMActionType.updateBGMProgress,
        payload: progress));
  });

  /// 背景音乐恢复播放
  _alivcLivePusher.setOnBGMResumed(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_bgm_resume_log));
  });

  /// 背景音乐开始播放
  _alivcLivePusher.setOnBGMStarted(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload:
            AppLocalizations.of(state.mContext)!.camerapush_bgm_start_log));
  });

  /// 背景音乐停止播放
  _alivcLivePusher.setOnBGMStoped(() {
    action.payload.dispatch(CameraPushAction(
        CameraPushActionType.updatePushStatusTip,
        payload: AppLocalizations.of(state.mContext)!.camerapush_bgm_stop_log));

    action.payload.dispatch(CameraPushBGMAction(
        CameraPushBGMActionType.updateBGMDuration,
        payload: 0));
    action.payload.dispatch(CameraPushBGMAction(
        CameraPushBGMActionType.updateBGMProgress,
        payload: 0));
  });

  /// 截图回调
  _alivcLivePusher.setOnSnapshot((saveResult, savePath, {dirTypeForIOS}) {
    if (saveResult == true) {
      String tip = AppLocalizations.of(state.mContext)!.camerapush_snapshot_tip;
      if (Platform.isIOS) {
        DirType saveDirType = DirType.document;
        if (dirTypeForIOS == AlivcLiveSnapshotDirType.document) {
          saveDirType = DirType.document;
        } else {
          saveDirType = DirType.library;
        }
        CommonUtil.getSaveDir(saveDirType, savePath).then((value) {
          Fluttertoast.showToast(
              msg: tip + value.path, gravity: ToastGravity.CENTER);
        });
      } else {
        Fluttertoast.showToast(
            msg: tip + savePath, gravity: ToastGravity.CENTER);
      }
    }
  });
}
