// Copyright © 2025 Alibaba Cloud. All rights reserved.
//
// Author: keria
// Date: 2025/1/15
// Brief:

import 'package:flutter/cupertino.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/constant/biz_constants.dart';
import 'package:flutter_livepush_demo/page/interactive_enrollment/middleware.dart';
import 'package:flutter_livepush_demo/page/interactive_enrollment/page.dart';
import 'package:flutter_livepush_demo/page/interactive_enrollment/reducer.dart';
import 'package:flutter_livepush_demo/page/interactive_enrollment/state.dart';
import 'package:flutter_livepush_demo/page/interactive_info/middleware.dart';
import 'package:flutter_livepush_demo/page/interactive_info/page.dart';
import 'package:flutter_livepush_demo/page/interactive_info/reducer.dart';
import 'package:flutter_livepush_demo/page/interactive_info/state.dart';
import 'package:flutter_livepush_demo/page/push_config/middleware.dart';
import 'package:flutter_livepush_demo/page/push_config/page.dart';
import 'package:flutter_livepush_demo/page/push_config/reducer.dart';
import 'package:flutter_livepush_demo/page/push_config/state.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:redux/redux.dart';
import 'package:url_launcher/url_launcher.dart';

import 'actions.dart';
import 'state.dart';

// reducer.dart
HomeState homeReducer(HomeState state, action) {
  print("homeReducer: state=$state, action=$action");
  if (action is HomeAction) {
    switch (action.type) {
      case HomeActionType.getSDKVersion:
        state.sdkVersion = action.payload;
        break;
      case HomeActionType.clickPrivacyPolicy:
        _clickPrivacyPolicy();
        break;
      case HomeActionType.clickCameraPush:
        _clickCameraPushAction(action);
        break;
      case HomeActionType.clickInteractiveLive:
        _clickInteractiveLiveAction(state, action);
        break;
      case HomeActionType.clickInteractivePK:
        _clickInteractivePKAction(state, action);
        break;
      default:
        break;
    }
  }
  return state;
}

Future<void> _clickPrivacyPolicy() async {
  final Uri url = Uri.parse(BizConstants.kAliyunPrivacyServicePolicyUrl);
  launchUrl(url);
}

// 摄像头推流
Future<void> _clickCameraPushAction(HomeAction action) async {
  bool hasPermission = await _checkPermission();
  if (hasPermission) {
    Navigator.of(action.payload).push(
      CupertinoPageRoute(
          builder: (_) => PushConfigPage(
                  store: Store<ConfigState>(
                pushConfigReducer,
                initialState: ConfigState(),
                middleware: createPushConfigMiddlewares(),
              ))),
    );
  }
}

Future<void> _clickInteractiveLiveAction(
    HomeState state, HomeAction action) async {
  jumpToInteractivePage(InteractiveTypeEnum.interactiveLive, state, action);
}

Future<void> _clickInteractivePKAction(
    HomeState state, HomeAction action) async {
  jumpToInteractivePage(InteractiveTypeEnum.interactivePK, state, action);
}

void jumpToInteractivePage(
    InteractiveTypeEnum type, HomeState state, HomeAction action) async {
  bool hasPermission = await _checkPermission();
  if (hasPermission) {
    Map<String, dynamic>? checkResult = _checkAppInfo();
    if (checkResult != null) {
      //跳转页面时，map 增加当前类型，用于判断是连麦互动还是PK互动。
      checkResult[interactiveTypeKey] = type.name;
      Navigator.of(action.payload).push(
        CupertinoPageRoute(
            builder: (_) => InteractiveEnrollmentPage(
                  store: Store<InteractiveEnrollmentState>(
                    interactiveEnrollmentReducer,
                    initialState: InteractiveEnrollmentState(),
                    middleware: createInteractiveEnrollmentMiddleware(),
                  ),
                  interactiveAppMap: checkResult,
                )),
      );
    } else {
      Navigator.of(action.payload).push(
        CupertinoPageRoute(
            builder: (_) => InteractiveInfoPage(
                    store: Store<InteractiveInfoState>(
                  interactiveInfoReducer,
                  initialState: InteractiveInfoState(
                      interactiveTypeEnum: type, context: state.mContext),
                  middleware: createInteractiveInfoMiddleware(),
                ))),
      );
    }
  }
}

Map<String, dynamic>? _checkAppInfo() {
  String appId = CommonUtil.getLocal(interactiveAppIdKey) as String? ?? "";
  String appKey = CommonUtil.getLocal(interactiveAppKeyKey) as String? ?? "";
  String playDomain =
      CommonUtil.getLocal(interactivePlayDomainKey) as String? ?? "";
  if (appId.isNotEmpty && appKey.isNotEmpty && playDomain.isNotEmpty) {
    return {
      interactiveAppIdKey: appId,
      interactiveAppKeyKey: appKey,
      interactivePlayDomainKey: playDomain,
    };
  } else {
    return null;
  }
}

Future<bool> _checkPermission() async {
  if (await Permission.contacts.request().isGranted) {
    return true;
  }
  Map<Permission, PermissionStatus> status = await [
    Permission.microphone,
    Permission.camera,
  ].request();

  PermissionStatus checkPermissionStatue = _checkPermissionEnable(status);
  if (checkPermissionStatue == PermissionStatus.permanentlyDenied) {
    openAppSettings();
  }
  return checkPermissionStatue.isGranted;
}

PermissionStatus _checkPermissionEnable(
  Map<Permission, PermissionStatus> status,
) {
  for (Permission permission in status.keys) {
    PermissionStatus permissionStatus =
        status[permission] ?? PermissionStatus.granted;
    if (permissionStatus == PermissionStatus.denied) {
      return PermissionStatus.denied;
    } else if (permissionStatus == PermissionStatus.permanentlyDenied) {
      return PermissionStatus.permanentlyDenied;
    }
  }
  return PermissionStatus.granted;
}
