// Copyright © 2025 Alibaba Cloud. All rights reserved.
//
// Author: keria
// Date: 2025/1/15
// Brief:

import 'package:flutter/cupertino.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/manager/push_config_manager.dart';
import 'package:flutter_livepush_demo/page/interactive_info/middleware.dart';
import 'package:flutter_livepush_demo/page/interactive_info/page.dart';
import 'package:flutter_livepush_demo/page/interactive_info/reducer.dart';
import 'package:flutter_livepush_demo/page/interactive_info/state.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/middleware.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/page.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/reducer.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/state.dart';
import 'package:flutter_livepush_demo/page/push_config/middleware.dart';
import 'package:flutter_livepush_demo/page/push_config/page.dart';
import 'package:flutter_livepush_demo/page/push_config/reducer.dart';
import 'package:flutter_livepush_demo/page/push_config/state.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_config.dart';
import 'package:flutter_livepush_plugin/pusher/live_pusher.dart';
import 'package:redux/redux.dart';
import 'package:flutter_livepush_demo/common/interactive_url_util.dart';

import 'actions.dart';
import 'state.dart';

List<Middleware<InteractiveEnrollmentState>>
    createInteractiveEnrollmentMiddleware() {
  return [
    TypedMiddleware<InteractiveEnrollmentState, InteractiveEnrollmentAction>(
      _handleActions,
    ),
  ];
}

late AlivcLivePushConfig _alivcLivePusherConfig;
late AlivcLivePusher _alivcLivePusher;

void _handleActions(
  Store<InteractiveEnrollmentState> store,
  action,
  NextDispatcher next,
) {
  if (action is InteractiveEnrollmentAction) {
    switch (action.type) {
      case InteractiveEnrollmentActionType.init:
        _init(store);
        break;
      case InteractiveEnrollmentActionType.clickNext:
        _clickNext(store);
        break;
      case InteractiveEnrollmentActionType.clickGoBack:
        _clickGoBack(store);
        break;
      case InteractiveEnrollmentActionType.settingParameters:
        _settingParameters(store);
        break;
      case InteractiveEnrollmentActionType.editAppInfo:
        _editAppInfo(store);
        break;
      // More cases can be added as needed
      default:
        break;
    }
  }
  next(action);
}

Future<void> _init(Store<InteractiveEnrollmentState> store) async {
  ConfigData.resetData();
  _alivcLivePusher = AlivcLivePusher.init();
  _alivcLivePusher.createConfig();
  _alivcLivePusherConfig = AlivcLivePushConfig.init();
}

void _clickNext(Store<InteractiveEnrollmentState> store) {
  final state = store.state;

  var map = {
    interactiveTypeKey: state.interactiveTypeEnum,
    interactiveAppIdKey: state.appId,
    interactiveAppKeyKey: state.appKey,
    interactivePlayDomainKey: state.playDomain,
    interactiveUserIdKey: state.userId,
    interactiveRoomIdKey: state.roomId,
    pushConfigKey: _alivcLivePusherConfig,
    interactiveIdentityKey: state.interactiveIdentityTypeEnum,
    interactiveTypeKey: state.interactiveTypeEnum
  };

  ///设置 appInfo，用于生成
  InteractiveUrlUtil.setAppInfo(state.appId, state.appKey, state.playDomain);

  Navigator.of(store.state.context).push(
    CupertinoPageRoute(
        builder: (_) => InteractiveModePage(
              store: Store<InteractiveModeState>(
                interactiveModeReducer,
                initialState: InteractiveModeState(),
                middleware: createInteractiveModeMiddleware(),
              ),
              interactiveModeMap: map,
            )),
  );
}

void _clickGoBack(Store<InteractiveEnrollmentState> store) {
  Navigator.of(store.state.context).pop();
}

void _settingParameters(Store<InteractiveEnrollmentState> store) {
  Navigator.of(store.state.context).push(
    CupertinoPageRoute(
        builder: (_) => PushConfigPage(
              store: Store<ConfigState>(
                pushConfigReducer,
                initialState: ConfigState(),
                middleware: createPushConfigMiddlewares(),
              ),
              pushConfigPageType: PushConfigPageType.interactive,
              pushConfigOnSettingConfigActionKey:
                  (AlivcLivePushConfig pusherConfig) {
                _alivcLivePusherConfig = pusherConfig;
              },
            )),
  );
}

void _editAppInfo(Store<InteractiveEnrollmentState> store) {
  InteractiveTypeEnum type;
  type = store.state.interactiveTypeEnum ==
          InteractiveTypeEnum.interactiveLive.name
      ? InteractiveTypeEnum.interactiveLive
      : InteractiveTypeEnum.interactivePK;

  var map = {
    interactiveTypeKey: store.state.interactiveTypeEnum,
    interactiveAppIdKey: store.state.appId,
    interactiveAppKeyKey: store.state.appKey,
    interactivePlayDomainKey: store.state.playDomain,
  };
  Navigator.of(store.state.context).push(
    CupertinoPageRoute(
        builder: (_) => InteractiveInfoPage(
              store: Store<InteractiveInfoState>(
                interactiveInfoReducer,
                initialState: InteractiveInfoState(
                    interactiveTypeEnum: type, context: store.state.context),
                middleware: createInteractiveInfoMiddleware(),
              ),
              interactiveInfoMap: map,
            )),
  );
}
