// Copyright © 2025 Alibaba Cloud. All rights reserved.
//
// Author: keria
// Date: 2025/1/15
// Brief:

import 'package:flutter/material.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/widget/interactive_input_view.dart';
import 'package:flutter_livepush_demo/widget/push_button.dart';
import 'package:flutter_livepush_demo/widget/custom_radio_widget.dart';
import 'package:flutter_livepush_demo/widget/interactive_input_appbar.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';
import 'package:redux/redux.dart';

import 'actions.dart';
import 'state.dart';

class InteractiveEnrollmentPage extends StatefulWidget {
  final Store<InteractiveEnrollmentState> store;
  final Map<String, dynamic>? interactiveAppMap;

  InteractiveEnrollmentPage({required this.store, this.interactiveAppMap});

  @override
  _InteractiveEnrollmentState createState() => _InteractiveEnrollmentState();
}

class _InteractiveEnrollmentState extends State<InteractiveEnrollmentPage> {
  @override
  void initState() {
    widget.store.state.context = context;
    widget.store.dispatch(InteractiveEnrollmentActionCreator.init());
    widget.store.state.appId = widget.interactiveAppMap![interactiveAppIdKey];
    widget.store.state.appKey = widget.interactiveAppMap![interactiveAppKeyKey];
    widget.store.state.playDomain =
        widget.interactiveAppMap![interactivePlayDomainKey];
    widget.store.state.interactiveTypeEnum =
        widget.interactiveAppMap![interactiveTypeKey];
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    return StreamBuilder<InteractiveEnrollmentState>(
      stream: widget.store.onChange,
      initialData: widget.store.state,
      builder: (context, snapshot) => _buildView(context),
    );
  }

  Widget _buildView(BuildContext context) {
    String title;
    if (widget.interactiveAppMap![interactiveTypeKey] ==
        InteractiveTypeEnum.interactiveLive.name) {
      title = AppLocalizations.of(context)!.interactive_co_streaming;
    } else {
      title = AppLocalizations.of(context)!.interactive_battles;
    }
    return Scaffold(
      appBar: buildAppBar(
        title,
        leadingClickListener: () {
          widget.store
              .dispatch(InteractiveEnrollmentActionCreator.clickGoBack());
        },
        showAction: true,
        actionTitle: AppLocalizations.of(context)!.config_page_theme,
        actionIconClickListener: () {
          widget.store
              .dispatch(InteractiveEnrollmentActionCreator.settingParameters());
        },
      ),
      body: Container(
        color: Colors.black,
        child: Container(
          margin: const EdgeInsets.only(left: 20, right: 20),
          child: Stack(
            children: [
              StreamBuilder<InteractiveEnrollmentState>(
                stream: widget.store.onChange,
                initialData: widget.store.state,
                builder: (context, snapshot) {
                  return _buildInputView(context);
                },
              ),
              StreamBuilder<InteractiveEnrollmentState>(
                stream: widget.store.onChange,
                initialData: widget.store.state,
                builder: (context, snapshot) {
                  return _buildBottomButton(context);
                },
              ),
            ],
          ),
        ),
      ),
    );
  }

  ///input UserId and RoomId
  Widget _buildInputView(BuildContext context) {
    return SizedBox.expand(
      child: SingleChildScrollView(
        child: Column(
          children: [
            const SizedBox(height: 40),
            InteractiveInputView(
              title: AppLocalizations.of(context)!.interactive_user_id,
              hint:
                  AppLocalizations.of(context)!.interactive_input_number_digits,
              onlyClearIcon: true,
              action: (text) => widget.store.dispatch(
                InteractiveEnrollmentActionCreator.changeUserId(text),
              ),
            ),
            const SizedBox(height: 20),
            InteractiveInputView(
              title: AppLocalizations.of(context)!.interactive_room_id,
              hint:
                  AppLocalizations.of(context)!.interactive_input_number_digits,
              onlyClearIcon: true,
              action: (text) => widget.store.dispatch(
                InteractiveEnrollmentActionCreator.changeRoomId(text),
              ),
            ),
            const SizedBox(height: 20),
            _buildAppInfoWidget(context),
          ],
        ),
      ),
    );
  }

  Widget _buildBottomButton(BuildContext context) {
    return Positioned(
      left: 0,
      right: 0,
      bottom: 20,
      child: StreamBuilder<InteractiveEnrollmentState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return PushButton(
            enable: widget.store.state.userId.isNotEmpty &&
                widget.store.state.roomId.isNotEmpty,
            title: AppLocalizations.of(context)!.interactive_next_button,
            action: () {
              widget.store
                  .dispatch(InteractiveEnrollmentActionCreator.clickNext());
            },
          );
        },
      ),
    );
  }

  Widget _buildAppInfoWidget(BuildContext context) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.end,
      children: [
        Row(
          mainAxisAlignment: MainAxisAlignment.spaceBetween,
          children: [
            Text(
              AppLocalizations.of(context)!.interactive_application_information,
              style: const TextStyle(color: Colors.white),
            ),
            IconButton(
              alignment: Alignment.centerRight,
              padding: const EdgeInsets.all(0.0),
              onPressed: () {
                widget.store.dispatch(
                  InteractiveEnrollmentActionCreator.changeAppInfoState(
                    !widget.store.state.showAppInfo,
                  ),
                );
              },
              icon: Transform.rotate(
                angle: widget.store.state.arrowAngle,
                child: Image.asset(
                  "assets/images/ic_chevron_down.png",
                  color: Colors.white,
                  width: 20,
                  height: 20,
                ),
              ),
            ),
          ],
        ),
        StreamBuilder<InteractiveEnrollmentState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return _showAppInfoView(context);
          },
        ),
        const Divider(color: Colors.white),
        const SizedBox(height: 20),
        StreamBuilder<InteractiveEnrollmentState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return _buildIdentityView(context);
          },
        ),
      ],
    );
  }

  ///app info
  Widget _showAppInfoView(BuildContext context) {
    if (widget.store.state.showAppInfo) {
      Map<String, dynamic>? interactiveAppMap = widget.interactiveAppMap;
      return Column(
        crossAxisAlignment: CrossAxisAlignment.end,
        children: [
          _createAppInfoView(
            AppLocalizations.of(context)!.interactive_app_id,
            interactiveAppMap![interactiveAppIdKey],
          ),
          const SizedBox(height: 10),
          _createAppInfoView(
            AppLocalizations.of(context)!.interactive_app_key,
            interactiveAppMap[interactiveAppKeyKey],
          ),
          const SizedBox(height: 10),
          _createAppInfoView(
            AppLocalizations.of(context)!.interactive_play_domain,
            interactiveAppMap[interactivePlayDomainKey],
          ),
          IconButton(
            alignment: Alignment.centerRight,
            padding: const EdgeInsets.all(0.0),
            onPressed: () {
              widget.store
                  .dispatch(InteractiveEnrollmentActionCreator.editAppInfo());
            },
            icon: Image.asset(
              "assets/images/ic_edit.png",
              width: 18,
              height: 18,
            ),
          ),
        ],
      );
    } else {
      return const SizedBox();
    }
  }

  ///custom widget
  Widget _createAppInfoView(String title, String value) {
    return Row(
      mainAxisAlignment: MainAxisAlignment.spaceBetween,
      children: [
        Text(
          title,
          style: const TextStyle(color: Color(0xFFB2B7C4), fontSize: 12),
        ),
        Text(
          value,
          style: const TextStyle(color: Color(0xFFB2B7C4), fontSize: 12),
        ),
      ],
    );
  }

  ///身份选择 Widget
  Widget _buildIdentityView(BuildContext context) {
    if (widget.interactiveAppMap![interactiveTypeKey] ==
        InteractiveTypeEnum.interactiveLive.name) {
      return Row(
        mainAxisAlignment: MainAxisAlignment.spaceAround,
        children: [
          CustomRadioWidget(
            title:
                AppLocalizations.of(context)!.interactive_co_streaming_streamer,
            value: InteractiveIdentityTypeEnum.anchor.name,
            groupValue: widget.store.state.interactiveIdentityTypeEnum,
            onChanged: (e) {
              widget.store.dispatch(
                InteractiveEnrollmentActionCreator.changeIdentity(e.toString()),
              );
            },
          ),
          CustomRadioWidget(
            title:
                AppLocalizations.of(context)!.interactive_co_streaming_viewers,
            value: InteractiveIdentityTypeEnum.audience.name,
            groupValue: widget.store.state.interactiveIdentityTypeEnum,
            onChanged: (e) {
              widget.store.dispatch(
                InteractiveEnrollmentActionCreator.changeIdentity(e.toString()),
              );
            },
          ),
        ],
      );
    } else {
      return const SizedBox();
    }
  }
}
