import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:flutter_aliplayer/flutter_aliplayer.dart';
import 'package:flutter_aliplayer/flutter_aliplayer_factory.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/common/interactive_url_util.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/actions.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/state.dart';
import 'package:flutter_livepush_plugin/base/live_base.dart';
import 'package:flutter_livepush_plugin/base/live_base_def.dart';
import 'package:flutter_livepush_plugin/player/live_player.dart';
import 'package:flutter_livepush_plugin/player/live_player_config.dart';
import 'package:flutter_livepush_plugin/player/live_player_def.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_def.dart';
import 'package:flutter_livepush_plugin/pusher/live_pusher.dart';
import 'package:flutter_livepush_plugin/pusher/live_transcoding_config.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';
import 'package:redux/redux.dart';

/**
 * Copyright © 2025 Alibaba Cloud. All rights reserved.
 * @author junhuiYe
 * @date 2025/2/11 09:45
 * @brief
 */
List<Middleware<InteractiveModeState>> createInteractiveModeMiddleware() {
  return [
    TypedMiddleware<InteractiveModeState, InteractiveModeAction>(
      _handleActions,
    ),
  ];
}

late AlivcLivePusher _alivcLivePusher;
late AlivcLivePlayer _alivcLivePlayer;
late AlivcLivePlayConfig _alivcLivePlayConfig;
late FlutterAliplayer _aliPlayer;

void _handleActions(
  Store<InteractiveModeState> store,
  action,
  NextDispatcher next,
) {
  if (action is InteractiveModeAction) {
    switch (action.type) {
      case InteractiveModeActionType.init:
        store.state.context = action.payload;
        _init(store);
        break;
      case InteractiveModeActionType.startPreview:
        _startPreview(store);
        break;
      case InteractiveModeActionType.clickMute:
        _clickMute(store, action);
        break;
      case InteractiveModeActionType.startLink:
        _startLink(store);
        break;
      case InteractiveModeActionType.stopLink:
        _stopLink(store);
        break;
      case InteractiveModeActionType.clickSwitchCamera:
        _clickSwitchCamera(store);
        break;
      case InteractiveModeActionType.clickSwitchSpeaker:
        _clickSwitchSpeaker(store);
        break;
      case InteractiveModeActionType.goBackPage:
        _goBackPage(store);
        break;
      // More cases can be added as needed
      default:
        break;
    }
  }
  next(action);
}

Future<void> _init(Store<InteractiveModeState> store) async {
  _registerSDK(store);
  _setRTCPusher(store);
  _setRTCPlayer(store);
  if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name &&
      store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.audience.name) {
    FlutterAliPlayerFactory.initLicenseServiceForIOS();
    _setCDNPlayer(store);
  }

  store.dispatch(InteractiveModeActionCreator.pageDidAppear(true));
}

Future<void> _registerSDK(Store<InteractiveModeState> store) async {
  AlivcLiveBase.registerSDK();
  AlivcLiveBase.setListener(AlivcLiveBaseListener(
    onLicenceCheck: (AlivcLiveLicenseCheckResultCode result, String reason) {
      if (result != AlivcLiveLicenseCheckResultCode.success) {
        Fluttertoast.showToast(msg: reason, gravity: ToastGravity.CENTER);
      }
    },
  ));
}

Future<void> _setRTCPusher(Store<InteractiveModeState> store) async {
  _alivcLivePusher = AlivcLivePusher.init();
  store.state.pusherConfig.setLivePushMode(AlivcLivePushMode.interactive);
  _alivcLivePusher.initLivePusher();
  _alivcLivePusher.setErrorDelegate();
  _alivcLivePusher.setInfoDelegate();
  _alivcLivePusher.setNetworkDelegate();

  /// SDK错误回调
  _alivcLivePusher.setOnSDKError((errorCode, errorDescription) {
    store.dispatch(InteractiveModeActionCreator.updatePushStatus(
        InteractivePushStatus.error));
    Fluttertoast.showToast(
      msg: AppLocalizations.of(store.state.context)!.camerapush_sdk_error,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 系统错误回调
  _alivcLivePusher.setOnSystemError((errorCode, errorDescription) {
    store.dispatch(InteractiveModeActionCreator.updatePushStatus(
        InteractivePushStatus.error));
    Fluttertoast.showToast(
      msg: AppLocalizations.of(store.state.context)!.camerapush_system_error,
      gravity: ToastGravity.CENTER,
    );
    print("[ERROR]-[SYSTEM]");
  });

  /// 开始预览回调
  _alivcLivePusher.setOnPreviewStarted(() {
    print("===== 预览开始 =====");
  });

  /// 停止预览回调
  _alivcLivePusher.setOnPreviewStoped(() {
    print("===== 预览结束 =====");
  });

  /// 渲染第一帧回调
  _alivcLivePusher.setOnFirstFramePreviewed(() {});

  /// 推流开始回调
  _alivcLivePusher.setOnPushStarted(() {
    print("===== 推流开始 =====");
  });

  /// 摄像头推流暂停回调
  _alivcLivePusher.setOnPushPaused(() {});

  /// 摄像头推流恢复回调
  _alivcLivePusher.setOnPushResumed(() {});

  /// 重新推流回调
  _alivcLivePusher.setOnPushRestart(() {});

  /// 推流停止回调
  _alivcLivePusher.setOnPushStoped(() {
    print("===== 推流停止 =====");
  });

  /// 推流链接失败
  _alivcLivePusher.setOnConnectFail((errorCode, errorDescription) {
    store.dispatch(InteractiveModeActionCreator.updatePushStatus(
        InteractivePushStatus.error));
    print("===== 推流连接失败 =====");
    Fluttertoast.showToast(
      msg: AppLocalizations.of(store.state.context)!
          .camerapush_connect_failed_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 网络恢复
  _alivcLivePusher.setOnConnectRecovery(() {});

  /// 连接被断开
  _alivcLivePusher.setOnConnectionLost(() {
    store.dispatch(InteractiveModeActionCreator.updatePushStatus(
        InteractivePushStatus.error));
    Fluttertoast.showToast(
      msg: AppLocalizations.of(store.state.context)!
          .camerapush_connection_lost_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 网络差回调
  _alivcLivePusher.setOnNetworkPoor(() {
    Fluttertoast.showToast(
      msg:
          AppLocalizations.of(store.state.context)!.camerapush_network_slow_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 重连失败回调
  _alivcLivePusher.setOnReconnectError((errorCode, errorDescription) {
    store.dispatch(InteractiveModeActionCreator.updatePushStatus(
        InteractivePushStatus.error));
    Fluttertoast.showToast(
      msg: AppLocalizations.of(store.state.context)!
          .camerapush_reconnect_fail_log,
      gravity: ToastGravity.CENTER,
    );
  });

  /// 重连开始回调
  _alivcLivePusher.setOnReconnectStart(() {});

  /// 重连成功回调
  _alivcLivePusher.setOnReconnectSuccess(() {});

  /// 发送数据超时
  _alivcLivePusher.setOnSendDataTimeout(() {
    store.dispatch(InteractiveModeActionCreator.updatePushStatus(
        InteractivePushStatus.error));
    Fluttertoast.showToast(
      msg: AppLocalizations.of(store.state.context)!
          .camerapush_send_data_timeout_log,
      gravity: ToastGravity.CENTER,
    );
  });
}

Future<void> _setRTCPlayer(Store<InteractiveModeState> store) async {
  _alivcLivePlayer = AlivcLivePlayer.init();
  _alivcLivePlayer.bindPlayConfig();
  _alivcLivePlayConfig = AlivcLivePlayConfig.init();
  _alivcLivePlayConfig.setRenderMode(AlivcLivePlayRenderMode.auto);

  _alivcLivePlayer.initLivePlayer();
  _alivcLivePlayer.setLivePlayerDelegate();

  _alivcLivePlayer.setOnError((errorCode, errorMessage) {
    if (errorCode == AlivcLivePlayerError.streamNotFound) {
      _stopRTCPlay(store);
      if (store.state.identityTypeEnum !=
          InteractiveIdentityTypeEnum.audience.name) {
        store.dispatch(InteractiveModeActionCreator.clickBtnLink(false));
      }
      Fluttertoast.showToast(
          msg: AppLocalizations.of(store.state.context)!
              .interactive_not_pull_stream,
          gravity: ToastGravity.CENTER);
    } else if (errorCode == AlivcLivePlayerError.streamStopped) {
      _stopRTCPlay(store);
      if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name) {
        if (store.state.identityTypeEnum ==
            InteractiveIdentityTypeEnum.anchor.name) {
          store.dispatch(InteractiveModeActionCreator.clickBtnLink(false));
          Fluttertoast.showToast(
              msg: AppLocalizations.of(store.state.context)!
                  .interactive_viewer_left,
              gravity: ToastGravity.CENTER);
        } else {
          _stopRTCPush(store);
          store.dispatch(InteractiveModeActionCreator.clickBtnLink(false));
          Fluttertoast.showToast(
              msg: AppLocalizations.of(store.state.context)!
                  .interactive_streamer_left,
              gravity: ToastGravity.CENTER);
          _goBackPage(store);
        }
      } else {
        store.dispatch(InteractiveModeActionCreator.clickBtnLink(false));
        Fluttertoast.showToast(
            msg: AppLocalizations.of(store.state.context)!
                .interactive_streamer_left,
            gravity: ToastGravity.CENTER);
      }
    } else {
      store.dispatch(InteractiveModeActionCreator.updatePullStatus(
          InteractivePullStatus.error));
      Fluttertoast.showToast(msg: errorMessage, gravity: ToastGravity.CENTER);
    }
  });
  _alivcLivePlayer.setOnPlayStarted(() {});
  _alivcLivePlayer.setOnPlayStopped(() {});
  _alivcLivePlayer.setOnFirstVideoFrameDrawn(() {});
}

Future<void> _setCDNPlayer(Store<InteractiveModeState> store) async {
  _aliPlayer = FlutterAliPlayerFactory.createAliPlayer();
  _aliPlayer.setAutoPlay(true);

  _aliPlayer.setOnCompletion((playerId) {});
  _aliPlayer.setOnError((errorCode, errorExtra, errorMsg, playerId) async {
    String errMsg = "Pull Play Error:[${errorCode}]${errorMsg}";
    print(errorMsg);
    Fluttertoast.showToast(msg: errMsg, gravity: ToastGravity.CENTER);
    store.dispatch(InteractiveModeActionCreator.updatePullStatus(
        InteractivePullStatus.error));
    _aliPlayer.stop();
    _aliPlayer.clearScreen();
    await Future.delayed(Duration(milliseconds: 2000));
    _aliPlayer.prepare();
  });
}

Future<void> _startPreview(Store<InteractiveModeState> store) async {
  if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name &&
      store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.audience.name) {
    _startCDNPull(store);
  } else {
    _startRTCPush(store);
  }
}

Future<void> _startLink(Store<InteractiveModeState> store) async {
  store.dispatch(InteractiveModeActionCreator.clickBtnLink(true));
  if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name &&
      store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.audience.name) {
    _stopCDNPull(store);
    await Future.delayed(const Duration(milliseconds: 200));
    _startRTCPush(store);
    await Future.delayed(const Duration(milliseconds: 100));
    _startRTCPlay(store);
  } else {
    await Future.delayed(const Duration(milliseconds: 500));
    _startRTCPlay(store);
  }
}

Future<void> _stopLink(Store<InteractiveModeState> store) async {
  store.dispatch(InteractiveModeActionCreator.clickBtnLink(false));
  if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name &&
      store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.audience.name) {
    _stopRTCPush(store);

    await Future.delayed(Duration(milliseconds: 500));
    _stopRTCPlay(store);

    await Future.delayed(Duration(milliseconds: 500));
    _startCDNPull(store);
  } else {
    _stopRTCPlay(store);
  }
}

Future<void> _clickSwitchCamera(Store<InteractiveModeState> store) async {
  _alivcLivePusher.switchCamera();
}

Future<void> _clickSwitchSpeaker(Store<InteractiveModeState> store) async {
  bool enable = await _alivcLivePusher.isEnableSpeakerphone();
  _alivcLivePusher.enableSpeakerphone(enable ? false : true);
}

Future<void> _stopRTCPlay(Store<InteractiveModeState> store) async {
  _alivcLivePlayer.stopPlay();

  if (store.state.identityTypeEnum !=
      InteractiveIdentityTypeEnum.audience.name) {
    _setMixStream(store, false);
  }

  store.dispatch(InteractiveModeActionCreator.updatePullStatus(
      InteractivePullStatus.stop));
}

Future<void> _stopRTCPush(Store<InteractiveModeState> store) async {
  _alivcLivePusher.stopPush();
  _alivcLivePusher.stopPreview();
  store.dispatch(InteractiveModeActionCreator.updatePushStatus(
      InteractivePushStatus.stop));
}

Future<void> _startRTCPlay(Store<InteractiveModeState> store) async {
  String rtcPlayURL = InteractiveUrlUtil.generateInteractivePullUrl(
      store.state.customerInfo.roomId, store.state.customerInfo.userId);
  print("rtcPlayURL:$rtcPlayURL}");
  if (rtcPlayURL.isEmpty) {
    Fluttertoast.showToast(
        msg: AppLocalizations.of(store.state.context)!
            .interactive_input_pull_stream_url,
        gravity: ToastGravity.CENTER);
    return;
  }

  if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name) {
    if (Platform.isAndroid) {
      _alivcLivePlayConfig.setIsFullScreen(store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.audience.name);
    }
  }

  _alivcLivePlayer.setPlayView();

  _alivcLivePlayer.startPlayWithURL(rtcPlayURL);
  if (store.state.identityTypeEnum !=
      InteractiveIdentityTypeEnum.audience.name) {
    _setMixStream(store, true);
  }

  store.dispatch(InteractiveModeActionCreator.updatePullStatus(
      InteractivePullStatus.pulling));
}

Future<void> _startRTCPush(Store<InteractiveModeState> store) async {
  String rtcPushURL = InteractiveUrlUtil.generateInteractivePushUrl(
      store.state.roomId, store.state.userId);
  print("rtcPushURL:${rtcPushURL}");
  if (rtcPushURL.isEmpty) {
    Fluttertoast.showToast(
        msg: "Start Push Error", gravity: ToastGravity.CENTER);
    return;
  }

  _alivcLivePusher.startPreview();

  _alivcLivePusher.startPushWithURL(rtcPushURL);

  store.dispatch(InteractiveModeActionCreator.updatePushStatus(
      InteractivePushStatus.pushing));
}

Future<void> _setMixStream(
    Store<InteractiveModeState> store, bool isNeed) async {
  if (isNeed) {
    AlivcLiveTranscodingConfig _liveTranscodingConfig =
        AlivcLiveTranscodingConfig.init();
    _alivcLivePusher.bindLiveMixTranscodingConfig();

    Size pushResolution = await store.state.pusherConfig.getPushResolution();

    AlivcLiveMixStream pusherMixStream = AlivcLiveMixStream();
    pusherMixStream.userId = store.state.userId;
    pusherMixStream.x = 0;

    if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name) {
      pusherMixStream.y = 0;
      pusherMixStream.width = pushResolution.width.round();
      pusherMixStream.height = pushResolution.height.round();
    } else {
      double pushMixStreamWidth = pushResolution.width / 2.0;
      double pushMixStreamHeight =
          (pushMixStreamWidth * pushResolution.height) / pushResolution.width;
      pusherMixStream.y =
          ((pushResolution.height - pushMixStreamHeight) / 2.0).round();
      pusherMixStream.width = pushMixStreamWidth.round();
      pusherMixStream.height = pushMixStreamHeight.round();
    }

    pusherMixStream.zOrder = 1;

    print(
        "pusherMixStream userId:${pusherMixStream.userId}, x:${pusherMixStream.x}, y:${pusherMixStream.y}, width:${pusherMixStream.width}, height:${pusherMixStream.height}");

    AlivcLiveMixStream pullerMixStream = AlivcLiveMixStream();
    pullerMixStream.userId = store.state.customerInfo.userId;

    if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name) {
      int pullerMixStreamWidth = 90;
      int pullerMixStreamHeight = 160;
      pullerMixStream.x =
          (pushResolution.width - pullerMixStreamWidth - 35).round();
      pullerMixStream.y =
          (pushResolution.height - pullerMixStreamHeight - 35).round();
      pullerMixStream.width = 90;
      pullerMixStream.height = 160;
    } else {
      double pusherMixStreamWidth = pushResolution.width / 2.0;
      double pusherMixStreamHeight =
          (pusherMixStreamWidth * pushResolution.height) / pushResolution.width;

      pullerMixStream.x = (pusherMixStreamWidth + 1).round();
      pullerMixStream.y =
          ((pushResolution.height - pusherMixStreamHeight) / 2.0).round();
      pullerMixStream.width = pusherMixStreamWidth.round();
      pullerMixStream.height = pusherMixStreamHeight.round();
    }

    pullerMixStream.zOrder = 2;
    print(
        "pullerMixStream userId:${pullerMixStream.userId}, x:${pullerMixStream.x}, y:${pullerMixStream.y}, width:${pullerMixStream.width}, height:${pullerMixStream.height}");
    _liveTranscodingConfig.setMixStreams([pusherMixStream, pullerMixStream]);
  }
  _alivcLivePusher.setLiveMixTranscodingConfig(isNeed);
}

Future<void> _startCDNPull(Store<InteractiveModeState> store) async {
  bool audioOnly = await store.state.pusherConfig.getAudioOnly();
  String cdnPlayURL = InteractiveUrlUtil.generateCDNPullUrl(
      store.state.customerInfo.roomId,
      store.state.customerInfo.userId,
      audioOnly);
  print("cdnPlayURL:${cdnPlayURL}");
  if (cdnPlayURL.isEmpty) {
    Fluttertoast.showToast(
        msg: AppLocalizations.of(store.state.context)!
            .interactive_input_pull_stream_url,
        gravity: ToastGravity.CENTER);
    return;
  }

  _aliPlayer.setPlayerView(store.state.cdnPlayerViewId);
  _aliPlayer.setUrl(cdnPlayURL);
  _aliPlayer.prepare();

  store.dispatch(InteractiveModeActionCreator.updatePullStatus(
      InteractivePullStatus.pulling));
}

Future<void> _stopCDNPull(Store<InteractiveModeState> store) async {
  _aliPlayer.stop();
  _aliPlayer.clearScreen();

  store.dispatch(InteractiveModeActionCreator.updatePullStatus(
      InteractivePullStatus.stop));
}

Future<void> _clickMute(
    Store<InteractiveModeState> store, InteractiveModeAction action) async {
  bool isMute = action.payload ?? false;
  _alivcLivePusher.setMute(isMute);
  store.state.mute = action.payload;
}

Future<void> _goBackPage(Store<InteractiveModeState> store) async {
  _destory(store);
  Navigator.of(store.state.context).pop();
}

Future<void> _destory(Store<InteractiveModeState> store) async {
  _alivcLivePusher.destroy();

  _alivcLivePlayer.stopPlay();

  if (store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name &&
      store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.audience.name) {
    _aliPlayer.stop();
    _aliPlayer.destroy();
  }
}
