import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter_aliplayer/flutter_aliplayer.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/actions.dart';
import 'package:flutter_livepush_demo/page/interactive_mode/state.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';
import 'package:flutter_livepush_demo/widget/common_view.dart';
import 'package:flutter_livepush_demo/widget/push_button.dart';
import 'package:flutter_livepush_demo/widget/interactive_meeting_info_dialog.dart';
import 'package:flutter_livepush_plugin/pusher/live_pusher_preview.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:redux/redux.dart';

/**
 * Copyright © 2025 Alibaba Cloud. All rights reserved.
 * @author junhuiYe
 * @date 2025/2/11 09:45
 * @brief
 */
class InteractiveModePage extends StatefulWidget {
  final Store<InteractiveModeState> store;
  final Map<String, dynamic>? interactiveModeMap;

  // 构造函数
  InteractiveModePage({required this.store, this.interactiveModeMap});

  @override
  _InteractiveModeState createState() => _InteractiveModeState();
}

class _InteractiveModeState extends State<InteractiveModePage> {
  @override
  void initState() {
    super.initState();
    widget.store.state.appId = widget.interactiveModeMap![interactiveAppIdKey];
    widget.store.state.appKey =
        widget.interactiveModeMap![interactiveAppKeyKey];
    widget.store.state.playDomain =
        widget.interactiveModeMap![interactivePlayDomainKey];
    widget.store.state.userId =
        widget.interactiveModeMap![interactiveUserIdKey];
    widget.store.state.roomId =
        widget.interactiveModeMap![interactiveRoomIdKey];
    widget.store.state.typeEnum =
        widget.interactiveModeMap![interactiveTypeKey];
    widget.store.state.identityTypeEnum =
        widget.interactiveModeMap![interactiveIdentityKey];
    widget.store.state.pusherConfig = widget.interactiveModeMap![pushConfigKey];
    widget.store.dispatch(InteractiveModeActionCreator.init(context));
  }

  @override
  Widget build(BuildContext context) {
    Widget contentWidget;
    if (widget.store.state.typeEnum ==
        InteractiveTypeEnum.interactiveLive.name) {
      contentWidget = StreamBuilder<InteractiveModeState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) => _buildInteractiveWidget(),
      );
    } else {
      contentWidget = StreamBuilder<InteractiveModeState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) => _buildInteractivePKWidget(),
      );
    }
    return Scaffold(
      body: WillPopScope(
        child: Container(
          color: Colors.black,
          child: Stack(
            fit: StackFit.expand,
            children: [
              contentWidget,
              StreamBuilder<InteractiveModeState>(
                stream: widget.store.onChange,
                initialData: widget.store.state,
                builder: (context, snapshot) {
                  return _buildTitleWidget();
                },
              ),
              StreamBuilder<InteractiveModeState>(
                stream: widget.store.onChange,
                initialData: widget.store.state,
                builder: (context, snapshot) {
                  return _buildFunctionWidget();
                },
              ),
            ],
          ),
        ),
        onWillPop: () async {
          return false;
        },
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  /// 顶部控件
  Widget _buildTitleWidget() {
    return Positioned(
      top: 44,
      left: 10,
      right: 10,
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          IconButton(
            icon: Image.asset(
              "assets/images/ic_close_white.png",
              width: 28,
              height: 28,
              color: Colors.white,
            ),
            onPressed: () {
              String _goBackTip =
                  AppLocalizations.of(widget.store.state.context)!
                      .intercative_room_back_tip;
              if (widget.store.state.typeEnum ==
                  InteractiveTypeEnum.interactiveLive.name) {
                if (widget.store.state.identityTypeEnum ==
                    InteractiveIdentityTypeEnum.anchor.name) {
                  if (widget.store.state.pullStatus ==
                      InteractivePullStatus.pulling) {
                    _goBackTip = AppLocalizations.of(
                            widget.store.state.context)!
                        .interactive_co_streaming_close_co_streaming_room_tip;
                  }
                } else {
                  if (widget.store.state.pushStatus ==
                      InteractivePushStatus.pushing) {
                    _goBackTip = AppLocalizations.of(
                            widget.store.state.context)!
                        .interactive_co_streaming_close_co_streaming_room_tip;
                  }
                }
              } else {
                if (widget.store.state.pullStatus ==
                    InteractivePullStatus.pulling) {
                  _goBackTip = AppLocalizations.of(widget.store.state.context)!
                      .interactive_battles_close_pk_room_tip;
                }
              }
              showAlert(widget.store.state.context, _goBackTip, onOk: () {
                widget.store
                    .dispatch(InteractiveModeActionCreator.goBackPage());
              });
            },
          ),
          Text(
            widget.store.state.roomId,
            style: const TextStyle(fontSize: 16, color: Colors.white),
          ),
          const SizedBox(width: 28),
        ],
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  ///右侧功能控件
  Widget _buildFunctionWidget() {
    String muteImageAsset;
    if (widget.store.state.mute) {
      muteImageAsset = "assets/images/ic_volume_off.png";
    } else {
      muteImageAsset = "assets/images/ic_volume_on.png";
    }
    return Positioned(
      top: 44,
      right: 10,
      child: Column(
        children: [
          IconButton(
            onPressed: () {
              widget.store
                  .dispatch(InteractiveModeActionCreator.clickSwitchCamera());
            },
            icon: Image.asset(
              "assets/images/ic_camera.png",
              width: 22,
              height: 22,
            ),
          ),
          IconButton(
            onPressed: () {
              widget.store.dispatch(InteractiveModeActionCreator.clickMute(
                  !widget.store.state.mute));
            },
            icon: Image.asset(
              muteImageAsset,
              width: 22,
              height: 22,
            ),
          ),
          IconButton(
            onPressed: () {
              widget.store
                  .dispatch(InteractiveModeActionCreator.clickSwitchSpeaker());
            },
            icon: Image.asset(
              "assets/images/ic_speaker_phone.png",
              width: 22,
              height: 22,
            ),
          ),
        ],
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  ///连麦互动
  Widget _buildInteractiveWidget() {
    String title = "";
    if (widget.store.state.identityTypeEnum ==
        InteractiveIdentityTypeEnum.anchor.name) {
      if (widget.store.state.pullStatus == InteractivePullStatus.pulling ||
          widget.store.state.pullStatus == InteractivePullStatus.error) {
        title = AppLocalizations.of(widget.store.state.context)!
            .interactive_co_streaming_end_link_mic;
      } else {
        title = AppLocalizations.of(widget.store.state.context)!
            .interactive_co_streaming_start_link_mic;
      }
    } else {
      if (widget.store.state.pushStatus == InteractivePushStatus.pushing ||
          widget.store.state.pushStatus == InteractivePushStatus.error) {
        title = AppLocalizations.of(widget.store.state.context)!
            .interactive_co_streaming_end_link_mic;
      } else {
        title = AppLocalizations.of(widget.store.state.context)!
            .interactive_co_streaming_start_link_mic;
      }
    }

    return Stack(
      children: [
        _buildRenderPreview(),
        _buildRenderCustomerView(),
        _buildInteractiveButton(title),
      ],
    );
  }

//---------------------------------------------------------------------------------------------------------------
  ///PK互动
  Widget _buildInteractivePKWidget() {
    String title = "";
    if (widget.store.state.pullStatus == InteractivePullStatus.pulling ||
        widget.store.state.pullStatus == InteractivePullStatus.error) {
      title = AppLocalizations.of(widget.store.state.context)!
          .interactive_battles_end_pk;
    } else {
      title = AppLocalizations.of(widget.store.state.context)!
          .interactive_battles_start_pk;
    }
    return Positioned(
      left: 0.0,
      top: MediaQuery.of(widget.store.state.context).padding.top + 100,
      width: MediaQuery.of(widget.store.state.context).size.width,
      height:
          MediaQuery.of(widget.store.state.context).size.height / 2.1 + 16 + 35,
      child: Stack(
        children: [
          StreamBuilder<InteractiveModeState>(
            stream: widget.store.onChange,
            initialData: widget.store.state,
            builder: (context, snapshot) => _buildRenderPreview(),
          ),
          _buildRenderCustomerView(),
          _buildInteractiveButton(title),
          Align(
            alignment: Alignment.topCenter,
            child: Image.asset(
              "assets/images/ic_pk_sign.png",
              width: 66,
              height: 23,
            ),
          ),
        ],
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  /// 视图
  Widget _buildRenderPreview() {
    if (widget.store.state.typeEnum ==
            InteractiveTypeEnum.interactiveLive.name &&
        widget.store.state.identityTypeEnum ==
            InteractiveIdentityTypeEnum.audience.name) {
      var left = 0.0;
      var top = 0.0;
      var width = MediaQuery.of(widget.store.state.context).size.width;
      var height = MediaQuery.of(widget.store.state.context).size.height;

      Widget renderView;
      if (widget.store.state.isClickLink) {
        renderView = StreamBuilder<InteractiveModeState>(
            stream: widget.store.onChange,
            initialData: widget.store.state,
            builder: (context, snapshot) {
              return _buildLiveView(
                AlivcPusherPreviewType.play,
                width,
                height,
                left: left,
                top: top,
              );
            });
      } else {
        renderView = StreamBuilder<InteractiveModeState>(
            stream: widget.store.onChange,
            initialData: widget.store.state,
            builder: (context, snapshot) {
              return _buildAliPlayerView(
                width,
                height,
                left: left,
                top: top,
                onCreated: (viewId) async {
                  if (widget.store.state.pageDidAppear) {
                    _showDialog(
                      pressHandle: () {
                        widget.store.dispatch(
                            InteractiveModeActionCreator.updateCDNPlayerViewId(
                                viewId));
                        widget.store.dispatch(
                            InteractiveModeActionCreator.startPreview());
                      },
                    );
                    widget.store.dispatch(
                        InteractiveModeActionCreator.pageDidAppear(false));
                  } else {
                    widget.store.dispatch(
                        InteractiveModeActionCreator.updateCDNPlayerViewId(
                            viewId));
                  }
                },
              );
            });
      }
      return StreamBuilder<InteractiveModeState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return renderView;
        },
      );
    } else {
      double left = 0.0;
      double top = 0.0;
      double width = 0.0;
      double height = 0.0;
      if (widget.store.state.typeEnum ==
          InteractiveTypeEnum.interactiveLive.name) {
        width = MediaQuery.of(widget.store.state.context).size.width;
        height = MediaQuery.of(widget.store.state.context).size.height;
      } else {
        width = MediaQuery.of(widget.store.state.context).size.width / 2;
        height = MediaQuery.of(widget.store.state.context).size.height / 2.1;
      }

      return StreamBuilder<InteractiveModeState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return _buildLiveView(
              AlivcPusherPreviewType.push,
              width,
              height,
              left: left,
              top: top,
              onCreated: (id) async {
                widget.store
                    .dispatch(InteractiveModeActionCreator.startPreview());
              },
            );
          });
    }
  }

//---------------------------------------------------------------------------------------------------------------
  ///连麦Button
  Widget _buildInteractiveButton(String title) {
    double? left;
    double? right;
    double? top;
    double? bottom;
    double width = 94;
    double height = 35;
    if (widget.store.state.typeEnum ==
        InteractiveTypeEnum.interactiveLive.name) {
      right = MediaQuery.of(widget.store.state.context).padding.right + 16;
      bottom = MediaQuery.of(widget.store.state.context).padding.bottom + 16;
    } else {
      top = MediaQuery.of(widget.store.state.context).size.height / 2.1 + 16;
      left =
          (MediaQuery.of(widget.store.state.context).size.width - width) / 2.0;
    }

    Color buttonColor = Colors.white;
    if (widget.store.state.identityTypeEnum ==
        InteractiveIdentityTypeEnum.anchor.name) {
      if (widget.store.state.pullStatus == InteractivePullStatus.pulling ||
          widget.store.state.pullStatus == InteractivePullStatus.error) {
        buttonColor = Colors.red;
      } else {
        buttonColor = Colors.blue;
      }
    } else {
      if (widget.store.state.pushStatus == InteractivePushStatus.pushing ||
          widget.store.state.pushStatus == InteractivePushStatus.error) {
        buttonColor = Colors.red;
      } else {
        buttonColor = Colors.blue;
      }
    }

    return positionedView(
      left: left,
      right: right,
      top: top,
      bottom: bottom,
      width: width,
      height: height,
      child: PushButton(
        width: 94,
        height: 30,
        titleSize: 14,
        color: buttonColor,
        title: title,
        action: () async {
          if (widget.store.state.typeEnum ==
              InteractiveTypeEnum.interactiveLive.name) {
            //连麦互动
            _startInteractiveAction();
          } else {
            //PK
            _startInteractivePKAction();
          }
        },
      ),
    );
  }

  //互动连麦，点击开始连麦/停止连麦
  _startInteractiveAction() {
    if (widget.store.state.identityTypeEnum ==
        InteractiveIdentityTypeEnum.anchor.name) {
      //主播身份
      if (widget.store.state.pullStatus == InteractivePullStatus.pulling ||
          widget.store.state.pullStatus == InteractivePullStatus.error) {
        showAlert(
          widget.store.state.context,
          AppLocalizations.of(widget.store.state.context)!
              .interactive_co_streaming_stop_link_mic_tip,
          onOk: () {
            widget.store.dispatch(InteractiveModeActionCreator.stopLink());
          },
        );
      } else {
        _showDialog(
          pressHandle: () {
            widget.store.dispatch(InteractiveModeActionCreator.startLink());
          },
        );
      }
    } else {
      //观众身份
      if (widget.store.state.pushStatus == InteractivePushStatus.pushing ||
          widget.store.state.pushStatus == InteractivePushStatus.error) {
        showAlert(
          widget.store.state.context,
          AppLocalizations.of(widget.store.state.context)!
              .interactive_co_streaming_stop_link_mic_tip,
          onOk: () {
            widget.store.dispatch(InteractiveModeActionCreator.stopLink());
          },
        );
      } else {
        widget.store.dispatch(InteractiveModeActionCreator.startLink());
      }
    }
  }

//PK互动，点击开始PK/停止PK
  _startInteractivePKAction() {
    if (widget.store.state.pullStatus == InteractivePullStatus.pulling ||
        widget.store.state.pullStatus == InteractivePullStatus.error) {
      showAlert(
        widget.store.state.context,
        AppLocalizations.of(widget.store.state.context)!
            .interactive_battles_stop_pk_tip,
        onOk: () {
          widget.store.dispatch(InteractiveModeActionCreator.stopLink());
        },
      );
    } else {
      _showDialog(
        pressHandle: () {
          widget.store.dispatch(InteractiveModeActionCreator.startLink());
        },
      );
    }
  }

//---------------------------------------------------------------------------------------------------------------

  Widget _buildRenderCustomerView() {
    double? right;
    double? top;
    double? bottom;
    double? width;
    double? height;
    AlivcPusherPreviewType viewType = AlivcPusherPreviewType.play;

    if (widget.store.state.typeEnum ==
        InteractiveTypeEnum.interactiveLive.name) {
      if (widget.store.state.identityTypeEnum ==
          InteractiveIdentityTypeEnum.anchor.name) {
        viewType = AlivcPusherPreviewType.play;
      } else {
        viewType = AlivcPusherPreviewType.push;
      }

      right = MediaQuery.of(widget.store.state.context).padding.right + 16;
      bottom = MediaQuery.of(widget.store.state.context).padding.bottom +
          16 * 2 +
          35;
      width = 90.0;
      height = 160.0;
    } else {
      viewType = AlivcPusherPreviewType.play;
      right = MediaQuery.of(widget.store.state.context).padding.right;
      top = 0.0;
      width = MediaQuery.of(widget.store.state.context).size.width / 2;
      height = MediaQuery.of(widget.store.state.context).size.height / 2.1;
    }

    if (widget.store.state.isClickLink) {
      return _buildLiveView(
        viewType,
        width,
        height,
        right: right,
        top: top,
        bottom: bottom,
      );
    } else {
      return _buildLinkPlaceholerView(
        width,
        height,
        right: right,
        top: top,
        bottom: bottom,
      );
    }
  }

//---------------------------------------------------------------------------------------------------------------

  Widget _buildLiveView(
    AlivcPusherPreviewType viewType,
    double width,
    double height, {
    double? left,
    double? right,
    double? top,
    double? bottom,
    Color? backgroundColor,
    Function? onCreated,
  }) {
    return positionedView(
      color: backgroundColor ?? Colors.black,
      left: left,
      right: right,
      top: top,
      bottom: bottom,
      width: width,
      height: height,
      child: AlivcPusherPreview(
        viewType: viewType,
        onCreated: (id) {
          onCreated?.call(id);
        },
        x: 0.0,
        y: 0.0,
        width: width,
        height: height,
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  Widget _buildAliPlayerView(
    double width,
    double height, {
    double? left,
    double? right,
    double? top,
    double? bottom,
    Color? backgroundColor,
    Function? onCreated,
  }) {
    return positionedView(
      color: backgroundColor ?? Colors.black,
      left: left,
      right: right,
      top: top,
      bottom: bottom,
      width: width,
      height: height,
      child: AliPlayerView(
        onCreated: (viewId) async {
          onCreated?.call(viewId);
        },
        x: 0.0,
        y: 0.0,
        width: width,
        height: height,
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  Widget _buildLinkPlaceholerView(
    double width,
    double height, {
    double? left,
    double? right,
    double? top,
    double? bottom,
  }) {
    return positionedView(
      color: Colors.black,
      left: left,
      right: right,
      top: top,
      bottom: bottom,
      width: width,
      height: height,
      child: Image.asset(
        "assets/images/ic_interactive_unconnected_bg.png",
        width: 16,
        height: 16,
      ),
    );
  }

//---------------------------------------------------------------------------------------------------------------
  void _showDialog({Function? pressHandle}) async {
    bool onlyUserId =
        widget.store.state.typeEnum == InteractiveTypeEnum.interactiveLive.name;
    var dialog = await showDialog(
      context: context,
      builder: (BuildContext context) {
        return InteractiveMeetingInfoDialog(
          onlyUserId: onlyUserId,
          onConfirm: (userId, roomId) async {
            String _customerRoomId = "";
            if (onlyUserId) {
              if (userId == widget.store.state.userId) {
                Fluttertoast.showToast(
                  msg: AppLocalizations.of(context)!
                      .interactive_co_streaming_input_viewer_id_error_toast,
                  gravity: ToastGravity.CENTER,
                );
                return;
              }
              _customerRoomId = widget.store.state.roomId;
            } else {
              if (userId == widget.store.state.userId) {
                Fluttertoast.showToast(
                  msg: AppLocalizations.of(context)!
                      .interactive_battles_input_streamer_user_id_error_toast,
                  gravity: ToastGravity.CENTER,
                );
                return;
              } else if (roomId == widget.store.state.roomId) {
                Fluttertoast.showToast(
                  msg: AppLocalizations.of(context)!
                      .interactive_battles_input_streamer_room_id_error_toast,
                  gravity: ToastGravity.CENTER,
                );
                return;
              }
              _customerRoomId = roomId;
            }

            InteractiveCustomerInfo customerInfo =
                InteractiveCustomerInfo.create(
                    userId: userId, roomId: _customerRoomId);
            widget.store.dispatch(
                InteractiveModeActionCreator.updateCustomerInfo(customerInfo));
            pressHandle?.call();
          },
        );
      },
    );
  }
//---------------------------------------------------------------------------------------------------------------
}
