import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter_easyloading/flutter_easyloading.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';
import 'package:flutter_livepush_demo/manager/push_config_manager.dart';
import 'package:flutter_livepush_demo/page/push_config/actions.dart';
import 'package:flutter_livepush_demo/page/push_config/state.dart';
import 'package:flutter_livepush_demo/widget/config_cell.dart';
import 'package:flutter_livepush_demo/widget/config_inputurl_view.dart';
import 'package:flutter_livepush_demo/widget/config_segmentbar.dart';
import 'package:flutter_livepush_demo/widget/push_button.dart';
import 'package:flutter_livepush_demo/widget/interactive_input_appbar.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_def.dart';
import 'package:redux/redux.dart';

/**
 * Copyright © 2025 Alibaba Cloud. All rights reserved.
 * @author junhuiYe
 * @date 2025/1/20 09:53
 * @brief
 */
class PushConfigPage extends StatefulWidget {
  final Store<ConfigState> store;
  final PushConfigPageType? pushConfigPageType;
  final dynamic pushConfigOnSettingConfigActionKey;

  PushConfigPage(
      {required this.store,
      this.pushConfigPageType,
      this.pushConfigOnSettingConfigActionKey});

  @override
  _PushConfigPage createState() => _PushConfigPage();
}

class _PushConfigPage extends State<PushConfigPage> {
  @override
  void initState() {
    super.initState();
    if (widget.pushConfigPageType != null) {
      widget.store.state.pageType = widget.pushConfigPageType!;
      widget.store.state.onSettingConfigAction =
          widget.pushConfigOnSettingConfigActionKey;
      ConfigData.resetData();
    }
    widget.store
        .dispatch(ConfigAction(ConfigActionType.initState, payload: context));
  }

  @override
  Widget build(BuildContext context) {
    return FlutterEasyLoading(
        child: Scaffold(
      appBar: _headerRenderView(),
      body: _bodyRenderView(widget.store.state),
    ));
  }

  AppBar _headerRenderView() {
    if (widget.store.state.pageType == PushConfigPageType.basic) {
      return AppBar(
        backgroundColor: Colors.black,
        leading: IconButton(
          icon: Image.asset(
            "assets/images/ic_back.png",
            width: 32,
            height: 32,
          ),
          onPressed: () {
            widget.store.dispatch(
                ConfigAction(ConfigActionType.goBackPage, payload: context));
          },
        ),
        title: ConfigInputURLView(
          placeholder: AppLocalizations.of(context)!.config_input_url,
          defaultText: widget.store.state.configData.pushURL,
          action: (value) {
            widget.store.dispatch(
                ConfigAction(ConfigActionType.updatePushURL, payload: value));
          },
        ),
      );
    } else {
      return buildAppBar(
        AppLocalizations.of(context)!.config_page_theme,
        leadingClickListener: () {
          widget.store.dispatch(
              ConfigAction(ConfigActionType.goBackPage, payload: context));
        },
      );
    }
  }

  Widget _bodyRenderView(ConfigState state) {
    final _screenSize = MediaQuery.of(context).size;
    const double _segmentBarHeight = 44;
    const double _listViewLeftGap = 16;
    const double _buttomPushButtonHeight = 48;
    const double _buttomPushButtonLeftGap = 20;
    const double _verticalGap = 8;
    const double _lineHeight = 1;

    Widget segmentContainer = Container(
      margin: const EdgeInsets.only(top: _verticalGap),
      child: ConfigSegmentBar(
        item: segmentItems,
        fontSize: 16,
        selectIndex: widget.store.state.configData.selectSegmentIndex,
        action: (value) {
          widget.store.dispatch(ConfigAction(
              ConfigActionType.selectSegmentIndex,
              payload: value));
        },
      ),
    );

    Widget renderListView = Container(
      margin: const EdgeInsets.only(
          top: _verticalGap + _segmentBarHeight + _lineHeight,
          left: _listViewLeftGap,
          right: _listViewLeftGap,
          bottom: _buttomPushButtonHeight + _verticalGap * 2),
      child: ListView(scrollDirection: Axis.vertical, children: <Widget>[
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            if (snapshot.data?.configData.selectSegmentIndex == 0) {
              return Column(
                  children: _parametersRenderContainer(widget.store.state));
            } else {
              return Column(
                  children: _featuresRenderContainer(widget.store.state));
            }
          },
        ),
      ]),
    );

    Widget pushButton = Positioned(
      left: _buttomPushButtonLeftGap,
      bottom: _verticalGap + MediaQuery.of(context).padding.bottom,
      width: _screenSize.width - _buttomPushButtonLeftGap * 2,
      height: _buttomPushButtonHeight,
      child: StreamBuilder<ConfigState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return PushButton(
            title: widget.store.state.pageType == PushConfigPageType.basic
                ? AppLocalizations.of(context)!.config_push_button_basic
                : AppLocalizations.of(context)!.config_push_button_interactive,
            action: () {
              widget.store.dispatch(ConfigAction(ConfigActionType.clickPushPage,
                  payload: widget.store.state.configData.pushURL));
            },
          );
        },
      ),
    );

    return GestureDetector(
      onTap: () {
        FocusScope.of(context).unfocus();
      },
      child: Container(
        color: Colors.black,
        child: Stack(
          children: [
            segmentContainer,
            ConfigDivider.create(),
            renderListView,
            pushButton,
          ],
        ),
      ),
    );
  }

  List<Widget> _parametersRenderContainer(ConfigState state) {
    List<Widget> renderView = [];

    if (state.pageType == PushConfigPageType.basic) {
      renderView.addAll([
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return ConfigCell(
              type: ConfigCellType.switchSelect,
              title: AppLocalizations.of(context)!.config_live_push_mode,
              defaultValue: ConfigItemValue.intToBool(
                  snapshot.data!.configData.livePushMode),
              action: (value) {
                int livePushMode = ConfigItemValue.boolToInt(value);
                widget.store.dispatch(ConfigAction(
                    ConfigActionType.setLivePushMode,
                    payload: livePushMode));
              },
            );
          },
        ),
      ]);
    }

    renderView.addAll([
      ConfigCell(
        type: ConfigCellType.header,
        title: AppLocalizations.of(context)!.config_resolution_label,
      ),
      StreamBuilder<ConfigState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return ConfigCell(
            type: ConfigCellType.slider,
            title: state.configData.resolution.stringV,
            defaultValue: state.configData.resolution.intV / 7.0,
            action: (value) {
              ConfigItemValue resolution =
                  ConfigCellCommon.getResolutionDataAtSliderValue(value);
              widget.store.dispatch(ConfigAction(ConfigActionType.setResolution,
                  payload: resolution));
            },
          );
        },
      ),
    ]);

    if (state.pageType == PushConfigPageType.basic) {
      renderView.addAll([
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_auto_bitrate,
          defaultValue: state.configData.enableAutoBitrate,
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setEnableAutoBitrate,
                payload: value));
          },
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_auto_resolution,
          defaultValue: state.configData.enableAutoResolution,
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setEnableAutoResolution,
                payload: value));
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_auto_resolution_tip,
        ),
      ]);
    }

    renderView.addAll([
      StreamBuilder<ConfigState>(
        stream: widget.store.onChange,
        initialData: widget.store.state,
        builder: (context, snapshot) {
          return ConfigCell(
            type: ConfigCellType.switchSelect,
            title: AppLocalizations.of(context)!.config_advanced,
            defaultValue: state.openAdvanced,
            action: (value) {
              widget.store.dispatch(
                  ConfigAction(ConfigActionType.openAdvanced, payload: value));
            },
          );
        },
      ),
    ]);

    if (state.openAdvanced) {
      renderView.addAll(_parametersAdvancedOpenRenderContainer(state));
    }
    return renderView;
  }

  List<Widget> _parametersAdvancedOpenRenderContainer(ConfigState state) {
    List<Widget> renderView = [];

    if (state.pageType == PushConfigPageType.basic) {
      renderView.addAll([
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return ConfigCell(
              type: ConfigCellType.pickerSelect,
              title: AppLocalizations.of(context)!.config_quality_mode_label,
              defaultValue: state.configData.qualityMode.stringV,
              pickerPanelTexts: qualityModeItems.values.toList(),
              action: (value) {
                widget.store.dispatch(ConfigAction(
                    ConfigActionType.setQualityMode,
                    payload: value.index));
              },
            );
          },
        ),
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return ConfigCell(
              type: ConfigCellType.header,
              title:
                  AppLocalizations.of(context)!.config_quality_mode_label_tip,
            );
          },
        ),
        ConfigCell(
          type: ConfigCellType.input,
          title: AppLocalizations.of(context)!.config_target_bitrate,
          defaultValue: state.configData.targetVideoBitrate.toString(),
          infoUnit: "/Kbps",
          enable: state.configData.qualityMode.intV == 2,
          action: (value) {
            int intV = int.parse(value);
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setTargetVideoBitrate,
                payload: intV));
            // dispatch(ConfigActionCreator.setTargetVideoBitrate(intV));
          },
        ),
        ConfigCell(
          type: ConfigCellType.input,
          title: AppLocalizations.of(context)!.config_min_bitrate,
          defaultValue: state.configData.minVideoBitrate.toString(),
          infoUnit: "/Kbps",
          enable: state.configData.qualityMode.intV == 2,
          action: (value) {
            int intV = int.parse(value);
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setMinVideoBitrate,
                payload: intV));
            // dispatch(ConfigActionCreator.setMinVideoBitrate(intV));
          },
        ),
        ConfigCell(
          type: ConfigCellType.input,
          title: AppLocalizations.of(context)!.config_initial_bitrate,
          defaultValue: state.configData.initialVideoBitrate.toString(),
          infoUnit: "/Kbps",
          enable: state.configData.qualityMode.intV == 2,
          action: (value) {
            int intV = int.parse(value);
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setInitialVideoBitrate,
                payload: intV));
            // dispatch(ConfigActionCreator.setInitialVideoBitrate(intV));
          },
        ),
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return ConfigCell(
              type: ConfigCellType.input,
              title: AppLocalizations.of(context)!.config_audio_bitrate,
              defaultValue: state.configData.audioBitrate.toString(),
              infoUnit: "/Kbps",
              action: (value) {
                int intV = int.parse(value);
                widget.store.dispatch(ConfigAction(
                    ConfigActionType.setAudioBitrate,
                    payload: intV));
              },
            );
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_min_fps,
        ),
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return ConfigCell(
              type: ConfigCellType.slider,
              title: state.configData.minFps.stringV.toString(),
              defaultValue: state.configData.minFps.intV / 7.0,
              infoUnit: "/fps",
              action: (value) {
                ConfigItemValue minFps =
                    ConfigCellCommon.getFpsDataAtSliderValue(
                        ConfigItemType.minFps, value);
                widget.store.dispatch(
                    ConfigAction(ConfigActionType.setMinFps, payload: minFps));
              },
            );
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_captrue_fps,
        ),
        ConfigCell(
          type: ConfigCellType.slider,
          title: state.configData.fps.stringV.toString(),
          defaultValue: state.configData.fps.intV / 7.0,
          infoUnit: "/fps",
          action: (value) {
            ConfigItemValue fps = ConfigCellCommon.getFpsDataAtSliderValue(
                ConfigItemType.fps, value);
            widget.store
                .dispatch(ConfigAction(ConfigActionType.setFps, payload: fps));
            // dispatch(ConfigActionCreator.setFps(fps));
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_audio_sample_rate,
        ),
        ConfigCell(
          type: ConfigCellType.slider,
          title: state.configData.audioSampleRate.stringV.toString(),
          defaultValue: state.configData.audioSampleRate.intV / 4.0,
          infoUnit: "/kHz",
          action: (value) {
            ConfigItemValue audioSampleRate =
                ConfigCellCommon.getAudioSampleRateDataAtSliderValue(value);
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setAudioSampleRate,
                payload: audioSampleRate));
            // dispatch(ConfigActionCreator.setAudioSampleRate(audioSampleRate));
          },
        ),
      ]);
    }

    renderView.addAll([
      ConfigCell(
        type: ConfigCellType.header,
        title: AppLocalizations.of(context)!.config_gop_size,
      ),
      ConfigCell(
        type: ConfigCellType.slider,
        title: state.configData.videoEncodeGop.toString(),
        defaultValue: state.configData.videoEncodeGop / 5.0,
        infoUnit: "/s",
        action: (value) {
          int videoEncodeGop =
              ConfigCellCommon.getVideoEncodeGopDataAtSliderValue(value);
          widget.store.dispatch(ConfigAction(ConfigActionType.setVideoEncodeGop,
              payload: videoEncodeGop));
          // dispatch(ConfigActionCreator.setVideoEncodeGop(videoEncodeGop));
        },
      ),
    ]);

    if (state.pageType == PushConfigPageType.basic) {
      renderView.addAll([
        ConfigCell(
          type: ConfigCellType.pickerSelect,
          title: AppLocalizations.of(context)!.config_audio_profile,
          pickerPanelTexts: AudioEncoderProfileData.items.values.toList(),
          defaultValue: state.configData.audioEncoderProfile.stringV,
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setAudioEncoderProfile,
                payload: value.index));
            // dispatch(ConfigActionCreator.setAudioEncoderProfile(value.index));
          },
        ),
        ConfigCell(
          type: ConfigCellType.pickerSelect,
          title: AppLocalizations.of(context)!.config_sound_track,
          pickerPanelTexts: audioChannelItems.values.toList(),
          defaultValue: state.configData.audioChannel.stringV,
          action: (value) {
            widget.store.dispatch(ConfigAction(ConfigActionType.setAudioChannel,
                payload: value.index));
            // dispatch(ConfigActionCreator.setAudioChannel(value.index));
          },
        ),
      ]);
    }

    renderView.addAll([
      ConfigCell(
        type: ConfigCellType.switchSelect,
        title: AppLocalizations.of(context)!.config_audio_hardware_encode,
        defaultValue: ConfigItemValue.intToBool(
            state.configData.audioEncoderMode == 0 ? 1 : 0),
        action: (value) {
          int audioEncoderMode =
              ConfigItemValue.boolToInt(value == false ? true : false);
          widget.store.dispatch(ConfigAction(
              ConfigActionType.setAudioEncoderMode,
              payload: audioEncoderMode));
          // dispatch(ConfigActionCreator.setAudioEncoderMode(audioEncoderMode));
        },
      ),
      ConfigCell(
        type: ConfigCellType.switchSelect,
        title: AppLocalizations.of(context)!.config_video_hardware_encode,
        defaultValue: ConfigItemValue.intToBool(
            state.configData.videoEncoderMode == 0 ? 1 : 0),
        action: (value) {
          int videoEncoderMode =
              ConfigItemValue.boolToInt(value == false ? true : false);
          widget.store.dispatch(ConfigAction(
              ConfigActionType.setVideoEncoderMode,
              payload: videoEncoderMode));
          // dispatch(ConfigActionCreator.setVideoEncoderMode(videoEncoderMode));
        },
      ),
    ]);

    if (state.configData.videoEncoderMode == 0) {
      renderView.addAll([
        ConfigCell(
          type: ConfigCellType.pickerSelect,
          title:
              AppLocalizations.of(context)!.config_video_hardware_encode_codec,
          pickerPanelTexts: videoHardEncoderCodecItems.values.toList(),
          defaultValue: state.configData.videoHardEncoderCodec.stringV,
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setVideoHardEncoderCodec,
                payload: value.index));
            // dispatch(ConfigActionCreator.setVideoHardEncoderCodec(value.index));
          },
        ),
      ]);

      if (state.pageType == PushConfigPageType.basic) {
        renderView.addAll([
          ConfigCell(
            type: ConfigCellType.switchSelect,
            title: AppLocalizations.of(context)!.config_BFrame,
            defaultValue: state.configData.openBFrame,
            action: (value) {
              widget.store.dispatch(
                  ConfigAction(ConfigActionType.setOpenBFrame, payload: value));
              // dispatch(ConfigActionCreator.setOpenBFrame(value));
            },
          ),
        ]);
      }
    }

    return renderView;
  }

  List<Widget> _featuresRenderContainer(ConfigState state) {
    List<Widget> renderView = [];

    if (state.pageType == PushConfigPageType.basic) {
      renderView.addAll([
        ConfigCell(
          type: ConfigCellType.pickerSelect,
          title: AppLocalizations.of(context)!.config_landscape_model,
          defaultValue: state.configData.orientation.stringV,
          pickerPanelTexts: orientationItems.values.toList(),
          action: (value) {
            widget.store.dispatch(ConfigAction(ConfigActionType.setOrientation,
                payload: value.index));
            // dispatch(ConfigActionCreator.setOrientation(value.index));
          },
        ),
        ConfigCell(
          type: ConfigCellType.pickerSelect,
          title: AppLocalizations.of(context)!.config_preview_display_mode,
          defaultValue: state.configData.previewDisplayMode.stringV,
          pickerPanelTexts: previewDisplayModeItems.values.toList(),
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setPreviewDisplayMode,
                payload: value.index));
            // dispatch(ConfigActionCreator.setPreviewDisplayMode(value.index));
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_auto_reconnection,
        ),
        ConfigCell(
          type: ConfigCellType.input,
          title: AppLocalizations.of(context)!.config_reconnect_duration,
          defaultValue:
              state.configData.connectRetryInterval.truncate().toString(),
          infoUnit: "/Kbps",
          action: (value) {
            double floatV = double.parse(value);
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setConnectRetryInterval,
                payload: floatV));
            // dispatch(ConfigActionCreator.setConnectRetryInterval(floatV));
          },
        ),
        ConfigCell(
          type: ConfigCellType.input,
          title: AppLocalizations.of(context)!.config_reconnect_times,
          defaultValue: state.configData.connectRetryCount.toString(),
          infoUnit:
              "/" + AppLocalizations.of(context)!.config_reconnect_times_unit,
          action: (value) {
            int intV = int.parse(value);
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setConnectRetryCount,
                payload: intV));
            // dispatch(ConfigActionCreator.setConnectRetryCount(intV));
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_mirroring,
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_push_mirror,
          defaultValue: state.configData.pushMirror,
          action: (value) {
            widget.store.dispatch(
                ConfigAction(ConfigActionType.setPushMirror, payload: value));
            // dispatch(ConfigActionCreator.setPushMirror(value));
          },
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_preview_mirror,
          defaultValue: state.configData.previewMirror,
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setPreviewMirror,
                payload: value));
            // dispatch(ConfigActionCreator.setPreviewMirror(value));
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_camera_control,
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_front_camera,
          defaultValue: ConfigItemValue.intToBool(state.configData.cameraType),
          action: (value) {
            int cameraType = ConfigItemValue.boolToInt(value);
            widget.store.dispatch(ConfigAction(ConfigActionType.setCameraType,
                payload: cameraType));
            // dispatch(ConfigActionCreator.setCameraType(cameraType));
          },
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_auto_focus,
          defaultValue: state.configData.autoFocus,
          action: (value) {
            widget.store.dispatch(
                ConfigAction(ConfigActionType.setAutoFocus, payload: value));
            // dispatch(ConfigActionCreator.setAutoFocus(value));
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_beauty_switch,
        ),
        StreamBuilder<ConfigState>(
          stream: widget.store.onChange,
          initialData: widget.store.state,
          builder: (context, snapshot) {
            return ConfigCell(
              type: ConfigCellType.switchSelect,
              title: AppLocalizations.of(context)!.config_beauty_button,
              defaultValue: state.configData.beautyOn,
              action: (value) {
                widget.store.dispatch(
                    ConfigAction(ConfigActionType.setBeautyOn, payload: value));
              },
            );
          },
        ),
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_ingest_standby_stream,
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_background_image,
          defaultValue: state.configData.openPauseImg,
          action: (value) {
            widget.store.dispatch(
                ConfigAction(ConfigActionType.setPauseImg, payload: value));
            // dispatch(ConfigActionCreator.setPauseImg(value));
          },
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_network_weak_image,
          defaultValue: state.configData.openNetworkPoorImg,
          action: (value) {
            widget.store.dispatch(ConfigAction(
                ConfigActionType.setNetworkPoorImg,
                payload: value));
            // dispatch(ConfigActionCreator.setNetworkPoorImg(value));
          },
        ),
      ]);
    }

    renderView.addAll([
      ConfigCell(
        type: ConfigCellType.pickerSelect,
        title: AppLocalizations.of(context)!.config_push_streaming,
        defaultValue: state.configData.pushStreamingMode.stringV,
        pickerPanelTexts: pushStreamingModeItems.values.toList(),
        action: (value) {
          widget.store.dispatch(ConfigAction(
              ConfigActionType.setPushStreamingMode,
              payload: value.index));
          // dispatch(ConfigActionCreator.setPushStreamingMode(value.index));
        },
      ),
    ]);

    if (state.pageType == PushConfigPageType.basic) {
      renderView.addAll([
        ConfigCell(
          type: ConfigCellType.header,
          title: AppLocalizations.of(context)!.config_other_features,
        ),
        ConfigCell(
          type: ConfigCellType.switchSelect,
          title: AppLocalizations.of(context)!.config_local_logs,
          defaultValue: state.configData.openLog,
          action: (value) {
            widget.store.dispatch(
                ConfigAction(ConfigActionType.openLog, payload: value));
            // dispatch(ConfigActionCreator.openLog(value));
          },
        ),
      ]);
    }

    return renderView;
  }
}
