import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:flutter_livepush_demo/common/common_util.dart';
import 'package:flutter_livepush_demo/manager/push_config_manager.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';
import 'package:flutter_livepush_demo/page/camera_push/middleware.dart';
import 'package:flutter_livepush_demo/page/camera_push/page.dart';
import 'package:flutter_livepush_demo/page/camera_push/reducer.dart';
import 'package:flutter_livepush_demo/page/camera_push/state.dart';
import 'package:flutter_livepush_demo/page/push_config/actions.dart';
import 'package:flutter_livepush_demo/page/push_config/state.dart';
import 'package:flutter_livepush_plugin/base/live_base.dart';
import 'package:flutter_livepush_plugin/base/live_base_def.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_config.dart';
import 'package:flutter_livepush_plugin/pusher/live_push_def.dart';
import 'package:flutter_livepush_plugin/pusher/live_pusher.dart';
import 'package:redux/redux.dart';

/**
 * Copyright © 2025 Alibaba Cloud. All rights reserved.
 * @author junhuiYe
 * @date 2025/1/20 09:52
 * @brief
 */

ConfigState pushConfigReducer(ConfigState state, action) {
  if (action is ConfigAction) {
    switch (action.type) {
      case ConfigActionType.initState:
        state.mContext = action.payload;
        _init(state, action);
        break;
      case ConfigActionType.updatePushURL:
        state.configData.pushURL = action.payload;
        break;
      case ConfigActionType.selectSegmentIndex:
        state.configData.selectSegmentIndex = action.payload;
        break;
      case ConfigActionType.setLivePushMode:
        _setLivePushMode(state, action);
        break;
      case ConfigActionType.setResolution:
        _setResolution(state, action);
        break;
      case ConfigActionType.setEnableAutoBitrate:
        _setEnableAutoBitrate(state, action);
        break;
      case ConfigActionType.setEnableAutoResolution:
        _setEnableAutoResolution(state, action);
        break;
      case ConfigActionType.openAdvanced:
        state.openAdvanced = action.payload;
        break;
      case ConfigActionType.setQualityMode:
        _setQualityMode(state, action);
        break;
      case ConfigActionType.setTargetVideoBitrate:
        _setTargetVideoBitrate(state, action);
        break;
      case ConfigActionType.setMinVideoBitrate:
        _setMinVideoBitrate(state, action);
        break;
      case ConfigActionType.setInitialVideoBitrate:
        _setInitialVideoBitrate(state, action);
        break;
      case ConfigActionType.setAudioBitrate:
        _setAudioBitrate(state, action);
        break;
      case ConfigActionType.setMinFps:
        _setMinFps(state, action);
        break;
      case ConfigActionType.setFps:
        _setFps(state, action);
        break;
      case ConfigActionType.setAudioSampleRate:
        _setAudioSampleRate(state, action);
        break;
      case ConfigActionType.setVideoEncodeGop:
        _setVideoEncodeGop(state, action);
        break;
      case ConfigActionType.setAudioEncoderProfile:
        _setAudioEncoderProfile(state, action);
        break;
      case ConfigActionType.setAudioChannel:
        _setAudioChannel(state, action);
        break;
      case ConfigActionType.setAudioEncoderMode:
        _setAudioEncoderMode(state, action);
        break;
      case ConfigActionType.setVideoEncoderMode:
        _setVideoEncoderMode(state, action);
        break;
      case ConfigActionType.setVideoHardEncoderCodec:
        _setVideoHardEncoderCodec(state, action);
        break;
      case ConfigActionType.setOpenBFrame:
        _setOpenBFrame(state, action);
        break;
      case ConfigActionType.setOrientation:
        _setOrientation(state, action);
        break;
      case ConfigActionType.setPreviewDisplayMode:
        _setPreviewDisplayMode(state, action);
        break;
      case ConfigActionType.setConnectRetryInterval:
        _setConnectRetryInterval(state, action);
        break;
      case ConfigActionType.setConnectRetryCount:
        _setConnectRetryCount(state, action);
        break;
      case ConfigActionType.setPushMirror:
        _setPushMirror(state, action);
        break;
      case ConfigActionType.setPreviewMirror:
        _setPreviewMirror(state, action);
        break;
      case ConfigActionType.setCameraType:
        _setCameraType(state, action);
        break;
      case ConfigActionType.setAutoFocus:
        _setAutoFocus(state, action);
        break;
      case ConfigActionType.setBeautyOn:
        _setBeautyOn(state, action);
        break;
      case ConfigActionType.setPauseImg:
        _setPauseImg(state, action);
        break;
      case ConfigActionType.setNetworkPoorImg:
        _setNetworkPoorImg(state, action);
        break;
      case ConfigActionType.setPushStreamingMode:
        _setPushStreamingMode(state, action);
        break;
      case ConfigActionType.openLog:
        _openLog(state, action);
        break;
      case ConfigActionType.goBackPage:
        _goBackPage(state, action);
        break;
      case ConfigActionType.clickPushPage:
        _clickPushPage(state, action);
        break;
      default:
        break;
    }
  }
  return state;
}

late AlivcLivePusher _alivcLivePusher;
late AlivcLivePushConfig _alivcLivePusherConfig;

Future<void> _init(ConfigState state, ConfigAction action) async {
  _alivcLivePusher = AlivcLivePusher.init();
  if (state.pageType == PushConfigPageType.basic) {
    ConfigData.resetData();
    _alivcLivePusher.createConfig();
  }
  _alivcLivePusherConfig = AlivcLivePushConfig.init();
  _setLivePusherConfig(state, action);
}

Future<void> _setLivePusherConfig(
    ConfigState state, ConfigAction action) async {
  ConfigAction(ConfigActionType.setResolution, payload: null);
  ConfigAction(ConfigActionType.setPauseImg, payload: true);
  ConfigAction(ConfigActionType.setNetworkPoorImg, payload: true);
}

// 返回
Future<void> _goBackPage(ConfigState state, ConfigAction action) async {
  FocusScope.of(action.payload).unfocus();
  Navigator.of(action.payload).pop();

  if (state.onSettingConfigAction != null) {
    state.onSettingConfigAction!(_alivcLivePusherConfig);
  }
}

// ---------------- 推流参数 TOP ----------------
// 互动模式
Future<void> _setLivePushMode(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.livePushMode = action.payload;
  }
  AlivcLivePushMode livePushMode =
      AlivcLivePushMode.values[state.configData.livePushMode];
  _alivcLivePusherConfig.setLivePushMode(livePushMode);
}

// 分辨率
Future<void> _setResolution(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.resolution = action.payload;
  }
}

// 码率自适应
Future<void> _setEnableAutoBitrate(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.enableAutoBitrate = action.payload;
  }
  _alivcLivePusherConfig
      .setEnableAutoBitrate(state.configData.enableAutoBitrate);
}

// 分辨率自适应
Future<void> _setEnableAutoResolution(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.enableAutoResolution = action.payload;
  }
  _alivcLivePusherConfig
      .setEnableAutoResolution(state.configData.enableAutoResolution);
}

// 码率模式
Future<void> _setQualityMode(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.qualityMode.intV = action.payload;
    state.configData.qualityMode.updateValue(action.payload);
  }
  AlivcLivePushQualityMode qualityMode =
      AlivcLivePushQualityMode.values[state.configData.qualityMode.intV];
  _alivcLivePusherConfig.setQualityMode(qualityMode);
  ConfigAction(ConfigActionType.setResolution, payload: null);
  ConfigAction(ConfigActionType.setFps, payload: null);
}

// 视频目标码率
Future<void> _setTargetVideoBitrate(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.targetVideoBitrate = action.payload;
  }
  _alivcLivePusherConfig
      .setTargetVideoBitrate(state.configData.targetVideoBitrate);
}

// 视频最小码率
Future<void> _setMinVideoBitrate(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.minVideoBitrate = action.payload;
  }
  _alivcLivePusherConfig
      .setInitialVideoBitrate(state.configData.minVideoBitrate);
}

// 视频初始码率
Future<void> _setInitialVideoBitrate(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.initialVideoBitrate = action.payload;
  }
  _alivcLivePusherConfig
      .setInitialVideoBitrate(state.configData.initialVideoBitrate);
}

// 音频码率
Future<void> _setAudioBitrate(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.audioBitrate = action.payload;
  }
  _alivcLivePusherConfig.setAudioBitrate(state.configData.audioBitrate);
}

// 最小帧率
Future<void> _setMinFps(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.minFps = action.payload;
  }
  AlivcLivePushFPS minFps =
      AlivcLivePushFPS.values[state.configData.minFps.intV];
  _alivcLivePusherConfig.setMinFps(minFps);
}

// 采集帧率
Future<void> _setFps(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.fps = action.payload;
  }
  AlivcLivePushFPS fps = AlivcLivePushFPS.values[state.configData.fps.intV];
  _alivcLivePusherConfig.setFps(fps);
}

// 音频采样率
Future<void> _setAudioSampleRate(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.audioSampleRate = action.payload;
  }
  AlivcLivePushAudioSampleRate audioSampleRate = AlivcLivePushAudioSampleRate
      .values[state.configData.audioSampleRate.intV];
  _alivcLivePusherConfig.setAudioSampleRate(audioSampleRate);
}

// 关键帧间隔
Future<void> _setVideoEncodeGop(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.videoEncodeGop = action.payload;
  }
  AlivcLivePushVideoEncodeGOP videoEncodeGOP =
      VideoEncodeGOPData.convertEnumValue(state.configData.videoEncodeGop);
  _alivcLivePusherConfig.setVideoEncodeGop(videoEncodeGOP);
}

// 音频格式
Future<void> _setAudioEncoderProfile(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.audioEncoderProfile.intV = action.payload;
    state.configData.audioEncoderProfile.updateValue(action.payload);
  }
  AlivcLivePushAudioEncoderProfile audioEncoderProfile =
      AlivcLivePushAudioEncoderProfile
          .values[state.configData.audioEncoderProfile.intV];
  _alivcLivePusherConfig.setAudioEncoderProfile(audioEncoderProfile);
}

// 声道数
Future<void> _setAudioChannel(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.audioChannel.intV = action.payload;
    state.configData.audioChannel.updateValue(action.payload);
  }
  AlivcLivePushAudioChannel audioChannel =
      AlivcLivePushAudioChannel.values[state.configData.audioChannel.intV];
  _alivcLivePusherConfig.setAudioChannel(audioChannel);
}

// 音频编码
Future<void> _setAudioEncoderMode(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.audioEncoderMode = action.payload;
  }
  AlivcLivePushAudioEncoderMode audioEncoderMode =
      AlivcLivePushAudioEncoderMode.values[state.configData.audioEncoderMode];
  _alivcLivePusherConfig.setAudioEncoderMode(audioEncoderMode);
}

// 视频编码
Future<void> _setVideoEncoderMode(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.videoEncoderMode = action.payload;
  }
  AlivcLivePushVideoEncoderMode videoEncoderMode =
      AlivcLivePushVideoEncoderMode.values[state.configData.videoEncoderMode];
  _alivcLivePusherConfig.setVideoEncoderMode(videoEncoderMode);
}

// 视频硬编Codec
Future<void> _setVideoHardEncoderCodec(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.videoHardEncoderCodec.intV = action.payload;
    state.configData.videoHardEncoderCodec.updateValue(action.payload);
  }
  AlivcLivePushVideoEncoderModeHardCodec videoEncoderModeHardCodec =
      AlivcLivePushVideoEncoderModeHardCodec
          .values[state.configData.videoHardEncoderCodec.intV];
  _alivcLivePusherConfig.setVideoHardEncoderCodec(videoEncoderModeHardCodec);
}

Future<void> _setOpenBFrame(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.openBFrame = action.payload;
  }
  _alivcLivePusherConfig.setOpenBFrame(state.configData.openBFrame);
}
// ---------------- 推流参数 END ----------------

// ---------------------------------------------

// ---------------- 推流功能 TOP ----------------
// 推流方向
Future<void> _setOrientation(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.orientation.intV = action.payload;
    state.configData.orientation.updateValue(action.payload);
  }
  AlivcLivePushOrientation orientation =
      AlivcLivePushOrientation.values[state.configData.orientation.intV];
  _alivcLivePusherConfig.setOrientation(orientation);

  if (state.configData.openPauseImg == true) {
    ConfigAction(ConfigActionType.setPauseImg, payload: true);
  }

  if (state.configData.openNetworkPoorImg == true) {
    ConfigAction(ConfigActionType.setNetworkPoorImg, payload: true);
  }
}

// 显示模式
Future<void> _setPreviewDisplayMode(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.previewDisplayMode.intV = action.payload;
    state.configData.previewDisplayMode.updateValue(action.payload);
  }
  AlivcPusherPreviewDisplayMode previewDisplayMode =
      AlivcPusherPreviewDisplayMode
          .values[state.configData.previewDisplayMode.intV];
  _alivcLivePusherConfig.setPreviewDisplayMode(previewDisplayMode);
}

// 重连时长
Future<void> _setConnectRetryInterval(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.connectRetryInterval = action.payload;
  }
  _alivcLivePusherConfig
      .setConnectRetryInterval(state.configData.connectRetryInterval);
}

// 重连次数
Future<void> _setConnectRetryCount(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.connectRetryCount = action.payload;
  }
  _alivcLivePusherConfig
      .setConnectRetryCount(state.configData.connectRetryCount);
}

// 推流镜像
Future<void> _setPushMirror(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.pushMirror = action.payload;
  }
  _alivcLivePusherConfig.setPushMirror(state.configData.pushMirror);
}

// 预览镜像
Future<void> _setPreviewMirror(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.previewMirror = action.payload;
  }
  _alivcLivePusherConfig.setPreviewMirror(state.configData.previewMirror);
}

// 摄像头设置
Future<void> _setCameraType(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.cameraType = action.payload;
  }
  AlivcLivePushCameraType cameraType =
      AlivcLivePushCameraType.values[state.configData.cameraType];
  _alivcLivePusherConfig.setCameraType(cameraType);
}

// 自动对焦
Future<void> _setAutoFocus(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.autoFocus = action.payload;
  }
  _alivcLivePusherConfig.setAutoFocus(state.configData.autoFocus);
}

// 美颜开关
Future<void> _setBeautyOn(ConfigState state, ConfigAction action) async {
  state.configData.beautyOn = action.payload;
}

// 暂停图片
Future<void> _setPauseImg(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.openPauseImg = action.payload;
  }
  if (state.configData.openPauseImg == true) {
    String imageName = "";
    if (state.configData.orientation.intV == 0) {
      imageName = "assets/images/pauseImg.png";
    } else {
      imageName = "assets/images/pauseImg_land.png";
    }

    DirType dirType;
    if (Platform.isAndroid) {
      dirType = DirType.externalFile;
    } else {
      dirType = DirType.document;
    }
    var split = imageName.split("/");
    CommonUtil.getSaveDir(dirType, split[2])
        .then((value) => _alivcLivePusherConfig.setPauseImg(value.path));
  } else {
    _alivcLivePusherConfig.setPauseImg("");
  }
}

// 网络差图片
Future<void> _setNetworkPoorImg(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.openNetworkPoorImg = action.payload;
  }
  if (state.configData.openNetworkPoorImg == true) {
    String imageName = "";
    if (state.configData.orientation.intV == 0) {
      imageName = "assets/images/poorNetworkImg.png";
    } else {
      imageName = "assets/images/poorNetworkImg_land.png";
    }

    DirType dirType;
    if (Platform.isAndroid) {
      dirType = DirType.externalFile;
    } else {
      dirType = DirType.document;
    }
    var split = imageName.split("/");
    CommonUtil.getSaveDir(dirType, split[2])
        .then((value) => _alivcLivePusherConfig.setNetworkPoorImg(value.path));
  } else {
    _alivcLivePusherConfig.setNetworkPoorImg("");
  }
}

// 推流模式
Future<void> _setPushStreamingMode(
    ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.pushStreamingMode.intV = action.payload;
    state.configData.pushStreamingMode.updateValue(action.payload);
  }
  _alivcLivePusherConfig
      .setAudioOnly(state.configData.pushStreamingMode.intV == 1);
  _alivcLivePusherConfig
      .setVideoOnly(state.configData.pushStreamingMode.intV == 2);
}

// 本地日志
Future<void> _openLog(ConfigState state, ConfigAction action) async {
  if (action.payload != null) {
    state.configData.openLog = action.payload;
  }
  if (state.configData.openLog == true) {
    String saveLogDir = "";
    String saveAudioDir = ExternalStreamData.downloadDir +
        "/" +
        ExternalStreamData.audioResourceName;
    if (Platform.isAndroid) {
      saveLogDir = await CommonUtil.getSystemPath(DirType.externalFile);
    } else {
      saveLogDir = await CommonUtil.getSystemPath(DirType.document);
    }
    AlivcLiveBase.setLogPath(saveLogDir, saveLogMaxPartFileSizeInKB);
    AlivcLiveBase.setLogLevel(AlivcLivePushLogLevel.debug);
    AlivcLiveBase.setConsoleEnable(true);
  } else {
    AlivcLiveBase.setLogLevel(AlivcLivePushLogLevel.none);
    AlivcLiveBase.setConsoleEnable(false);
  }
}
// ---------------- 推流功能 END ----------------

// 开始推流
Future<void> _clickPushPage(ConfigState state, ConfigAction action) async {
  if (state.pageType == PushConfigPageType.basic) {
    if (action.payload == null || action.payload.length == 0) {
      var tip = AppLocalizations.of(state.mContext)!.config_enter_url_tip;
      print("[TIP-1]: $tip");
    } else {
      if (state.configData.fps.intV < state.configData.minFps.intV) {
        var tip = AppLocalizations.of(state.mContext)!.config_fps_tip;
        print("[TIP-2]: $tip");
        return;
      }
      var url = action.payload;
      print("[TIP-3]: $url");
      FocusScope.of(state.mContext).unfocus();
      Navigator.of(state.mContext).push(CupertinoPageRoute(
          builder: (_) => CameraPushPage(
                store: Store<CameraPushState>(
                  cameraPushReducer,
                  initialState: CameraPushState(),
                  middleware: createCameraPushStateMiddlewares(),
                ),
                pushURLKey: action.payload,
                configData: state.configData,
                pushConfigKey: _alivcLivePusherConfig,
                pushBeautyOnKey: state.configData.beautyOn,
                pushMode: state.configData.livePushMode,
              )));
    }
  } else {
    FocusScope.of(state.mContext).unfocus();
    Navigator.of(state.mContext).pop();

    if (state.onSettingConfigAction != null) {
      state.onSettingConfigAction!(_alivcLivePusherConfig);
    }
  }
}

class ConfigDataUpdatePayload {
  ConfigActionType actionType = ConfigActionType.updateConfigData;
  dynamic value;

  static ConfigDataUpdatePayload create(
      ConfigActionType actionType, dynamic value) {
    ConfigDataUpdatePayload payload = ConfigDataUpdatePayload();
    payload.actionType = actionType;
    payload.value = value;
    return payload;
  }
}
