import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter_livepush_demo/manager/push_config_manager.dart';
import 'package:flutter_gen/gen_l10n/app_localizations.dart';

enum ConfigCellType {
  header,
  input,
  slider,
  switchSelect,
  pickerSelect,
}

class PickerSelectValue {
  var index = 0;
  var item = "";
  var tempIndex = 0;
  var tempItem = "";
}

typedef void ConfigCellActionCallback(value);

class ConfigCell extends StatefulWidget {
  final ConfigCellType type;
  final String? title;
  final dynamic defaultValue;
  final String? infoUnit;
  final List? pickerPanelTexts;
  final bool? enable;
  final ConfigCellActionCallback? action;

  ConfigCell(
      {Key? key,
      required this.type,
      this.title,
      this.defaultValue,
      this.infoUnit,
      this.pickerPanelTexts,
      this.enable,
      this.action})
      : assert(type != null),
        super(key: key);

  @override
  State<StatefulWidget> createState() => _ConfigCellState();
}

class _ConfigCellState extends State<ConfigCell> {
  TextEditingController _inputController = TextEditingController();
  bool _inputFieldBackValueToNull = false;
  double _sliderValue = 0.0;
  bool _switchValue = false;
  FixedExtentScrollController _pickerScrollController =
      FixedExtentScrollController(initialItem: 0);
  PickerSelectValue _pickerSelectValue = PickerSelectValue();
  FocusNode _userFocusNode = FocusNode();

  @override
  void initState() {
    super.initState();
  }

  @override
  void dispose() {
    _inputController.dispose();
    _pickerScrollController.dispose();
    _userFocusNode.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    Widget renderView;
    switch (widget.type) {
      case ConfigCellType.header:
        renderView = _headerView();
        break;
      case ConfigCellType.input:
        {
          if ((widget.defaultValue != null &&
                  widget.defaultValue?.length != 0) &&
              (_inputFieldBackValueToNull == false &&
                  _inputController.text != widget.defaultValue)) {
            _inputController.text = widget.defaultValue!;
          }
          _inputController.addListener(() {
            _inputFieldBackValueToNull = _inputController.text.length == 0;
            if (widget.action != null && _inputFieldBackValueToNull == false) {
              widget.action!(_inputController.text);
            }
          });
          renderView = _inputView();
        }
        break;
      case ConfigCellType.slider:
        {
          if (widget.defaultValue != null) {
            _sliderValue = widget.defaultValue;
          }
          renderView = _sliderView();
        }
        break;
      case ConfigCellType.switchSelect:
        {
          if (widget.defaultValue != null) {
            _switchValue = widget.defaultValue;
          }
          renderView = _switchSelectView();
        }
        break;
      case ConfigCellType.pickerSelect:
        {
          if (widget.pickerPanelTexts != null &&
              widget.pickerPanelTexts?.length != 0 &&
              widget.defaultValue != null &&
              widget.defaultValue?.length != 0) {
            String itemStr = AppLocalizations.of(context)!
                .config_cell_picker_item(widget.defaultValue);
            _pickerSelectValue.item = itemStr;
            _pickerSelectValue.index =
                widget.pickerPanelTexts!.indexOf(widget.defaultValue);
            _pickerSelectValue.tempIndex = _pickerSelectValue.index;
            _pickerSelectValue.tempItem = _pickerSelectValue.item;
          }
          renderView = _pickerSelectView(context);
        }
        break;
    }

    return Column(
      children: [
        renderView,
        ConfigDivider.create(),
      ],
    );
  }

  Widget _headerView() {
    if (widget.title != null && widget.title?.length != 0) {
      return Container(
        alignment: Alignment.centerLeft,
        padding: EdgeInsets.only(top: 15, bottom: 15),
        child: _textView(widget.title ?? "",
            fontSize: 12, color: Colors.grey, maxLines: 4),
      );
    } else {
      return Container(
        height: 44,
      );
    }
  }

  Widget _inputView() {
    bool _enable = widget.enable ?? true;
    return Container(
        height: 64,
        padding: EdgeInsets.only(top: 5, bottom: 5),
        child: TextField(
          enabled: _enable,
          textAlign: TextAlign.right,
          style: TextStyle(fontSize: 15, color: Colors.white),
          controller: _inputController,
          keyboardType: TextInputType.number,
          focusNode: _userFocusNode,
          decoration: InputDecoration(
              icon: _textView(widget.title ?? "", fontSize: 15, maxLines: 2),
              suffixText: widget.infoUnit!,
              suffixStyle: TextStyle(fontSize: 11, color: Colors.white),
              border: InputBorder.none),
        ));
  }

  Widget _sliderView() {
    bool _enable = widget.enable ?? true;
    return Container(
      height: 64,
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          SliderTheme(
              data: SliderTheme.of(context).copyWith(
                trackShape: CustomTrackShape(),
              ),
              child: Slider(
                min: 0.0,
                max: 1.0,
                value: _sliderValue,
                onChanged: (value) {
                  if (_enable) {
                    setState(() {
                      _sliderValue = value;
                    });
                  }
                },
                onChangeEnd: (value) {
                  if (_enable) {
                    if (widget.action != null) {
                      widget.action!(value);
                    }
                  }
                },
              )),
          _sliderRightView(),
        ],
      ),
    );
  }

  Widget _sliderRightView() {
    if (widget.infoUnit != null) {
      return Row(
        mainAxisAlignment: MainAxisAlignment.end,
        children: [
          _textView(widget.title ?? "", fontSize: 15),
          _textView(widget.infoUnit ?? "", fontSize: 11),
        ],
      );
    } else {
      return _textView(widget.title!, fontSize: 15);
    }
  }

  Widget _switchSelectView() {
    bool _enable = widget.enable ?? true;
    return Container(
      height: 64,
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          _textView(widget.title ?? "", fontSize: 15),
          CupertinoSwitch(
              value: _switchValue,
              activeColor: Colors.blue,
              onChanged: (value) {
                if (_enable) {
                  setState(() {
                    _switchValue = value;
                  });
                  if (widget.action != null) {
                    widget.action!(value);
                  }
                }
              }),
        ],
      ),
    );
  }

  Widget _pickerSelectView(BuildContext context) {
    bool _enable = widget.enable ?? true;
    if (widget.pickerPanelTexts == null ||
        widget.pickerPanelTexts?.length == 0) {
      _enable = false;
    }
    return Container(
      height: 64,
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          _textView(widget.title ?? "", fontSize: 15),
          Row(
            mainAxisAlignment: MainAxisAlignment.end,
            children: [
              TextButton(
                child:
                    _textView(_pickerSelectValue.item, align: TextAlign.right),
                onPressed: () {
                  if (_enable) {
                    _didClickPickerSelected(context);
                  }
                },
              ),
              Image.asset(
                "assets/images/ic_arrow.png",
                width: 16,
                height: 16,
              ),
            ],
          ),
        ],
      ),
    );
  }

  void _didClickPickerSelected(BuildContext context) {
    _pickerScrollController =
        FixedExtentScrollController(initialItem: _pickerSelectValue.index);
    showCupertinoModalPopup(
        context: context,
        builder: (BuildContext context) {
          return Container(
            height: 200,
            padding: EdgeInsets.only(
              left: MediaQuery.of(context).padding.left,
              right: MediaQuery.of(context).padding.right,
            ),
            color: Colors.black,
            child: Column(children: [
              Row(
                mainAxisAlignment: MainAxisAlignment.spaceBetween,
                children: [
                  TextButton(
                    child: _textView(AppLocalizations.of(context)!
                        .config_cell_cancel_button),
                    onPressed: () {
                      Navigator.pop(context);
                      setState(() {
                        _pickerSelectValue.tempIndex = _pickerSelectValue.index;
                        _pickerSelectValue.tempItem = _pickerSelectValue.item;
                      });
                    },
                  ),
                  TextButton(
                    child: _textView(
                        AppLocalizations.of(context)!.config_cell_ok_button,
                        align: TextAlign.right),
                    onPressed: () {
                      Navigator.pop(context);
                      setState(() {
                        _pickerSelectValue.index = _pickerSelectValue.tempIndex;
                        _pickerSelectValue.item = _pickerSelectValue.tempItem;
                      });
                      if (widget.action != null) {
                        widget.action!(_pickerSelectValue);
                      }
                    },
                  ),
                ],
              ),
              Expanded(
                  child: DefaultTextStyle(
                style: TextStyle(color: Colors.white, fontSize: 22),
                child: _pickerView(context),
              )),
            ]),
          );
        });
  }

  Widget _pickerView(BuildContext context) {
    return CupertinoPicker(
        itemExtent: 40,
        scrollController: _pickerScrollController,
        onSelectedItemChanged: (value) {
          _pickerSelectValue.tempItem = widget.pickerPanelTexts![value];
          _pickerSelectValue.tempIndex = value;
        },
        children: widget.pickerPanelTexts!.map((data) {
          return Text(
            AppLocalizations.of(context)!.config_cell_picker_item(data),
            style: TextStyle(color: Colors.white),
          );
        }).toList());
  }

  Widget _textView(String text,
      {double? fontSize, Color? color, TextAlign? align, int? maxLines}) {
    fontSize ??= 14;
    color ??= Colors.white;
    align ??= TextAlign.left;
    maxLines ??= 1;

    return Container(
      child: Text(
        text,
        style: TextStyle(fontSize: fontSize, color: color),
        textAlign: align,
        maxLines: maxLines,
      ),
    );
  }
}

class ConfigCellCommon {
  static ConfigItemValue getResolutionDataAtSliderValue(double value) {
    double total = 7;
    ConfigItemValue data = ConfigItemValue.create(ConfigItemType.resolution);
    if (value <= (1.0 / total)) {
      data.updateValue(0);
    } else if (value > (1.0 / total) && value <= (2.0 / total)) {
      data.updateValue(1);
    } else if (value > (2.0 / total) && value <= (3.0 / total)) {
      data.updateValue(2);
    } else if (value > (3.0 / total) && value <= (4.0 / total)) {
      data.updateValue(3);
    } else if (value > (4.0 / total) && value <= (5.0 / total)) {
      data.updateValue(4);
    } else if (value > (5.0 / total) && value <= (6.0 / total)) {
      data.updateValue(5);
    } else if (value > (6.0 / total) && value <= (7.0 / total)) {
      data.updateValue(6);
    }
    return data;
  }

  static ConfigItemValue getFpsDataAtSliderValue(
      ConfigItemType type, double value) {
    double total = 7;
    ConfigItemValue data = ConfigItemValue.create(type);
    if (value < 1 / total || value == 0) {
      data.updateValue(0);
    } else if ((value > 1 / total) && (value < 2 / total)) {
      data.updateValue(1);
    } else if ((value > 2 / total) && (value < 3 / total)) {
      data.updateValue(2);
    } else if ((value > 3 / total) && (value < 4 / total)) {
      data.updateValue(3);
    } else if ((value > 4 / total) && (value < 5 / total)) {
      data.updateValue(4);
    } else if ((value > 5 / total) && (value < 6 / total)) {
      data.updateValue(5);
    } else if ((value > 6 / total) && (value < 7 / total)) {
      data.updateValue(6);
    } else if (value == 1) {
      data.updateValue(6);
    }
    return data;
  }

  static ConfigItemValue getAudioSampleRateDataAtSliderValue(double value) {
    double total = 4;
    ConfigItemValue data =
        ConfigItemValue.create(ConfigItemType.audioSampleRate);
    if (value <= (1.0 / total) || value == 0) {
      data.updateValue(0);
    } else if (value > (1.0 / total) && value <= (2.0 / total)) {
      data.updateValue(1);
    } else if (value > (2.0 / total) && value <= (3.0 / total)) {
      data.updateValue(2);
    } else if (value > (3.0 / total) && value <= (4.0 / total)) {
      data.updateValue(3);
    } else {
      data.updateValue(3);
    }
    return data;
  }

  static int getVideoEncodeGopDataAtSliderValue(double value) {
    double total = 5;
    int data = 0;
    if (value < 1 / total || value == 0) {
      data = 1;
    } else if ((value > 1 / total) && (value < 2 / total)) {
      data = 2;
    } else if ((value > 2 / total) && (value < 3 / total)) {
      data = 3;
    } else if ((value > 3 / total) && (value < 4 / total)) {
      data = 4;
    } else if ((value > 4 / total) && (value < 5 / total)) {
      data = 5;
    } else if (value == 1) {
      data = 5;
    }
    return data;
  }
}

class CustomTrackShape extends RoundedRectSliderTrackShape {
  Rect getPreferredRect({
    required RenderBox parentBox,
    Offset offset = Offset.zero,
    required SliderThemeData sliderTheme,
    bool isEnabled = false,
    bool isDiscrete = false,
  }) {
    final double? trackHeight = sliderTheme.trackHeight;
    final double trackLeft = offset.dx + 10;
    final double trackTop =
        offset.dy + (parentBox.size.height - trackHeight!) / 2;
    final double trackWidth = parentBox.size.width - 20;
    return Rect.fromLTWH(trackLeft, trackTop, trackWidth, trackHeight);
  }
}

class ConfigDivider {
  static Divider create() {
    return Divider(
      height: 1.0,
      color: Colors.grey,
    );
  }
}
